/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//  Class FDCalibrate: Provides methods for calibrating ASQ power spectra
//  into units of physical displacement (nanometers or strain).
//    version:  0.5
//       date:  2005.10.10   
//     author:  Kevin C. Schlaufman (kcs149@psu.edu)
//              Patrick J. Sutton (psutton@ligo.caltech.edu)
//              Aaron Rogan (arogan@wsunix.wsu.edu)
//
///////////////////////////////////////////////////////////////////////////


    // To-do list:

    //----- UpdateAlphaBeta: If the desired time is not covered by the file 
    //      then we may get garbage results for alphabeta, alpha, 
    //      but maybe a good value for beta (like 1).  Should check 
    //      values before accepting them.

    // To-do: Darm averaging should be done on product of DARM channels, not
    // on individual channels.  Did not matter in the past, since one 
    // of the two channels was fixed. 
    //---------- Get darm average.  First try: multiply product of averages 
    //           of each channel (whole method assumes beta does not change 
    //           much over one stride, so let's see if this works).



#include "FDCalibrate.hh"

#include <fstream>
#include "DVecType.hh"
#include "FSeries.hh"
#include "Hanning.hh" 
#include "LscCalib.hh"
#include "PSD.hh"
#include "xsil/Xreader.hh"
#include "xsil/xsil.hh"
#include "xsil/Xwriter.hh"
#include <iostream>
#include <cmath>

using namespace std;

FDCalibrate::FDCalibrate()
  : f(0), Npoint(0), modH(0), phH(0), modC(0), phC(0)
{
    return;
}


FDCalibrate::FDCalibrate(DaccAPI *access, const char *refcal_file, 
			 bool GenerateCalParam, double Fmin, double Fstep, 
			 int npoint)
  : genCalParam(true), mDataAccess(access), alpha(0.0), beta(0.0),
    ampl_asq(0.0), ampl_exc(0.0), gain(0.0), f(0), Npoint(0),
    modH(0), phH(0), modC(0), phC(0)
{

/**  Procedure:
  *  1. Call routine which reads ref cal file and extracts data into 
  *     private members.  This routine should also interpolate the 
  *     reference G(f), C(f) functions to the desired frequencies.
  *     Return error if file read unsuccesful or data missing.
  *  2. Set private member transfunc to reference value (alpha=1=beta).
  */

/* Variables that should be set after constructor finishes (x := done):
    bool genCalParam   x  
    Dacc* mDataAccess  x
    double alpha       x
    double beta        x
    double ampl_asq;   x
    double ampl_exc;   x
    double gain;       x
    double freq_min    x
    double freq_max    x
    double freq_step   x
    double* f          x
    double* modC       x
    double* modH       x
    double* phC        x
    double* phH        x
    int Npoint         x
    float arm_length   x
    Are there any others?
    //
    //----------- ReferenceCalibration struct:
    mRefCal.strain_channel_name;  x
    mRefCal.ref_file_name;        x
    mRefCal.ref_file_comment;     x
    mRefCal.ampl_asq_ref;         -> if genCalParam
    mRefCal.ampl_exc_ref;         -> if genCalParam
    mRefCal.freq;                 -> if genCalParam
    mRefCal.exc_channel_name;     -> if genCalParam
    mRefCal.darm_ref;             -> if genCalParam
    mRefCal.num_darm_channels;    -> if genCalParam
    mRefCal.darm_channel_name;    -> if genCalParam
    mRefCal.Alpha;                -> if !genCalParam
    mRefCal.AlphaBeta;            -> if !genCalParam   
    mRefCal.Freq_min;             x
    mRefCal.Freq_max;             x
    mRefCal.reh;   x
    mRefCal.imh;   x
    mRefCal.ugf   x
    mRefCal.minab; x
    mRefCal.maxab; x
    Are there any others?
*/

    //----- Initialize some ReferenceCalibration struct elements to dummy values.
    mRefCal.reh = 0.0;
    mRefCal.imh = 0.0;
    mRefCal.ugf = 0.0;
    mRefCal.minab =   0.0;  //-- smaller than smallest possible stable value.
    mRefCal.maxab = 100.0; //-- larger than largest conceivable value.

    //----- Record name of reference calibration file to data member.
    mRefCal.ref_file_name = refcal_file;

    //----- Do we want to generate new alpha and beta parameters on-the-fly?
    genCalParam = GenerateCalParam;

    //----- Create LscCalib calibration record holder and use its 
    //      xml parsing method to read in the reference calibration 
    //      file.
    LscCalib mCalib("Calib Doc", "");
    try {
        ifstream infile(mRefCal.ref_file_name.c_str());
        xsil::Xreader xrd(infile);
        //xrd.setDebug(999);
        mCalib.readXml(xrd, "");
        //xsil::xobj* doc = xrd.readDoc();
    } catch (exception& e) {
        cerr << "ERROR: FDCalibrate::FDCalibrate(): Exception caught while reading mCalib: " << e.what() << endl;
        return;
    }

    //----- :TEST: Dump entire record to cout:
    //xsil::Xwriter xw(cout);
    //mCalib.writeXml(xw);

    //----- Extract reference calibratiton data from LscCalib object.

    //----- Top-level data.
    //----- Which IFO?
    mRefCal.strain_channel_name = mCalib.getChannel();
    //----- Which calibration?
    mRefCal.ref_file_comment = mCalib.getComment();
    //---------- Add channel to list requested from data accessor.
    if (! mDataAccess->isChannelRead(mRefCal.strain_channel_name.c_str())) {
        mDataAccess->addChannel(mRefCal.strain_channel_name.c_str());
    } 

    //----- If generating new alpha, beta on-the-fly, then load required 
    //      info on calibration line and DARM channels.
    //      Otherwise, load stored alphas and betas.
    if (genCalParam) {

        //----- Calibration line data
        mRefCal.ampl_asq_ref  = mCalib.getCalLineAmplASQ();
        mRefCal.ampl_exc_ref  = mCalib.getCalLineAmplEXC();
        mRefCal.freq = mCalib.getCalLineFreq();
        mRefCal.exc_channel_name = mCalib.getEXCChannel();
        //---------- Add channel to list requested from data accessor.
        if (!mRefCal.exc_channel_name.empty() &&
	    !mDataAccess->isChannelRead(mRefCal.exc_channel_name.c_str())) {
            mDataAccess->addChannel(mRefCal.exc_channel_name.c_str());
        } 

        //----- DARM gain data
        mRefCal.num_darm_channels = mCalib.getNGainChan();
        mRefCal.darm_channel_name = new string[mRefCal.num_darm_channels]; 
        mRefCal.darm_ref = new double[mRefCal.num_darm_channels]; 
        //cout << "Number of DARM Channels:" << mRefCal.num_darm_channels << endl;
        for (int iChan=0; iChan<mRefCal.num_darm_channels; iChan++)
        {
            mRefCal.darm_channel_name[iChan] = mCalib.getGainChan(iChan);
            //cout << "Stored as: " << mRefCal.darm_channel_name[iChan] << endl;
            mRefCal.darm_ref[iChan] = mCalib.getGainRefValue(iChan);
            //cout << "Value    : " << mRefCal.darm_ref[iChan] << endl;
            //---------- Add channel to list requested from data accessor.
            if (!mDataAccess->isChannelRead(mRefCal.darm_channel_name[iChan].c_str())) {
                mDataAccess->addChannel(mRefCal.darm_channel_name[iChan].c_str());
            } 
        }
 
    } else {

        //----- Alpha, beta.
        mRefCal.Alpha = mCalib.refAlpha();
        //(mRefCal.Alpha).Dump(cout);
        mRefCal.AlphaBeta = mCalib.refAlphaBeta();
        //(mRefCal.Alpha).Dump(cout);

    }
    // :KLUDGE: Should do checks that required data is loaded; e.g., check for 
    // empty TSeries for alpha and beta.  Exit if requested calibrations 
    // aren't available.

    //----- Reference open loop gain and sensing functions.

    //---------- Read fseries and determine frequency range they cover.
    FSeries mOLG = mCalib.refOpenLoopGain();
    FSeries mSF  = mCalib.refSensingFunction();
    if (mOLG.getCenterFreq() > mSF.getCenterFreq())
    {
        mRefCal.Freq_min = mOLG.getCenterFreq();
    } else {
        mRefCal.Freq_min = mSF.getCenterFreq();
    } 
    if (mOLG.getHighFreq() < mSF.getHighFreq())
    {
        mRefCal.Freq_max = mOLG.getHighFreq();
    } else {
        mRefCal.Freq_max = mSF.getHighFreq();
    } 

    //---------- Extract subset of desired frequency range which
    //           is covered by calibration data, and allocate storage.
    double Fmax = Fmin + (npoint-1)*Fstep;
    //cout << Fmin << " " << Fmax << "\n";
    //cout << mRefCal.Freq_min << " " << mRefCal.Freq_max << "\n";

    //---------- Adjust low frequency. 
    int count = 0;
    while (Fmin + count*Fstep < mRefCal.Freq_min) ++count;
    Fmin += count*Fstep;
    npoint -= count;

    //---------- Adjust high frequency.
    count = 0;
    while (Fmax - count*Fstep > mRefCal.Freq_max) ++count;
    Fmax -= count*Fstep;
    npoint -= count;

    //---------- Allocate storage for calibration data.
    Npoint = npoint;
    freq_min = Fmin;
    freq_max = Fmax;
    freq_step = Fstep;
    //cout << "Extracting calibration data for f=[" << Fmin;
    //cout << ":" << Fstep << ":" << Fmin+(Npoint-1)*Fstep << "].\n";

    //---------- Write frequencies 'f'.
    f = new double[Npoint];
    for (int i=0; i<Npoint; ++i) {
        f[i] = Fmin + i*Fstep;
    }

    //---------- Storage for Open-Loop Gain `H', Sensing Function `C', and 
    //           ResponseFunction `RF'.
    modH = new double[Npoint];
    phH = new double[Npoint];
    modC = new double[Npoint];
    phC = new double[Npoint];

    //----- Interpolate reference OLG, SF frequency series.
    //---------- Do OLG first.  
    //---------- Extract frequency (f), modulus (mod), and 
    //           phase (ph) data into double arrays.
    FSeries fs = mOLG;
    int N = fs.getNStep()+1;
    double* temp_f = new double[N];
    double* temp_mod = new double[N];
    double* temp_ph = new double[N];
    fComplex* dat = new fComplex[N];
    fs.getData(N, dat);
    double f0 = fs.getCenterFreq();
    double dF = fs.getFStep();
    for (int i=0; i<N; i++) {
       temp_f[i] = f0 + dF*double(i);
       temp_mod[i] = dat[i].Mag();
       temp_ph[i]  = dat[i].Arg();
    }
    delete[] dat;

    //----- Need OLG at cal.-line frequency if computing on-the-fly 
    //      alphas and betas.
    if (genCalParam) {

        //----- Compute open loop gain at calibration line frequency.
        double modh;
        double phh;
        LinearInterpolation(&(mRefCal.freq), &modh, 1, temp_f, temp_mod, N); 
        LinearInterpolation(&(mRefCal.freq),  &phh, 1, temp_f, temp_ph, N); 
        mRefCal.reh = modh*cos(phh);  // compute these once, store forever!
        mRefCal.imh = modh*sin(phh);
        //cout << "mRefCal.reh = " << mRefCal.reh << endl;
        //cout << "mRefCal.imh = " << mRefCal.imh << endl;

    } 

    //----- Compute unity gain frequency, physically allowed (stable) range 
    //      of alpha*beta.
    //----- Compute unity gain frequency.
    //----- This procedure is accurate to one bin -- close enough for government
    //      work, and given the accuracy of the alpha, beta
    //      values this code computes).
    int bin=findUGFbin(N, temp_mod, 1.0);
    mRefCal.ugf = temp_f[bin];
    //cout << "Unity gain bin      : " << bin         << endl;
    //cout << "Unity gain freq (Hz): " << temp_f[bin] << endl;
    //cout << "Unity gain magnitude: " << temp_mod[bin] << endl;
    //cout << "Unity gain phase    : " << temp_ph[bin] << endl;
    // 
    //----- Find closest frequencies on either side of ugf where phase = pi.
    // KLUDGE: This code assumes that the phase data is restricted to [-pi:pi], 
    // which is the range returned by LscCalib when writing the reference 
    // calibration xml files.  If this is not the case, the following will 
    // fail.  Need to modify code to fail gracefully.
    // For unwrapped data, the following lines should be used to find bin_p 
    // and bin_m:
    //
    //=========================================================================
    //double ph_ugf = temp_ph[bin];
    //int Npi = 0;
    //int odd;
    //if (ph_ugf>=0) {
    //    while (ph_ugf >= Npi*M_PI) ++Npi;
    //    odd = Npi % 2;
    //} else {
    //    while (ph_ugf <  Npi*M_PI) --Npi;
    //    odd = (-Npi) % 2;
    //}
    //Npi += odd;
    //while(temp_ph[bin_p]<(Npi+1)*M_PI && temp_ph[bin_p]>(Npi-1)*M_PI) ++bin_p;
    //while(temp_ph[bin_m]<(Npi+1)*M_PI && temp_ph[bin_m]>(Npi-1)*M_PI) --bin_m;
    //=========================================================================

    int bin_p = bin;
    while(bin_p<N-1 && fabs(temp_ph[bin_p+1]-temp_ph[bin_p])<0.95*M_PI)++bin_p;
    mRefCal.maxab = 1.0/temp_mod[bin_p];
    int bin_m = bin;
    while (bin_m>0 && fabs(temp_ph[bin_m-1]-temp_ph[bin_m])<0.95*M_PI) --bin_m;
    mRefCal.minab = 1.0/temp_mod[bin_m];
    //cout << "bin_p =               " << bin_p         << endl;
    //cout << "Maximum alpha*beta  : " << mRefCal.maxab << endl;
    //cout << "bin_m =               " << bin_m         << endl;
    //cout << "Minimum alpha*beta  : " << mRefCal.minab << endl;

    //---------- Perform linear interpolation for H to requested frequencies. 
    LinearInterpolation(f,modH,Npoint,temp_f,temp_mod,N);
    //cout << "FDCalibrate: Finished LinearInterpolation() for modH.\n";
    LinearInterpolation(f,phH,Npoint,temp_f,temp_ph,N);
    //cout << "FDCalibrate: Finished LinearInterpolation() for phH.\n";
    delete[] temp_f;
    delete[] temp_mod;
    delete[] temp_ph;

    //---------- Now interpolate Sensing Function C.
    fs = mSF; 
    N=fs.getNStep()+1;
    temp_f = new double[N];
    temp_mod = new double[N];
    temp_ph = new double[N];
    dat = new fComplex[N];
    fs.getData(N, dat);
    f0 = fs.getCenterFreq();
    dF = fs.getFStep();
    for (int i=0; i<N; i++) {
       temp_f[i] = f0 + dF*double(i);
       temp_mod[i] = dat[i].Mag();
       temp_ph[i]  = dat[i].Arg();
    }
    delete[] dat;

    //---------- Perform linear interpolation for C.
    LinearInterpolation(f,modC,Npoint,temp_f,temp_mod,N);
    //cout << "FDCalibrate: Finished LinearInterpolation() for modC.\n";
    LinearInterpolation(f,phC,Npoint,temp_f,temp_ph,N);
    //cout << "FDCalibrate: Finished LinearInterpolation() for phC.\n";
    delete[] temp_f;
    delete[] temp_mod;
    delete[] temp_ph;

    //---------- Assign internally stored value of the interferometer 
    //           arm length based on the channel name.
    SetArmLength();
    //cout << "arm_length = " << GetArmLength() << endl;

    //---------- Initialize response function to its reference value.
    ComputeResponseFunction();
}


FDCalibrate::~FDCalibrate()
{
    //cout << "FDCalibrate::~FDCalibrate() called.\n";

    //----- These pointers always set in constructor.
    delete[] f;
    delete[] modH;
    delete[] phH;
    delete[] modC;
    delete[] phC;
    if (genCalParam) {
        delete[] mRefCal.darm_ref;
        delete[] mRefCal.darm_channel_name;
    }
}

int 
FDCalibrate::findUGFbin(int N, const double* vec, double unity) const {

    //----------------------------------  Set the search limits (1 <= i < N).
    int l=1;
    int h=N;

    //----------------------------------  Perform a binary search.
    while (h>l+1) {
	int m=(l+h)>>1;
	if (vec[m] < unity) h = m;
	else                l = m;
    }

    //----------------------------------  Choose the bin closest to 1
    if (l+1<N && vec[l]+vec[l+1]>2*unity) ++l;
    return l;
}

void 
FDCalibrate::ComputeAlpha(double ratio) {
    //----- WARNING:  This method uses the currently stored value of beta
    //      to compute alpha.  YOU MUST CALL UpdateBeta() BEFORE THIS 
    //      METHOD!

    //cout << "FDCalibrate::ComputeAlpha(" << ratio << ") called.\n";

    //----- Procedure:  
    // 1. Get ReH, modH^2, mod(1+H)^2 at cal-line freq (do once).
    // 2. Compute A, B, C (quadratic coeff).
    //      A = (gain)^2*(modH)^2 - mod(1+H)^2/beta^2
    //      B = 2*(gain)^2*ReH
    //      C = (gain)^2
    //    where gain := "ratio" argument.
    // 3. Solve 
    //      A(alpha*beta)^2 + B(alpha*beta) + C = 0.  
    //    Solution:
    //      alpha = (-B +/- [B^2-4*A*C]^0.5)/2*A*beta
    //    Tricky - have to look out for imaginary numbers.
    //    Temp:
    //      disc = B^2 - 4*A*C;
    //      if (disc < 0.0) cerr << "Help!\n";
    //      alpha1 = (-B - sqrt(disc))/(2*A*beta);
    //      alpha2 = (-B + sqrt(disc))/(2*A*beta);
    //    Expect alpha1 to be correct solution (this is the case 
    //    for (R->1,beta->1) and also for H->0.

    //----- Force use of physically acceptable beta.
    if (IsBetaGood()) {

        //----- Compute alpha.
        //----- Compute terms in quadratic equation for alpha*beta.
	double A =  pow(ratio*mRefCal.reh, 2) + pow(ratio*mRefCal.imh, 2) 
	         - (pow(1+mRefCal.reh, 2) + pow(mRefCal.imh, 2))/(beta*beta);
	double C = ratio*ratio;
	double B = 2.0*C*mRefCal.reh;
	double disc = B*B - 4.0*A*C;
        //cout << "A = " << A << endl;
        //cout << "B = " << B << endl;
        //cout << "C = " << C << endl;
        //cout << "disc = " << disc  << endl;
	//----- Verify that discriminant term is positive, so alpha will 
        //      be real.  If not, return alpha=0.
	if (disc >= 0.0) {
 	    //----- Compute alpha by solving quadratic equation.
	    //      (Must choose correct solution of quadratic equation.)
	    alpha = (-B - sqrt(disc))/(2*A*beta);
        } else {
            //----- Alpha value is going to be complex (unphysical in our 
            //      simple model), so set to zero.
 	    alpha = 0.0;
	}

    } else {

        //----- Beta is not physical.  Can't compute a valid alpha.
        alpha = 0.0;

    }

    //----- Go home and have a stiff drink.
}


float FDCalibrate::ComputeLineAmplitude(const TSeries *ts, double Freq) const
{
    //cout << "FDCalibrate::ComputeLineAmplitude(TSeries *ts, double Freq) called.\n";

    //----- Estimate amplitude of calibration line using
    //      sum of power in bins around line freq.
    //      Procedure:
    //       1. Read in TSeries (preferably O(100) sec).
    //       2. Compute PSD with Hanning window, 1 average.
    //       3. Line power = power in bin containing line freq, 
    //          + power in neighboring bins (3 bins total).
    //          - noise power in counted bins,
    //            estimated as average power in nearby bins.
    //       4. Line ampl = [ 2*dF*Line power ]^(0.5)
    //          where dF = n_windows/T = frequency resolution.

    //----- Define needed objects.
    Hanning hann;
    //:KLUDGE: Figure out some rule to determine the number of averages.   
    //Eg: If the stride is an integer, we could use that number with 1Hz resolution.
    //----- :KLUDGE: Using only 1 average for PSD-based line estimation.\n";
    PSD psd(&hann,1);
    FSpectrum fs;
    psd.generate(fs, ts);
    double norm = sqrt( 2.0*fs.getFStep());

    //----- Estimate line power using central bin plus one on either side:
    double power = fs.getSum(Freq - fs.getFStep(), 3.0*(fs.getFStep()) );

    //----- Estimate noise power/bin using 5 bins on either side:
    double noise = fs.getSum(Freq - 10.0*(fs.getFStep()), 5.0*(fs.getFStep()) );
    noise += fs.getSum(Freq + 5.0*(fs.getFStep()), 5.0*(fs.getFStep()) );
    noise /= 10.0;

    //----- Compute line amplitude in AS_Q counts.  Note that if line has 
    //      died we could get power - noise < 0, so check for that.
    double ampl;
    if (power > 3.0*noise) {
	ampl = norm * sqrt(power - 3.0*noise);
    } else {
        ampl = 0.0;
    }

    //----- Finished.
    return(ampl);
}


void FDCalibrate::ComputeResponseFunction(void)
{

    //cout << "FDCalibrate::ComputeResponseFunction() called.\n";

/*
    Goal: 

    Compute FSpectrum *ResponseFunc containing transfer function 
    calculated from the open-loop gain, sensing function, and alpha, 
    beta calibration parameters.

    Procedure:

    Response Function RF related to sensing function C, open-loop gain 
    H, and calibration parameters alpha and beta by

      RF = alpha*C/(1+alpha*beta*H) 

    where C = sf and H = olg are complex, and alpha, beta are real.  
    Assuming data files are in freq -- mod -- phase format, then 
    we want formulae for the modulus and phase of RF in terms of the 
    modulus and phase of C and H.

    Defining Hprime := alpha*beta*H, we have 

      ReHprime = alpha*beta*modH*cos(phH);
      ImHprime = alpha*beta*modH*sin(phH);

      1+Hprime = [1+alpha*beta*modH*cos(phH)] + i*[alpha*beta*modH*sin(phH)]
               = [1+ReHprime] + i*[ImHprime]
      mod(1+Hprime) = [(1+ReHprime)^2 +(ImHprime)^2]^0.5
      ph(1+Hprime) = atan([a*b*mH*sin(phH)]/[1+a*b*mH*cos(phH)])
               = atan(ImHprime/(1+ReHprime))

      modRF = alpha*modC/mod(1+Hprime)
            = alpha*modC/[(1+ReHprime)^2 +(ImHprime)^2]^0.5
      phRF = phC - ph(1+Hprime)
           = phC - atan(ImHprime/(1+ReHprime))

    Notes:  
    - Units are AS_Qcounts/nanometer.
    - FSpectrum supports floats but not doubles - all of this precision 
      may be unnecessary.
*/


    //----- Make sure calibration is physically valid. 
    //      If not, use alpha=1=beta to compute response function 
    //      (i.e., use reference response function). 
    //      In this case alpha and beta are not reset from unphysical 
    //      values; they are just ignored.
    double alpha_temp = alpha;
    double albet_temp = alpha*beta;
    if (!IsCalibrationGood()) {
        alpha_temp = 1.0;
        albet_temp = 1.0;
    }

    //---------- Temporary variables.
    DVectC dv(Npoint);
    for (int i=0; i<Npoint; ++i) { 
	dComplex Hprime = modH[i] * dComplex(cos(phH[i]), sin(phH[i]));
	dComplex Hsense = modC[i] * dComplex(cos(phC[i]), sin(phC[i]));
	dComplex Hresp  = alpha_temp*Hsense/(albet_temp*Hprime + dComplex(1.0));
	dv[i] = fComplex((~Hresp)*arm_length/Hresp.MagSq());
    }

    //----- New FDFilter code:  Store magnitude of response function as 
    //      transfer function in parent FDFilter class.  Specifically, we 
    //      want the transfer function ASQ^2->nm^2; this is 1/RF^2.
    RF = FSeries(f[0], freq_step, Time(0), Interval(1.0/freq_step), dv);
    setFDFilter(RF);

    //----- Done
}

void FDCalibrate::SetAlpha(double value)
{
    alpha = value;
    return;
}


void FDCalibrate::SetArmLength(void)
{
    //---------- Assign arm length in nanometers based on the channel name.
    //           This value is returned and also stored in the data member
    //           float arm_length.
    //
    //           This is used to change the response function from ASQ/strain
    //           to ASQ/nm.  It's a bit whacko, but Range.hh assumes the
    //           diferential arm length is in nm rather than strain.  It's
    //           also convenient for avoiding over/underflow problems with
    //           number storage.
  if        ( mRefCal.strain_channel_name.substr(0,2) == "H1" ) {
        arm_length = 4.0e12;
  } else if ( mRefCal.strain_channel_name.substr(0,2) == "H2" ) {
        arm_length = 2.0e12;
  } else if ( mRefCal.strain_channel_name.substr(0,2) == "L1" ) {
        arm_length = 4.0e12;
  } else if ( mRefCal.strain_channel_name.substr(0,2) == "V1" ) {
        arm_length = 3.0e12;
  } else {
    cerr << "FDCalibrate::SetArmLength() error: " 
	      << mRefCal.strain_channel_name
	      << " not recognised.  Exiting.\n";
    exit(399);
  }
}


void FDCalibrate::SetBeta(double value)
{
    beta = value;
    return;
}


//----- Compute new alpha value on-the-fly based on excitation channel.
//      Must call UpdateBeta() first to make sure beta value is correct
//      (beta is needed for computing alpha).
//      In case alpha is not computable or is not real, set alpha=0.
void FDCalibrate::UpdateAlpha()
{
    //----------------------------------  Calculate error-signal line amplitude
    TSeries* ts_asq = mDataAccess->refData(mRefCal.strain_channel_name.c_str());
    ampl_asq = ComputeLineAmplitude(ts_asq, mRefCal.freq);

    //----------------------------------  Calculate excitation line amplitude
    //  Note that if the excitation readback channel name is blank, the
    //  line amplitude is assumed to be the nominal value (mRefCal.ampl_exc_ref)
    if (mRefCal.exc_channel_name.empty()) {
	ampl_exc = mRefCal.ampl_exc_ref;
    } else {
	TSeries* ts_exc=mDataAccess->refData(mRefCal.exc_channel_name.c_str());
	ampl_exc = ComputeLineAmplitude(ts_exc,mRefCal.freq);
    }

    //----------------------------------  Calculate alpha
    if (ampl_asq*ampl_exc>0.0) {
        gain = (ampl_asq/ampl_exc)*(mRefCal.ampl_exc_ref/mRefCal.ampl_asq_ref);
        ComputeAlpha(gain);
    } else {
        gain  = 0.0;
        alpha = 0.0;
    }
}


//----- Updates "current" values of alpha AND beta using stored reference data.
//      Notes: 
//      1. If t < start time of TSeries then getBin returns bin=0 (first 
//      bin), and so using getDouble() will return the first entry in the 
//      TSeries even if that time is long after the requested time.
//      2. If t < end time of TSeries then getBin returns bin=getNSample(),  
//      which is one PAST the last sample (which is getNSample()-1, by standard 
//      C/C++ conventions).  In this case using getDouble() will return 
//      a garbage value.
void FDCalibrate::UpdateAlphaBeta(Time t)
{	
    //cout << "FDCalibrate::UpdateAlphaBeta(Time t) called.\n";

    //----- Note: If the desired time is not covered by the file 
    //      then we may get garbage results for alphabeta, alpha, 
    //      but maybe a good value for beta (like 1).  Should check 
    //      values before accepting them.
    if (t < (mRefCal.AlphaBeta).getStartTime()) {
        alpha = 0.0;
        beta = 0.0;
        cerr << t << ": calibration not available for this time.\n";
    } else if (t >= (mRefCal.AlphaBeta).getEndTime()) {
        alpha = 0.0;
        beta = 0.0;
        cerr << t << ": calibration not available for this time.\n";
    } else {
        int index = (mRefCal.AlphaBeta).getBin(t);
        double alphabeta = (mRefCal.AlphaBeta).getDouble(index);
        index = (mRefCal.Alpha).getBin(t);
        alpha = (mRefCal.Alpha).getDouble(index);
        beta = alphabeta/alpha;
    }

    return;
}


//----- Compute new beta value on-the-fly based on DARM gain channels.
void FDCalibrate::UpdateBeta()
{	
    //---------- Get darm average.  First try: multiply product of averages 
    //           of each channel (whole method assumes beta does not change 
    //           much over one stride, so let's see if this works).
    beta = 1.0; 
    for (int j=0; j<mRefCal.num_darm_channels; ++j) {
        //cout << "Gain channel: " << (mRefCal.darm_channel_name[j]).c_str() << endl;
        //cout << "Ref value =   " << mRefCal.darm_ref[j] << endl;
        beta *= (mDataAccess->refData((mRefCal.darm_channel_name[j]).c_str()))->getAverage();
        beta /= (mRefCal.darm_ref[j]);
        //cout << "current beta = " << beta        << endl;
    }
    return;
}


//----- Compute new unity gain frequency on the fly for the DARM loop.
//      UGF is set to -1 if the alpha or beta is invalid.
//      Note: modified to leave alpha, beta as is. If they need to be
//      recalculated, that should be done before calling UpdateUGF()
//      JGZ 2005.11.17
void FDCalibrate::UpdateUGF(void) {
    mRefCal.ugf = -1.0; //invalidate, jic
    if (IsCalibrationGood()) {
        int bin = findUGFbin(Npoint, modH, 1./(alpha*beta));
        mRefCal.ugf = f[bin]; 
    }
}

void 
FDCalibrate::LinearInterpolation(const double *X,       double *Y, int N, 
				 const double *x, const double *y, int n) const
{
    //----- Note: LinearInterpolation() assumes X,x are monotonically increasing. 

    //----- Make sure that X ranges are subset of x.
    if ((X[0] < x[0]) || (X[N-1] > x[n-1])) {
        cerr << "ERROR:  In FDCalibrate::LinearInterpolation() data "
             << "to be interpolated does not cover requested range.\n"
             << "Requested: [" << X[0] << "," << X[N-1] << "].\n"
             << "Available: [" << x[0] << "," << x[n-1] << "].\n"
             << "Exiting.\n";
        cout << "ERROR:  In FDCalibrate::LinearInterpolation() data "
             << "to be interpolated does not cover requested range.\n"
             << "Requested: [" << X[0] << "," << X[N-1] << "].\n"
             << "Available: [" << x[0] << "," << x[n-1] << "].\n"
             << "Exiting.\n";
        exit(315);
    }

    //----- Perform interpolation.
    int last = 0;
    for (int i=0; i<N; ++i) {
	double Xi = X[i];

        //----- For each X[i] (0 <= i <= N-1) find int k 
        //      (0 <= k <= n-2) such that x[k] <= Xi <= x[k+1].
        //      If no such k exists then data does not cover range of 
	//      interest, so exit with error message.
        int k=last;
        while ( !(x[k] <= Xi && Xi <= x[k+1]) && k<n-2 ) ++k;
        if (k>n-2) {
            cerr << "FDCalibrate::LinearInterpolation ERROR: Calibration file ";
            cerr << "data do not cover frequency range requested."  << endl;
	    cerr << "N = " << N << ", n = " << n << endl;
	    cerr << "X[" << i << "] = " << Xi << ", x[" << k << "] = " 
		 << x[k] << endl; 
	    cerr << " Exiting.\n";
            cout << "FDCalibrate::LinearInterpolation ERROR: Calibration file ";
            cout << "data do not cover frequency range requested." << endl;
	    cout << "N = " << N << ", n = " << n << endl;
	    cout << "X[" << i << "] = " << Xi << ", x[" << k << "] = " 
		 << x[k] << endl; 
	    cout << " Exiting.\n";
            exit(304);
        }
        //----- Interpolate for Y[i] using (x[k],y[k]), (x[k+1],y[k+1]).  
        Y[i] = y[k] + (y[k+1] - y[k])*(Xi - x[k])/(x[k+1] - x[k]);
	last = k;
    }
}
