/* -*- mode: c++; c-basic-offset: 4; -*- */
#include <cstdlib>
#include <string>
#include <iostream>
#include "SigFlag.hh"
#include "DMTifc.h"
#include "Time.hh"
#include "Interval.hh"

using namespace std;

extern "C" 
{
#include "FrameL.h"
void FrDicFree(struct FrFile*);
}

//-------------------------------------  Static variables
static bool     mActive;      // Run flag
static int      mDebug;       // Debug level
static int      mContig;      // Boundary frame flag
static int      mNFrames;     // Number of frames processed
static int      mNContig;     // Number of contiguous segments
static SigFlag* mTerm;
static SigFlag* mAttn;
static struct FrFile* mFile;

//-------------------------------------  Internal functions
static void DMTC_Init(int argc, const char *argv[]);
static void DMTC_Halt(void);
static void DMTC_MainLoop(void);

//-------------------------------------  Print FrameL error
const char*
FrErrorMsg(struct FrFile* file) {
    switch (file->error) {
    case FR_ERROR_NO_FRAME:
        return "No Frame";
    case FR_ERROR_NO_FILE:
        return "No File";
    case FR_ERROR_MALLOC_FAILED:
        return "Malloc Failed";
    case FR_ERROR_BUFF_OVERFLOW:
        return "Output buffer overflowed";
    case FR_ERROR_WRITE_ERROR:
        return "Write Error";
    case FR_ERROR_READ_ERROR:
        return "Read error";
    case FR_ERROR_FRSET:
        return "Frame reset?";
    case FR_ERROR_BAD_END_OF_FRAME:
        return "Bad End of Frame";
    case FR_ERROR_OPEN_ERROR:
        return "Open Error";
    default:
        return 0;
    }
    return 0;
}

//-------------------------------------  Main entry point
int main(int argc, const char *argv[]) {
    DMTC_Init(argc,argv);
    DMTC_MainLoop();
    DMTC_Halt();
    return 0;}

//-------------------------------------  DMTCBase constructor.
static void
DMTC_Init(int argc, const char *argv[]) {
    mActive  = false;
    mDebug   = 0;
    mContig  = 0;
    mNFrames = 0;
    mNContig = 0;

    const char*     partition = getenv("LIGOSMPART");
    if (!partition) partition = "LIGO_Online";
    const char*     file      = (const char*) 0;

    //---------------------------------  Parse the arguments
    for (int i=1 ; i<argc && argv ; i++) {
        if (!strcmp("-partition", argv[i])) {
	    partition = argv[++i];
	} else if (!strcmp("-infile", argv[i])) {
	    file = argv[++i];
	} else if (!strcmp("-debug", argv[i])) {
	    if (++i == argc) mDebug = 1;
	    else             mDebug = strtol(argv[i], 0, 0);
	}
    }

    //---------------------------------  Handle signals
    mTerm = new SigFlag(SIGTERM);
    if (!mTerm) return;
    mTerm->add(SIGHUP);
    mTerm->add(SIGINT);
    mAttn = new SigFlag(SIGUSR1, SigFlag::kBlock);

    //---------------------------------  Open a frame stream (FrameCPP)
    FrLibIni(0, 0, mDebug);
    if (file) {
	mFile = FrFileINew(const_cast<char*>(file));
	if (mFile) {
	    cout << "DMTCBase: Opened file " << file 
		 << " for frame input." << endl;
	} else {
	    cerr << "DMTCBase: Unable to open file " << file 
		 << " for frame input." << endl;
	    return;
	}
    } else {
        string inname = "/online/";
        inname += partition;
	mFile = FrFileINew(const_cast<char*>(inname.c_str()));
	if (mFile) {
	    cout << "DMTCBase: Opened partition " << partition
		 << " for frame input." << endl;
	} else {
	    cerr << "DMTCBase: Unable to access partiton " << partition
		 << " for frame input." << endl;
	    return;
	}
    }
    mActive = true;

    //---------------------------------  Let the user do his stuff
    DMT_Initialize(argc, argv);
}

//-------------------------------------  DMTCBase object destructor.
static void 
DMTC_Halt(void) 
{
    //---------------------------------  Release signals
    if (mTerm) delete mTerm;
    mTerm = 0;

    if (mAttn) delete mAttn;
    mAttn = 0;

    //---------------------------------  Close the Input file/partition.
    FrFileIEnd(mFile);

    //---------------------------------  Print statistics
    if (mDebug != 999) {
        cout << "Frame reading terminated after " << mNFrames 
	     << " frames and " << mNContig << " boundaries." << endl;
    }
    DMT_Halt();
}

//-------------------------------------  Main processing loop function.
static void
DMTC_MainLoop(void) 
{
    //---------------------------------  Loop until terminated.
    int  FNext = 0;
    Time TNext(0);
    while(!(*mTerm) && mActive) {

        //-----------------------------  Check for Attention Interrupt
        if (*mAttn) {
	    mAttn->clear();
	    DMT_Attention();
	    if (!mActive) continue;
	}

        //-----------------------------  Read a frame
        struct FrameH* frame = FrameRead(mFile);
	if (frame) {

	    //-------------------------  Check continuity
	    int  FrameID = frame->frame;
	    Time FTime(frame->GTimeS, frame->GTimeN);
	    Interval dT(frame->dt);
	    mContig = !mNFrames || (FNext != FrameID) || (TNext != FTime);
	    if (mContig) {
	        mNContig++;
	        if (Debug() && mNFrames) {
		    cout << "Frame " << FrameID << " (GPS " << FTime 
			 << ") found when frame " << FNext << " (GPS " 
			 << TNext << ") was expected." << endl;
		}
	    }
	    FNext = FrameID + 1;
	    TNext = FTime   + dT;

	    //------------------------  Process a frame.
	    mNFrames++;
	    DMT_Frame(frame);

	    //------------------------  Delete the frame.
	    FrameFree(frame);
	    FrDicFree(mFile);
	} else {
	    cerr << "DMTCBase: Error " << mFile->error 
		 << " while reading frame." << endl;
	    break;
	}
    }
}

//-------------------------------------  Return debug level.
int  Debug(void) {
    return mDebug;}

//-------------------------------------  Request job termination.
void finish(void) {
    mActive = false;}

//-------------------------------------  Return frame count
int  getNFrames(void) {
    return mNFrames;}

//-------------------------------------  Return contiguous segment count
int  getNContig(void) {
    return mNContig;}

//-------------------------------------  Return Boundary flag
int Boundary(void) {
    return mContig;}
