/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "column.hh"
#include <iostream>
#include <strings.h>

using namespace table;
using namespace std;

//======================================  Column constructor
Column::Column(const std::string& title) 
  : mTitle(title), mOrdering(kUnordered)
{}

//======================================  Add a row
static const unsigned int min_reserve(32);
static const double       frac_alloc(0.25);
void 
Column::insertRow(row_index b4, const Cell& c) {
    unsigned int N=mRows.size();
    unsigned int nCap=mRows.capacity();
    if (nCap < min_reserve) {
        reserve(min_reserve);
    } else if (nCap <= N) {
        unsigned int nReserve = nCap + (unsigned int)(nCap*frac_alloc);
	reserve(nReserve);
    }
    if (b4 >= N) mRows.push_back(c);
    else         mRows.insert(mRows.begin()+b4, c);
    testOrder(b4);
}

//======================================  Add a row
void 
Column::insertRow(row_index b4, const Column& c, row_index s, row_index n) {
    mRows.insert(mRows.begin()+b4, c.mRows.begin()+s, c.mRows.begin()+s+n);
    if (mOrdering != kUnordered && mOrdering == c.getOrdering()) {
	testOrder(b4);
	testOrder(b4+n);
    } else {
	mOrdering = kUnordered;
    }
}

//======================================  Sum up a column
double 
Column::Sum(void) const {
    if (getType() != Cell::kNumeric) throw runtime_error("Data not numeric");
    int N = mRows.size();
    double Sigma = 0.0;
    for (int i=0; i<N; i++) Sigma += mRows[i].getNumeric();
    return Sigma;
}

//======================================  Get column type
Cell::cell_type 
Column::getType(void) const {
    Cell::cell_type  t = Cell::kNull;
    int N = mRows.size();
    for (int i=0; i<N; i++) {
        Cell::cell_type  ti = mRows[i].getType(); 
	if (t == Cell::kNull) t = ti;
	else if (t != ti) t = Cell::kUnknown;
    }
    return t;
}

//======================================  Get width of widest cell
unsigned int
Column::getWidth(void) const {
    unsigned int w(0);
    unsigned int N = mRows.size();
    for (unsigned int i=0; i<N; i++) {
        unsigned int  wi = mRows[i].getWidth(); 
	if (w < wi) w = wi;
    }
    return w;
}

//======================================  Move a row
void
Column::move(row_index i, row_index j) {
    if (i == j) return;
    if (i >= size() || j >= size()) 
        throw runtime_error("Row number out of range");
    Cell t=mRows[i];
    if (i > j) {
	for (row_index k=i; k>j; k--) mRows[k] = mRows[k-1];
    } else {
	for (row_index k=i; k<j; k++) mRows[k] = mRows[k+1];
    }
    mRows[j] = t;
    testOrder(j);
}

//======================================  Remove unselected rows
void 
Column::reserve(row_index N) {
    mRows.reserve(N);
}

//======================================  Remove unselected rows
void 
Column::selectRows(const std::vector<bool>& mask) {
    row_index N    = mRows.size();
    if (N > mask.size()) N = mask.size();
    row_index fill = 0;
    while (fill < N &&  mask[fill]) ++fill;
    for (row_index i=fill+1; i<N; ++i) {
        if (mask[i]) mRows[fill++] = mRows[i];
    }
    if (fill < mRows.size()) mRows.erase(mRows.begin()+fill, mRows.end());
}

//======================================  Column mutators
void
Column::setFormat(const string& form) {
    mFormat.parse(form);
    row_index N = mRows.size();
    for (row_index i=0; i<N; ++i) {
	mRows[i].reformat(mFormat);
    }
}

void
Column::setJustify(const string& title) {
    cerr << "Column::setJustify() not implemented" << endl;
}

void
Column::setOrdering(col_order co) {
    mOrdering = co;
}

void
Column::setTitle(const string& title) {
    mTitle = title;
}

void
Column::setXType(const string& type) {
    mXType = type;
}

static const char*
print_ordered(Column::col_order co) {
    switch (co) {
    case Column::kUnordered: 
	return "unordered";
    case Column::kIncreasing:
	return "increasing";
    case Column::kDecreasing:
	return "decreasing";
    case Column::kUndefined:
	return "unknown";
    }
    return "";
}

void
Column::status(void) const {
    cout << mTitle << ": type=" << mXType << " ordered=" 
	 << print_ordered(mOrdering) << endl;
}

void 
Column::testOrder(row_index i) {
    mOrdering = kUnordered;
}

//======================================  Case insenstitive title compare
bool 
Column::titleq(const std::string& name) const {
    if (name.size() != mTitle.size()) return false;
    return !strcasecmp(name.c_str(), mTitle.c_str());
}
