/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef SEG_READER_HH
#define SEG_READER_HH

#include <map>
#include <string>
#include <iosfwd>
#include "SegList.hh"

class segID;

/**  The %seg_iobase class defines an API to read and write segment lists 
  *  from a LIGO xml file, flat ascii text table, or an hdf5 file. Segment
  *  lists are read into a local segment map which can then be merged into
  *  an external map using the merge() function.
  *  \brief Segment list I/O API.
  *  \author John Zweizig
  *  \version $Id$
  */
class seg_iobase {
public:
    /**  Map of segment names to segment lists.
      */
    typedef std::map<segID, LockSegList> seg_map;

    /**  Segment map iterator.
     */
    typedef seg_map::iterator seg_iter;

    /**  Constant segmentmap iterator.
     */
    typedef seg_map::const_iterator const_seg_iter;

    /**  Segment map node definition.
     */
    typedef seg_map::value_type seg_value;

public:

    /**  Default constructor for a segment i/o base class.
      *  \brief Default constructor
      */ 
    seg_iobase(void);

    /**  Destructor for a segment I/O base class.
      *  \brief Destructor
      */
    virtual ~seg_iobase(void);

    /**  Get the debug print level
      *  \brief Debug print level
      *  \returns Requested print level.
      */
    int debug(void) const;

    /**  Merge the reader segments into the specified global map. If the
      *  \c sorted argument is true (default) the resulting segment lists
      *  will be coalesced and sorted. 
      *  \brief Merge the reader map segments into the specified global map.
      *  \param outmap Global output map.
      *  \param sorted If true the global map will be sorted.
      */
    void merge(seg_map& outmap, bool sorted=true);

    /**  Read the specified segment(s) from the input file.  The input file
      *  format is inferred from the file name extension. The segments 
      *  specified with \a select are read in from the file.
      *  If the name, ifo or version field of \a select is empty, all segments
      *  matching the non-empty fields are read in. If the input file does not
      *  specify the segment name, ifo  or version, the requested name, ifo
      *  and version are used. The \a format string is used for ascii
      *  file input to specify column assignments.
      *  \brief read segments from a file.
      *  \param file   Input file name.
      *  \param select Segment ID(s) to be read in.
      *  \param format File format (ascii files only - see seg_iotext).
      */
    virtual void read_seg(const std::string& file, const segID& select, 
		  const std::string& format="") = 0;

    /**  Set the debug printout level.
      *  \brief Set debug level.
      *  \param lvl Debug verbosity level.
      */
    void set_debug(int lvl);

    /**  Write selected segments from the specified map.
      *  \brief Write segments to a file.
      *  \param smap   Segment map containing segment list(s) to be written.
      *  \param select Segment ID(s) to be read in.
      *  \param format File format (ascii files only - see read_txt).
      *  \param file   Output file name.
      */
    virtual void write_seg(const seg_map& smap, const segID& select, 
		   const std::string& format, const std::string& file) = 0;

protected:
    int mDebug;      ///< Debug flag
    seg_map mSegMap; ///< Local segment map
};

/**  The segID  class contains all the segment identification information
  *  including name, ifo, and version.
  */
class segID {
public:
    static const int kUnknown = -1; ///< Segment version is unknown
    static const int kAny     = -2; ///< Match any segment version number
    // static const int kLast = -3; //< Match the latest segment version
public:

    /**  Construct a %segID with the specified name ifo and version.
      *  \brief Constructor.
      *  \param name    Segment name
      *  \param ifo     Segment ifo
      *  \param version Segment version number
      */
    segID(const std::string& name, const std::string ifo, int version=1);

    /**  Construct a %segID from a full segment name, \e i.e. a string in the 
      *  format \c "\<ifo\>:\<name\>;\<version\>".
      *  \brief Construct a %segID from a full segment name.
      *  \param full_name Full segment name.
      */
    explicit segID(const std::string& full_name);

    /**  Destroy a segID and free any allocated storage.
      *  \brief Destructor
      */
    virtual ~segID(void);

    /**  Get the full segment name in the format 
      *  \c "\<ifo\>:\<name\>;\<version\>".
      *  \brief Get the full segment name
      *  \return Full segment name string.
      */
    std::string full_name(void) const;

    /**  Return the ifo name string for this segment type.
      *  \brief Get the segment ifo name
      *  \return Segment ifo name string.
      */
    const std::string& ifo(void) const;

    /**  Return the name string for this segment type.
      *  \brief Get the segment name.
      *  \return Segment name string.
      */
    const std::string& name(void) const;

    /**  Return the version number for this segment type.
      *  \brief Get the segment version.
      *  \return Segment version.
      */
    int version(void) const;

    /**  Compare two segment IDs. The name field is considered most 
      *  significant followed by IFO name and version.
      *  \brief Compare segment IDs.
      *  \param x SegID to be compared to this instance.
      *  \return True if this instance is less than the argument ID.
      */
    bool operator< (const segID& x) const;

    /**  Compare two segment IDs. The name field is considered most 
      *  significant followed by IFO name and version.
      *  \brief Compare segment IDs.
      *  \param x SegID to be compared to this instance.
      *  \return True if this instance is equal to the argument ID.
      */
    bool operator==(const segID& x) const;

    /**  Compare two segment IDs. The name field is considered most 
      *  significant followed by IFO name and version.
      *  \brief Compare segment IDs.
      *  \param x SegID to be compared to this instance.
      *  \return True if this instance is not equal to the argument ID.
      */
    bool operator!=(const segID& x) const;

    /**  Return the root SegID for a summary segment.
      *  \brief get the root segID;
      *  \return Root SegID.
      */ 
    segID root_segid(void) const;

    /**  Set the segment identifier ifo name string.
      *  \brief Set the Ifo name.
      *  \param Ifo New value for the ifo name.
      */
    void set_ifo(const std::string& Ifo);

    /**  Set the segment identifier name string.
      *  \brief Set the name.
      *  \param Name New value for the name.
      */
    void set_name(const std::string& Name);

    /**  Set the segment identifier version.
      *  \brief Set the segment version.
      *  \param vrsn New value for the version.
      */
    void set_version(int vrsn);

    /**  Test whether the id refers to a summary segment, i.e. whether the 
      *  name ends in "_summary"
      *  \brief Test for summary segment
      *  \return True if 
      */
    bool summary(void) const;

    /**  Compare %segID fields to \c \{name,ifo,version\}. If an argument is 
      *  not specified (null string or kAny) the corresponding field is
      *  assumed to match.
      *  \brief Match %segID to field values.
      *  \param nam Name string to match or null string.
      *  \param ifo Ifo string to match or null string.
      *  \param ver Version number.
      *  \return True if %segID fields match argument values.
      */
    bool test(const std::string& nam, const std::string ifo, int ver) const;

    /**  Compare %segID fields to argument. If an argument field is 
      *  not specified (null string or kAny) the corresponding field is
      *  assumed to match.
      *  \brief Match %segID to field values.
      *  \param select %segID to compare to this instance.
      *  \return True if %segID fields match argument values.
      */
    bool test(const segID& select) const;
private:
    std::string _name;
    std::string _ifo;
    int         _version;
};

//======================================  Non member methods
std::ostream&
operator<<(std::ostream& out, const segID& sid);

//======================================  Inline member methods.
inline const std::string&
segID::ifo(void) const {
    return _ifo;
}

inline bool
segID::operator!=(const segID& x) const {
    return ! (*this == x);
}

inline const std::string&
segID::name(void) const {
    return _name;
}

inline int
segID::version(void) const {
    return _version;
}

inline int 
seg_iobase::debug(void) const {
    return mDebug;
}

#endif // !defined(SEG_READER_HH)
