/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "framexmit/framexmittypes.hh"
#include "framexmit/matchInterface.hh"
#include "checksum_crc32.hh"
#include <iostream>
#include <string>
#include <cstring>
#include <cstdlib>
#include <cstdio>
#include <vector>
#include "sockutil.h"
#include <sys/socket.h>
#include "SigFlag.hh"
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <net/if.h>

/** pktdump class defines the fxpktdump 
 **/
class pktdump {
public:
   pktdump(int argc, const char* argv[]);
   ~pktdump();
   void close(void);
   void dump(const framexmit::packetHeader& pkt);
   bool getPacket(framexmit::auto_pkt_ptr& ppkt, bool block);
   bool open(const char* mcast_addr, const char* interface, int port);
   void run(void);
private:
   bool    terse;
   bool    verbose;
   SigFlag term;
   int     sock;
   bool    multicast;
   ip_mreq group;
   struct  sockaddr_in 	name;
   struct  sockaddr_in 	peer;
   in_addr i_addr;
   long    nBroadcast;
   long    nRebroadcast;
};

bool
operator!=(const struct sockaddr_in& x, const struct sockaddr_in& y) {
    return (x.sin_addr.s_addr != y.sin_addr.s_addr)
	|| (x.sin_port != y.sin_port);
}

using namespace std;
using namespace framexmit;

int
main(int argc, const char* argv[]) {
   pktdump mon(argc, argv);
   mon.run();
}

//======================================  Constructor
pktdump::pktdump(int argc, const char* argv[]) 
   : terse(false), verbose(false), term(SIGTERM), sock(-1), multicast(false), 
     nBroadcast(0), nRebroadcast(0)

{
   bool syntax = false;
   for (int i=1; i < argc-2 && !syntax; i++) {
      string argi = argv[i];
      if (argi == "--help") {
	 cout << "pktdump prints out framexmit (llldd) protocol packets."
	      << endl;
	 cout << "By default, the pktdump prints a list of packet counts for"
	      << endl;
	 cout << "each frame and packet type (broadcast or rebroadcast) "
	      << endl;
	 cout << "received. In terse mode (--terse option), each packet "
	      << endl;
	 cout << "received results in a single line printout with the "
	      << endl;
	 cout << "packet type, frame timestamp, and frame timestamp."
	      << endl;
	 cout << "In verbose mode, the full packet header dump is produced for"
	      << endl;
	 cout << "each packet received. The internal consistency of each"
	      << endl
	      << "packet (length and checksum) is checked and an error message"
	      << endl
	      << "is written to stdout if an error is found." << endl;
	 syntax = true;
      }
      else if (argi == "--terse") {
	 terse = true;
      }
      else if (argi == "--verbose") {
	 verbose = true;
      }
      else {
	 cerr << "Unrecognized command argument: " << argi << endl;
	 syntax = true;
      }
   }

   if (syntax) {
      cerr << "fxpktdump command syntax: " << endl;
      cerr << "  fxpktdump [<options>] <mcast-ip>:<port> <iface-addr>" << endl;
      cerr << "Try 'fxpktdump --help' for a complete description" << endl;
      term.setSig0();
      return;
   }

   string mcast = argv[argc-2];
   int port = 9876;
   string::size_type inx = mcast.find(':');
   if (inx != string::npos) {
      port = strtol(mcast.c_str()+inx+1, 0, 0);
      mcast.erase(inx);
   }
   string iface = argv[argc-1];
   bool ok = open(mcast.c_str(), iface.c_str(), port);
   if (!ok) {
      cerr << "Open failed" << endl;
      exit(1);
   }
}

//======================================  Destructor
pktdump::~pktdump(void) {
   cout << "Number of broadcast packets:   " << nBroadcast << endl;
   cout << "Number of rebroadcast packets: " << nRebroadcast << endl;
   close();
}

//======================================  Dump a packet header
void
pktdump::dump(const packetHeader& ph) {
   string type;
   switch (ph.pktType) {
   case PKT_BROADCAST:
      type = "Broadcast";
      nBroadcast++;
      break;
   case PKT_REBROADCAST:
      type = "Rebroadcast";
      nRebroadcast++;
      break;
   case PKT_REQUEST_RETRANSMIT:
      type = "Request retransmit";
      break;
   default:
      type = "Unknown";
   }
   if (terse) {
      cout << type << " " << ph.seq << " (" << ph.timestamp << ") " 
	   << ph.pktNum << "/" << ph.pktTotal << endl;
   }
   else if (verbose) {
      cout << endl;
      cout << "Packet type:               " << type         << endl;
      cout << "Packet length:             " << ph.pktLen    << endl;
      cout << "Packet sequence:           " << ph.seq       << endl;
      cout << "Packet number:             " << ph.pktNum    << endl;
      cout << "Total packets in sequence: " << ph.pktTotal  << endl;
      cout << "Checksum of packet:        " << ph.checksum  << endl;
      cout << "Packet timestamp:          " << ph.timestamp << endl;
      cout << "Packet duration:           " << ph.duration  << endl;
   }
}

//======================================  Open a socket.
bool
pktdump::open(const char* mcast_addr, const char* interface, int port) {
   in_addr_t mcast_ip = htonl(inet_addr(mcast_addr));

   if (sock >= 0) {
      close();
   }
   
   // open socket
   sock = socket(PF_INET, SOCK_DGRAM, 0);
   if (sock < 0) {
      perror("Unable to create a socket");
      return false;
   }
   
   // set reuse socket option 
   int reuse = 1;
   if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, 
		  (char*) &reuse, sizeof (int)) != 0) {
      perror("Unable to nake socket reusable");
      ::close (sock);
      sock = -1;
      return false;
   }
   
   // set receive buffer size 
   int bufsize = rcvInBuffersize;
   for (int bTest=bufsize; bTest<=rcvInBufferMax; bTest+=rcvInBuffersize) {
      if (setsockopt (sock, SOL_SOCKET, SO_RCVBUF, 
		      (char*) &bTest, sizeof (int)) != 0) break;
      bufsize = bTest;
   }
   if (setsockopt (sock, SOL_SOCKET, SO_RCVBUF, 
		   (char*) &bufsize, sizeof (int)) != 0) {
      perror("Unable to set receive buffer size");
      ::close (sock);
      sock = -1;
      return false;
   }
   socklen_t arglen = sizeof(int);
   int readsize = 0;
   getsockopt (sock, SOL_SOCKET, SO_RCVBUF, &readsize, &arglen);
   cout << "Receive buffer size = " << bufsize << "(set), " << readsize
	<< "(readback)" << endl;
  
   // bind socket
   name.sin_family = AF_INET;
   name.sin_port = htons (port);
   name.sin_addr.s_addr = htonl(INADDR_ANY);
   if (::bind(sock, (struct sockaddr*) &name, sizeof (name))) {
      perror("Unable to bind socket");
      ::close (sock);
      sock = -1;
      return false;
   }
   
   // options
   if (mcast_addr != 0) {
      char addr_tmp[32];
      multicast = true;
      
      // check multicast address
      if (!IN_MULTICAST(mcast_ip)) {
	 cerr << "Address not multicast addr: " << hex << mcast_ip << endl;
	 ::close (sock);
	 sock = -1;
	 return false;
      }
      
      // get interface address
      if (!matchInterface (sock, interface, i_addr)) {
	 cerr << "No matching interface address" << endl;
	 ::close (sock);
	 sock = -1;
	 return false;
      }
      
      // multicast: join
      group.imr_multiaddr.s_addr = inet_addr (mcast_addr);
      group.imr_interface.s_addr = i_addr.s_addr;
      if (setsockopt (sock, IPPROTO_IP, IP_ADD_MEMBERSHIP, 
		      (char*) &group, sizeof (ip_mreq)) == -1) {
	 perror("Unable to join multicast group");
	 ::close (sock);
	 sock = -1;
	 return false;
      }
      cout << "join multicast " << mcast_addr << " at "
	   << inet_ntop(AF_INET, &i_addr, addr_tmp, sizeof(addr_tmp))
	   << endl;
   }
   else {
      multicast = false;
      // broadcast: enable
      int bset = 1;
      if (setsockopt (sock, SOL_SOCKET, SO_BROADCAST, 
		      (char*) &bset, sizeof (bset)) == -1) {
	 ::close (sock);
	 sock = -1;
	 return false;
      }
      cout << "broadcast port "  << port << endl;
   }
   
   // check quality of service
   //if ((qos < 0) || (qos > 2)) {
   //   qos = 2;
   //}
   return true;
}


void
pktdump::close(void) {
   if (sock < 0) {
      return;
   }
   
   if (multicast) {
      // multicast drop
      if (setsockopt (sock, IPPROTO_IP, IP_DROP_MEMBERSHIP, 
		      (char*) &group, sizeof (ip_mreq)) == -1) {
	 // do nothing; close anyway
      }
   }
  
   ::close (sock);
   sock = -1;
}

bool 
pktdump::getPacket (auto_pkt_ptr& pkt, bool block) {
   
   // allocate packet buffer
   pkt = auto_pkt_ptr (new (nothrow) packet);
   if (pkt.get() == 0) {
      cerr << "Unable to allocate buffer" << endl;
      return false;
   }

   // receive a packet from socket
   checksum_crc32 crc;
   uint32_t save_crc;
   size_t n;
   string ckstat;
   do {
      // if not blocking poll socket
      if (!block) {
	 timeval 	wait;		// timeout=0
	 wait.tv_sec = 0;
	 wait.tv_usec = 0;
	 fd_set 		readfds;	
	 FD_ZERO (&readfds);
	 FD_SET (sock, &readfds);
	 int nset = select (FD_SETSIZE, &readfds, 0, 0, &wait);
	 if (nset < 0) {
	    perror("select failed on receiver socket");
	    return false;
	 }
	 else if (nset == 0) {
	    return false;
	 }
      }
      // read packet
      #define USE_RECVFROM 1 
#ifdef USE_RECVFROM
      socklen_t max = sizeof (peer);
      int nread = recvfrom(sock, (char*) pkt.get(), sizeof (packet), 0, 
			  (struct sockaddr*) &peer, &max);
#else
      int nread = recv(sock, (char*) pkt.get(), sizeof (packet), 0);
#endif
      if (nread < 0) {
	 perror("Error in recvfrom");
	 return false;
      }
      n = nread;

      // Checksum
      save_crc = ntohl(pkt->header.checksum);
      if (save_crc != 0) {
	crc.reset();
	pkt->header.checksum = 0;
	crc.add(reinterpret_cast<unsigned char*>(pkt.get()), nread);
	pkt->header.checksum = crc.result();
      }
      // swap if necessary
      pkt->ntoh();
      if (n != sizeof(packetHeader) + pkt->header.pktLen) {
	 cerr << "Error in packet length, size: " << n << " expected: "
	      << sizeof(packetHeader) + pkt->header.pktLen << endl;
      }
      else if (save_crc != pkt->header.checksum) {
         cerr << "Checksum Error in packet " << pkt->header.pktNum 
	      << "/" << pkt->header.pktTotal << " of " << pkt->header.seq
              << " (gps " << pkt->header.timestamp << ")" << endl;
      }

      // repeat until valid packet
   } while (n < sizeof(packetHeader) || 
	    (n != sizeof(packetHeader) + pkt->header.pktLen));

   return true;
}

void
pktdump::run(void) {
   auto_pkt_ptr pkt;
   std::vector<int> received, rebroadcast;
   struct sockaddr_in save_peer;
   int idnow = 0;
   int N = 0;
   while (!term) {
      bool pktok = getPacket(pkt, true);
      if (!pktok) continue;
      dump(pkt->header);
      if (((peer.sin_addr.s_addr ^ i_addr.s_addr) & 0xffff) != 0) {
	cout << "peer (" << hex << peer.sin_addr.s_addr 
	     << ") isn't on the multicast network (" << i_addr.s_addr << dec
	     << ")" << endl;
      }
      int id = pkt->header.timestamp;
      if (!idnow) {
         cout << "Peer port: " << hex << peer.sin_addr.s_addr 
              << dec << ":" << peer.sin_port << endl;
	 idnow = id;
	 N =  pkt->header.pktTotal;	 
 	 received.clear();
	 received.resize(N, 0);
         rebroadcast.clear();
         rebroadcast.resize(N, 0);
	 save_peer = peer;
      }
      else if (id > idnow) {
	 int N = received.size();
	 int nFirst = -1;
	 cout << "record: " << idnow << " from: " << hex 
	      << peer.sin_addr.s_addr << dec << " received: ";
	 for (int i=0; i<N; i++) {
	    if (received[i]) {
	       if (nFirst < 0) nFirst = i;
	    }
	    else if (nFirst >= 0) {
	       if (nFirst == i-1) cout << " " << i-1;
	       else               cout << " " << nFirst << "-" << i-1;
	       nFirst = -1;
	    }
	 }
	 if (nFirst < 0)         cout << endl;
	 else if (nFirst == N-1) cout << " " << nFirst << endl;
	 else                    cout << " " << nFirst << "-" << N-1 << endl;
	 cout << "record: " << idnow << " rebroadcast: ";
	 for (int i=0; i<N; i++) cout  << " " << rebroadcast[i];
	 cout << endl;
         idnow = id;
         N =  pkt->header.pktTotal;
 	 received.clear();
	 received.resize(N, 0);
         rebroadcast.clear();
         rebroadcast.resize(N, 0);
      }
      if (save_peer != peer) {
	cout << "Changed to peer: " << hex << peer.sin_addr.s_addr 
	     << dec << ":" << peer.sin_port << endl;
	save_peer = peer;
      }
      int pktno = pkt->header.pktNum;
      switch ( pkt->header.pktType) {
      case PKT_BROADCAST:
	 received[pktno]++;
	 break;
      case PKT_REBROADCAST:
         rebroadcast[pktno]++;
         break;
      default:
         break;
      }
   }
}
