#ifndef _LIGO_EVENTWINDOW_H
#define _LIGO_EVENTWINDOW_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Window							*/
/*                                                         		*/
/* Module Description: Manages a time window with events		*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Window.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <vector>
#include <deque>
#include "Time.hh"
#include "Interval.hh"
#include "events/TimeWindow.hh"
#include "events/Iterator.hh"


namespace events {

   class Event;

/** An event window is a utility class which manages a set of events 
    which fall within a time window. The idea is that an analysis
    program will defined a maximum duration within to look for events.
    During the analysis this time window is moved through the series 
    of events until all events are porcessed. An event window 
    can be any coincidence order. An order of one means that the window
    refernces one current event and the list of events with in the 
    time window. An order of two means that two events are currently
    selected for a concidence analysis. A triple coincidence would
    have an order of three, and so on.
    
    An event window provides iterators to go through the list of 
    events in the window.
       
    @memo Defines an event window
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Window {
   public:
      /// Value type
      typedef Event value_type;
      /// Pointer type
      typedef Event* pointer;
      /// Const pointer type
      typedef const Event* const_pointer;
      /// Reference type
      typedef Event& reference;
      /// Const reference type
      typedef const Event& const_reference;
      /// Iterator type
      typedef Iterator iterator;
      /// Const iterator type
      typedef ConstIterator const_iterator;
      /// List of event pointers
      typedef std::deque<pointer> list;
      /// List of current event pointers
      typedef std::vector<pointer> current;
   
      /** Creates an event window. For a symmetric window set the 
          offset to the negative of half the window width.
          @memo Constructor
          @param window Time window
          @param order Coincidence order
       ******************************************************************/
      explicit Window (const TimeWindow& window = TimeWindow (1.0), 
                      int order = 1);
      /** Creates an event window. For a symmetric window set the 
          offset to the negative of half the window width.
          @memo Constructor
          @param width Width of window
          @param offset Window offset
          @param order Coincidence order
       ******************************************************************/
      Window (const Interval& width, const Interval& offset, 
             int order = 1);
      /** Creates a window from a single event.
          @memo Constructor
       ******************************************************************/
      explicit Window (pointer event);
      /** Creates a window from two events.
          @memo Constructor
       ******************************************************************/
      Window (pointer e1, pointer e2);
   
      /** Equality operator. Two windows are the same if they have the
          same order and the same current events.
          @memo Equality operator
       ******************************************************************/
      bool operator== (const Window& win) const;
      /** Inequality operator
          @memo Inequality operator
       ******************************************************************/
      bool operator!= (const Window& win) const {
         return !(*this == win); }
   
      /** Time of first event in the list
          @memo Time of first event
       ******************************************************************/
      Time GetTimeFirst() const;
      /** Time of last event in the list
          @memo Time of last event
       ******************************************************************/
      Time GetTimeLast() const;
      /** Time. Averaged time of current event(s).
          @memo Time of current events
       ******************************************************************/
      Time GetTime() const;
      /** Start time of window
          @memo Window start time
       ******************************************************************/
      Time GetStartTime() const;
      /** Stop time of window
          @memo Window stop time
       ******************************************************************/
      Time GetStopTime() const;
   
      /** Set the time window.
          @memo Set time window
       ******************************************************************/
      void SetWindow (const TimeWindow& window) {
         mWindow = window; }
      /** Set the time window.
          @memo Set time window
          @param width Window duration
          @param offset Window offset
       ******************************************************************/
      void SetWindow (const Interval& width, const Interval& offset) {
         mWindow = TimeWindow (width, offset); }
      /** Set the time offset.
          @memo Set time offset
       ******************************************************************/
      void SetOffset (const Interval& offset) {
         mWindow.SetOffset (offset); }
      /** Get the time offset.
          @memo Get time offset
       ******************************************************************/
      Interval GetOffset () const {
         return mWindow.GetOffset(); }
      /** Set the time duration.
          @memo Set time duration
       ******************************************************************/
      void SetWidth (const Interval& duration) {
         mWindow.SetOffset (duration); }
      /** Get the time duration.
          @memo Get time duration
       ******************************************************************/
      Interval GetWidth () const {
         return mWindow.GetWidth(); }
      /** Set window order. How many current or coincidence events can 
          be selected?
          @memo Set window order
       ******************************************************************/
      void SetOrder (int n) {
         mCurrent.resize (n, 0); }
      /** Get window order. How many current or coincidence events are 
          selected?
          @memo Window order
       ******************************************************************/
      int GetOrder() const {
         return mCurrent.size(); }
   
      /** Access the current event(s).
          @memo Current event access
       ******************************************************************/
      reference operator() (int n) {
         return *mCurrent[n]; }
      /** Access the current event(s).
          @memo Current event access
       ******************************************************************/
      const_reference operator() (int n) const {
         return *mCurrent[n]; }
      /** Access the current event(s).
          @memo Current event access
       ******************************************************************/
      reference Current (int n = 0) {
         return *mCurrent[n]; }
      /** Access the current event(s).
          @memo Current event access
       ******************************************************************/
      const_reference Current (int n = 0) const {
         return *mCurrent[n]; }
      /** Set the current event(s).
          @memo Set current event
       ******************************************************************/
      void SetCurrent (int n, pointer current) {
         mCurrent[n] = current; }
      /** Get the list of current event(s).
          @memo Get current event list
       ******************************************************************/
      current& GetCurrentList () {
         return mCurrent; }
      /** Get the list of current event(s).
          @memo Get current event list
       ******************************************************************/
      const current& GetCurrentList () const {
         return mCurrent; }
   
      /** First event in window.
          @memo First
       ******************************************************************/
      reference Front() {
         return *mList.front(); }
      /** First event in window.
          @memo First
       ******************************************************************/
      const_reference Front() const {
         return *mList.front(); }
      /** Last event in window.
          @memo Last
       ******************************************************************/
      reference Back() {
         return *mList.back(); }
      /** Last event in window.
          @memo Last
       ******************************************************************/
      const_reference Back() const {
         return *mList.back(); }
      /** Beginning of list.
          @memo Begin
       ******************************************************************/
      iterator Begin();
      /** Beginning of list.
          @memo Begin
       ******************************************************************/
      const_iterator Begin() const;
      /** End of list.
          @memo End
       ******************************************************************/
      iterator End();
      /** End of list.
          @memo End
       ******************************************************************/
      const_iterator End() const;
      /** List access.
          @memo List access
       ******************************************************************/
      reference operator[] (int n) {
         return *mList[n]; }
      /** List access.
          @memo List access
       ******************************************************************/
      const_reference operator[] (int n) const {
         return *mList[n]; }
      /** List access.
          @memo List access
       ******************************************************************/
      reference At (int n) {
         return *mList[n]; }
      /** List access.
          @memo List access
       ******************************************************************/
      const_reference At (int n) const {
         return *mList[n]; }
      /** List size.
          @memo List size
       ******************************************************************/
      int Size() const {
         return mList.size(); }
      /** List empty?.
          @memo List empty?
       ******************************************************************/
      bool Empty() const {
         return mList.empty(); }
      /** Add an element at the end.
          @memo Add an element
       ******************************************************************/
      void PushBack (pointer event) {
         mList.push_back (event); }
      /** Add an element at the beginning.
          @memo Add an element
       ******************************************************************/
      void PushFront (pointer event) {
         mList.push_front (event); }
      /** Remove an element from the end.
          @memo Remove an element
       ******************************************************************/
      void PopBack() {
         mList.pop_back(); }
      /** Remove an element from the beginning.
          @memo Remove an element
       ******************************************************************/
      void PopFront() {
         mList.pop_front(); }
      /** Clear list.
          @memo Clear list
       ******************************************************************/
      void Clear() {
         mList.clear(); mCurrent.assign (mCurrent.size(), 0); }
   
   private:
      /// Time window
      TimeWindow	mWindow;
      /// List of current event(s)
      current		mCurrent;
      /// List of all events within the window
      list		mList;
   };


}

#endif // _LIGO_EVENTWINDOW_H
