#ifndef _LIGO_EVENTSET_H
#define _LIGO_EVENTSET_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Set							*/
/*                                                         		*/
/* Module Description: Defines a set of of event chains			*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Chain.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <iosfwd>
#include "Time.hh"
#include "Interval.hh"
#include "events/Chain.hh"
#include "events/ChainPtr.hh"
#include "events/Condition.hh"
#include "events/ConditionPtr.hh"
#include "events/Column.hh"
#include "events/Function.hh"
#include "events/FunctionPtr.hh"
#include "events/Iterator.hh"
#include "events/WindowIterator.hh"


   class Histogram1;
   class Histogram2;
   class TSeries;


namespace events {


   class Window;
   class Function;


/** An event set consists of a several event chains which are
    are treated in parallel. This allows combining events sets
    generated by different sources on the same time stretch of
    the data without physically merging the files containig the events.
    The event set is the set used by the user to manipulate, select
    and histogram events.
   
    @memo Chain of event lists
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Set {
   public:
      /// Basic event set type
      typedef std::vector<ChainPtr> eventset;
      /// Basic chain type
      typedef Chain::eventchain eventchain;
      /// Basic list type
      typedef List::eventlist eventlist;
      /// Value type
      typedef Chain::value_type value_type;
      /// Size type
      typedef Chain::size_type size_type;
      /// Difference type 
      typedef Chain::difference_type difference_type;
      /// Reference type
      typedef Chain::reference reference;
      /// Const reference type
      typedef Chain::const_reference const_reference;
      /// Pointer type
      typedef Chain::pointer pointer;
      /// Const pointer type
      typedef Chain::const_pointer const_pointer;      
      /// Iterator
      typedef Iterator iterator;
      /// Const iterator
      typedef ConstIterator const_iterator;
      /// Iterator
      typedef ReverseIterator reverse_iterator;
      /// Const iterator
      typedef ConstReverseIterator const_reverse_iterator;
   
      /** Creats a event set with N chains.
          @memo Constructor
       ******************************************************************/
      explicit Set (int N = 1);
      /** Creats a event set from a single file (one chain with one 
          list).
          @memo Constructor
       ******************************************************************/
      explicit Set (const char* filename) 
      : mDefault (0) {
         AddChain (filename); }
      /** Returns a copy of the event chain. This method must be 
          overriden by all descendents.
          @memo Copy the event
          @return event copy
       ******************************************************************/
      Set* Copy() const {
         return new Set (*this); }
   
      /** Add an event chain to the set.
          @memo Add
       ******************************************************************/
      bool AddChain (const char* filename);
      /** Add an event chain to the set.
          @memo Add
       ******************************************************************/
      bool AddChain (const Chain& chain);
      /** Get an event chain from the set.
          @memo Get
       ******************************************************************/
      Chain& GetChain (int lnum) {
         return *mSet[lnum]; }
      /** Get an event chain from the set.
          @memo Get
       ******************************************************************/
      const Chain& GetChain (int lnum) const {
         return *mSet[lnum]; }
      /** Remove an event chain form the set.
          @memo Remove
       ******************************************************************/
      bool RemoveChain (int lnum);
      /** Number of event chains in set.
          @memo N
       ******************************************************************/
      int N() const {
         return mSet.size(); }
      /** Set the default chain. The default chain is used in several
          operations such as PushBack and Insert.
          @memo Set the default chain
          @param def Default chain index
       ******************************************************************/
      void SetDefaultChain (int def = 0) {
         mDefault = def; }
      /** Get the index of the default chain.
          @memo Get the index of the default chain
          @return Index of the default chain
       ******************************************************************/
      int GetDefaultChain() const {
         return mDefault; }
   
      /** Merge all event chains into one.
          @memo Merge
       ******************************************************************/
      void Merge();
      /** Join all events from the specified set into the currenmt set. 
          The specified set will be empty afterwards. This method simply 
          transfers the chains. It will not merge the chains!
          @memo Merge
       ******************************************************************/
      void Join (Set& set);
      /** Add the event chains specified in the configuration file to
          the event set.
          The configuration file must contain a list of files
          (one per line) each representing an event chain configuration
          file. Lines starting with a UNIX comment letter are ignored.
          @memo Configure
       ******************************************************************/
      bool Configure (const char* filename);
      /** Save the set to the specified file(s). If a non-negative 
          perfile is specified and if the number of events in a set 
          is larger than perfile, multiple files are created for each
          chain with a maximum of perfile events each. The file number 
          is increased automatically until maxevents is reached or all 
          events are processed.
          @memo Save
          @param filename Name of file
          @param perfile Number of events per written file
          @param maxevents Maximum number of events written
       ******************************************************************/
      bool Save (const char* filename, int perfile = 0,
                int maxevents = -1) const;
   
   	/** Restore an event set from a single file (one chain with one list).
   	    All events in the set are cleared before importing the file.
   		 @memo Restore
   		 @param filename Name of file
   	******************************************************************/
      bool Restore(const char* filename);
   
      /** Size of list (number of events in chain).
          @memo Size
       ******************************************************************/
      int Size() const;
      /** Empty list?
          @memo Empty
       ******************************************************************/
      bool Empty() const;
      /** Equality operator.
          @memo Equality
       ******************************************************************/
      bool operator== (const Set& l) const;
      /** Inequality operator.
          @memo Equality
       ******************************************************************/
      bool operator!= (const Set& l) const {
         return !(*this == l); }
      /** Check if events are stored in proper time order.
          @memo Check order
       ******************************************************************/
      bool CheckOrder() const;
      /** Swap the list.
          @memo Swap
       ******************************************************************/
      void Swap (Set& l);
      /** Lower bound.
          @memo Lower bound
       ******************************************************************/
      iterator LowerBound (const Time& t) {
         return LowerBound (Event (t)); }
      /** Lower bound.
          @memo Lower bound
       ******************************************************************/
      const_iterator LowerBound (const Time& t) const {
         return LowerBound (Event (t)); }
      /** Upper bound.
          @memo Upper bound
       ******************************************************************/
      iterator UpperBound (const Time& t) {
         return UpperBound (Event (t)); }
      /** Upper bound.
          @memo Upper bound
       ******************************************************************/
      const_iterator UpperBound (const Time& t) const {
         return UpperBound (Event (t)); }
      /** Lower bound.
          @memo Lower bound
       ******************************************************************/
      iterator LowerBound (const Event& t);
      /** Lower bound.
          @memo Lower bound
       ******************************************************************/
      const_iterator LowerBound (const Event& t) const;
      /** Upper bound.
          @memo Upper bound
       ******************************************************************/
      iterator UpperBound (const Event& t);
      /** Upper bound.
          @memo Upper bound
       ******************************************************************/
      const_iterator UpperBound (const Event& t) const;
      /** Sort the set by event time (the set will be merged first).
          @memo Sort
       ******************************************************************/
      void Sort();
   
      /** At. Rather inefficient function.
          @memo At
       ******************************************************************/
      reference At (size_type idx);
      /** At. Rather inefficient function.
          @memo At
       ******************************************************************/
      const_reference At (size_type idx) const;
      /** Operator []. Rather inefficient function.
          @memo Operator[]
       ******************************************************************/
      reference operator[] (size_type idx) {
         return At (idx); }
      /** Operator []. Rather inefficient function.
          @memo Operator[]
       ******************************************************************/
      const_reference operator[] (size_type idx) const {
         return At (idx); }
      /** Front.
          @memo Front
       ******************************************************************/
      reference Front();
      /** Front.
          @memo Front
       ******************************************************************/
      const_reference Front() const;
      /** Back.
          @memo Back
       ******************************************************************/
      reference Back();
      /** Back.
          @memo Back
       ******************************************************************/
      const_reference Back() const;
      /** Begin iterator. 
          @memo Begin
          @return Iterator
       ******************************************************************/
      iterator Begin();   
      /** Begin iterator. 
          @memo Begin
          @return Iterator
       ******************************************************************/
      const_iterator Begin() const;   
      /** End iterator. 
          @memo End
          @return Const iterator
       ******************************************************************/
      iterator End();   
      /** End iterator. 
          @memo End
          @return Const iterator
       ******************************************************************/
      const_iterator End() const;   
      /** RBegin.
          @memo RBegin
       ******************************************************************/
      reverse_iterator RBegin () {
         return reverse_iterator (End()); }
      /** RBegin.
          @memo RBegin
       ******************************************************************/
      const_reverse_iterator RBegin () const {
         return const_reverse_iterator (End()); }
      /** REnd.
          @memo REnd
       ******************************************************************/
      reverse_iterator REnd () {
         return reverse_iterator (Begin()); }
      /** REnd.
          @memo REnd
       ******************************************************************/
      const_reverse_iterator REnd () const {
         return const_reverse_iterator (Begin()); }
   
      /** Insert an event into the default chain at its proper location.
          @memo Insert
       ******************************************************************/
      void Insert (const Event& event);
      /** Insert an event to the set at the specified location.
          @memo Insert
       ******************************************************************/
      iterator Insert (const iterator& pos, const Event& event);
      /** Inserts a range of events.
          @memo Add
       ******************************************************************/
      void Insert (const iterator& beg, const iterator& end);
      /** Inserts an event at the back.
          @memo PushBack
       ******************************************************************/
      void PushBack (const Event& event);
      /** Erase an event. Returns an iterator pointing the next event.
          @memo Erase
       ******************************************************************/
      iterator Erase (const iterator& pos);
      /** Erase a range of events. Returns an iterator pointing the 
          next event.
          @memo Erase
       ******************************************************************/
      iterator Erase (const iterator& beg, const iterator& end);
      /** Remove an event from the end of the default chain. 
          @memo PopBack
       ******************************************************************/
      void PopBack();
      /** Clear the chain.
          @memo Clear
       ******************************************************************/
      void Clear () {
         mSet.clear(); mDefault = 0; }
   
      /** Set the time window of the analysis. A sliding window is
          used to loop over the events. Only events within this window
          are considered for veto or coincidence. The window must be
          at least as wide as the larges coincidenc or veto window.
          @memo Set time window
       ******************************************************************/
      void SetWindow (const TimeWindow& window) {
         mWindow = window; }
      /** Set the time window of the analysis. A sliding window is
          used to loop over the events. Only events within this window
          are considered for veto or coincidence. The window must be
          at least as wide as the larges coincidenc or veto window.
          @memo Set time window
          @param width Window duration
          @param offset Window offset
       ******************************************************************/
      void SetWindow (const Interval& width, const Interval& offset) {
         mWindow = TimeWindow (width, offset); }
      /** Set the time offset of the window.
          @memo Set time offset
       ******************************************************************/
      void SetOffset (const Interval& offset) {
         mWindow.SetOffset (offset); }
      /** Get the time offset of the window.
          @memo Get time offset
       ******************************************************************/
      Interval GetOffset () const {
         return mWindow.GetOffset(); }
      /** Set the time duration of the window.
          @memo Set time duration
       ******************************************************************/
      void SetWidth (const Interval& duration) {
         mWindow.SetOffset (duration); }
      /** Get the time duration of the window.
          @memo Get time duration
       ******************************************************************/
      Interval GetWidth () const {
         return mWindow.GetWidth(); }
   
   	/** Set column to a new value. Loops through the list of events
    		and calculates the expression for each event. The specified 
    		column is the set to then new value. The column name can be
    		either a string or a column class.
   
    		Example:
    		\begin{verbatim}
    		// Calculate the significance from the amplitude and sigma
    		Set s1 ("input.xml");
    		s1.SetColumn (Column ("Significance"),
                       Column ("Amplitude") / Column ("Sigma"));
    		\end{verbatim}
   
    		@memo Set column to a new value
    		@param column Name of column/Column object
    		@param expression New column value
    		@return void
   	************************************************************************/
      void SetColumn (const Column& column, const Function& expression);
   
      void SetColumn (const Column& column, const FunctionPtr& expression) {
         SetColumn (column, *expression); }
   
   	/** Set column to a new value. Loops through the list of events
    		and calculates the expression for each event which fullfills
    		the condition. The specified column is then set to the new value.
    		The column name can be either a string or a column class.
   
    	Example:
    	\begin{verbatim}
    	// Calculate the significance from the amplitude and sigma
    	// for all burts events
    		Set s1 ("input.xml");
    		s1.SetColumn (Column ("Significance"),
       	  	           Column ("Amplitude") / Column ("Sigma"),
        		           Filter ("burst::*"));
    	\end{verbatim}
   
    		@memo Set column to a new value of selected events
    		@param column Name of column/Column object
    		@param expression New column value
    		@param cond Event condition
    		@param window Time window for analysis
    		@return void
   	************************************************************************/
      void SetColumn (const Column& column, const Function& expression,
                     const Condition& cond, const TimeWindow& window = TimeWindow (1.));
   
      void SetColumn (const Column& column, const FunctionPtr& expression,
                     const ConditionPtr& cond, const TimeWindow& window = TimeWindow (1.)) {           
         SetColumn (column, *expression, *cond, window); }
   
      /** Selects the events which fullfill the event condition.
          @memo Select
          @param cond Event condition
          @return Number of selected events
       ******************************************************************/
      int Select (const Condition& cond);
      int Select (const ConditionPtr& cond) {
         return Select (*cond); }
      /** Selects the events from the specified set which fullfill the 
          event condition. The destination set is cleared first.
          @memo Select
          @param events Set to pick events from
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int Select (const Set& events, const Condition& cond);
      int Select (const Set& events, const ConditionPtr& cond) {
         return Select (events, *cond); }
      /** Selects the events from the specified set which fullfill the 
          event condition and adds them to the current set.
          @memo Select and add
          @param events Set to pick events from
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int SelectAdd (const Set& events, const Condition& cond);
      int SelectAdd (const Set& events, const ConditionPtr& cond) {
         return SelectAdd (events, *cond); }
   
      /** Sorts the events using the event function.
          @memo Sort
          @param func Event function
          @param ascending Sort order (true default)
          @param n Maximum number of sorted events (0 = all)
          @return Number of sorted events
       ******************************************************************/
      int Sort (const Function& func, bool ascending = true, int n = 0);
      int Sort (const FunctionPtr& func, bool ascending = true,
               int n = 0) {
         return Sort (*func, ascending, n); }
      /** Sorts the events form the specified set using the event function.
          @memo Sort
          @param func Event function
          @param ascending Sort order (true default)
          @param n Maximum number of sorted events (0 = all)
          @return Number of sorted events
       ******************************************************************/
      int Sort (const Set& events, const Function& func, bool ascending = true, 
               int n = 0);
      int Sort (const Set& events, const FunctionPtr& func, 
               bool ascending = true, int n = 0) {
         return Sort (events, *func, ascending, n); }
   
      /** Selects the events which fullfill the event coincidence 
          condition. Uses the time window of the set.
          If two events e1 and e2 fullfill the event condition, a 
          multi event consisting of e1 and e2 is added.
          @memo Coincidence
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int Coincidence (const Condition& cond = Condition::True()) {
         return Coincidence (mWindow, cond); }
      int Coincidence (const ConditionPtr& cond) {
         return Coincidence (mWindow, cond); }
      /** Selects the events which fullfill the event coincidence 
          condition. Specifies the time window explicitly.
          If two events e1 and e2 fullfill the event condition, a 
          multi event consisting of e1 and e2 is added.
          @memo Coincidence
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int Coincidence (const TimeWindow& window,
                      const Condition& cond = Condition::True()) {
         return MultiCoincidence (2, window, cond); }
      int Coincidence (const TimeWindow& window,
                      const ConditionPtr& cond) {
         return MultiCoincidence (2, window, cond); }
      /** Selects the events from the specified set which fullfill the 
          event coincidence condition. The destination set is cleared 
          first. Uses the time window of the set.
          If two events e1 and e2 fullfill the event condition, a 
          multi event consisting of e1 and e2 is added.
          @memo Coincidence
          @param events Set to pick events from
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int Coincidence (const Set& events, 
                      const Condition& cond = Condition::True()) {
         return Coincidence (events, mWindow, cond); }
      int Coincidence (const Set& events, 
                      const ConditionPtr& cond) {
         return Coincidence (events, mWindow, cond); }
      /** Selects the events from the specified set which fullfill the 
          event coincidence condition. The destination set is cleared 
          first. Specifies the time window explicitly.
          If two events e1 and e2 fullfill the event condition, a 
          multi event consisting of e1 and e2 is added.
          @memo Coincidence
          @param events Set to pick events from
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int Coincidence (const Set& events, const TimeWindow& window,
                      const Condition& cond = Condition::True()) {
         return MultiCoincidence (2, events, window, cond); }
      int Coincidence (const Set& events, const TimeWindow& window,
                      const ConditionPtr& cond) {
         return MultiCoincidence (2, events, window, cond); }
      /** Selects the events from the specified set which fullfill the 
          event coincidence condition and adds them to the current set.
          Uses the time window of the set.
          If two events e1 and e2 fullfill the event condition, a 
          multi event consisting of e1 and e2 is added.
          @memo Coincidence and add
          @param events Set to pick events from
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int CoincidenceAdd (const Set& events,
                        const Condition& cond = Condition::True()) {
         return CoincidenceAdd (events, mWindow, cond); }
      int CoincidenceAdd (const Set& events, const ConditionPtr& cond) {
         return CoincidenceAdd (events, mWindow, cond); }
      /** Selects the events from the specified set which fullfill the 
          event coincidence condition and adds them to the current set.
          Specifies the time window explicitly.
          If two events e1 and e2 fullfill the event condition, a 
          multi event consisting of e1 and e2 is added.
          @memo Coincidence and add
          @param events Set to pick events from
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int CoincidenceAdd (const Set& events, const TimeWindow& window,
                        const Condition& cond = Condition::True()) {
         return MultiCoincidenceAdd (2, events, window, cond); }   
      int CoincidenceAdd (const Set& events, const TimeWindow& window,
                        const ConditionPtr& cond) {
         return MultiCoincidenceAdd (2, events, window, cond); }   
      /** Selects the events which fullfill the event triple coincidence.
          Uses the time window of the set.
          If three events e1, e2 and e3 fullfill the event condition, a 
          multi event consisting of e1, e2 and e3 is added.
          @memo TripleCoincidence
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int TripleCoincidence (const Condition& cond = Condition::True()) {
         return TripleCoincidence (mWindow, cond); }
      int TripleCoincidence (const ConditionPtr& cond) {
         return TripleCoincidence (mWindow, cond); }
   
      /** Selects the events which fullfill the event triple coincidence.
          Specifies the time window explicitly.
          If three events e1, e2 and e3 fullfill the event condition, a 
          multi event consisting of e1, e2 and e3 is added.
          @memo TripleCoincidence
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int TripleCoincidence (const TimeWindow& window,
                        const Condition& cond = Condition::True()) {
         return MultiCoincidence (3, window, cond); }
      int TripleCoincidence (const TimeWindow& window,
                        const ConditionPtr& cond) {
         return MultiCoincidence (3, window, cond); }
      /** Selects the events from the specified set which fullfill the 
          event triple coincidence condition. The destination set is 
          cleared first. Uses the time window of the set.
          If three events e1, e2 and e3 fullfill the event condition, a 
          multi event consisting of e1, e2 and e3 is added.
          @memo TripleCoincidence
          @param events Set to pick events from
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int TripleCoincidence (const Set& events,
                        const Condition& cond = Condition::True()) {
         return TripleCoincidence (events, mWindow, cond); }
      int TripleCoincidence (const Set& events, const ConditionPtr& cond) {
         return TripleCoincidence (events, mWindow, cond); }
      /** Selects the events from the specified set which fullfill the 
          event triple coincidence condition. The destination set is 
          cleared first. Specifies the time window explicitly.
          If three events e1, e2 and e3 fullfill the event condition, a 
          multi event consisting of e1, e2 and e3 is added.
          @memo TripleCoincidence
          @param events Set to pick events from
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int TripleCoincidence (const Set& events, const TimeWindow& window,
                        const Condition& cond = Condition::True()) {
         return MultiCoincidence (3, events, window, cond); }
      int TripleCoincidence (const Set& events, const TimeWindow& window,
                        const ConditionPtr& cond) {
         return MultiCoincidence (3, events, window, cond); }
      /** Selects the events from the specified set which fullfill the 
          event triple coincidence condition and adds them to the 
          current set. Uses the time window of the set.
          If three events e1, e2 and e3 fullfill the event condition, a 
          multi event consisting of e1, e2 and e3 is added.
          @memo TripleCoincidence Add
          @param events Set to pick events from
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int TripleCoincidenceAdd (const Set& events,
                        const Condition& cond = Condition::True()) {
         return TripleCoincidenceAdd (events, mWindow, cond); }
      int TripleCoincidenceAdd (const Set& events, const ConditionPtr& cond) {
         return TripleCoincidenceAdd (events, mWindow, cond); }
      /** Selects the events from the specified set which fullfill the 
          event triple coincidence condition and adds them to the 
          current set. Specifies the time window explicitly.
          If three events e1, e2 and e3 fullfill the event condition, a 
          multi event consisting of e1, e2 and e3 is added.
          @memo TripleCoincidence Add
          @param events Set to pick events from
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int TripleCoincidenceAdd (const Set& events, const TimeWindow& window,
                        const Condition& cond = Condition::True()) {
         return MultiCoincidenceAdd (3, events, window, cond); }
      int TripleCoincidenceAdd (const Set& events, const TimeWindow& window,
                        const ConditionPtr& cond) {
         return MultiCoincidenceAdd (3, events, window, cond); }
   
      /** Selects the events which fullfill the event multi coincidence.
          Uses the time window of the set.
          If events e1, ... eN fullfill the event condition, a 
          multi event consisting of e1, ... en is added.
          @memo MultiCoincidence
          @param n Number of required coincidences
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int MultiCoincidence (int n,
                        const Condition& cond = Condition::True()) {
         return MultiCoincidence (n, mWindow, cond); }
      int MultiCoincidence (int n, const ConditionPtr& cond) {
         return MultiCoincidence (n, mWindow, cond); }
      /** Selects the events which fullfill the event multi coincidence.
          Specifies the time window explicitly.
          If events e1, ... eN fullfill the event condition, a 
          multi event consisting of e1, ... en is added.
          @memo MultiCoincidence
          @param n Number of required coincidences
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int MultiCoincidence (int n, const TimeWindow& window,
                        const Condition& cond = Condition::True());
      int MultiCoincidence (int n, const TimeWindow& window,
                        const ConditionPtr& cond) {
         return MultiCoincidence (n, window, *cond); }
      /** Selects the events from the specified set which fullfill the 
          event multi coincidence condition. The destination set is 
          cleared first. Uses the time window of the set.
          If events e1, ... eN fullfill the event condition, a 
          multi event consisting of e1, ... en is added.
          @memo MultiCoincidence
          @param n Number of required coincidences
          @param events Set to pick events from
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int MultiCoincidence (int n, const Set& events,
                        const Condition& cond = Condition::True()) {
         return MultiCoincidence (n, events, mWindow, cond); }
      int MultiCoincidence (int n, const Set& events,
                        const ConditionPtr& cond) {
         return MultiCoincidence (n, events, mWindow, cond); }
      /** Selects the events from the specified set which fullfill the 
          event multi coincidence condition. The destination set is 
          cleared first. Specifies the time window explicitly.
          If events e1, ... eN fullfill the event condition, a 
          multi event consisting of e1, ... en is added.
          @memo MultiCoincidence
          @param n Number of required coincidences
          @param events Set to pick events from
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int MultiCoincidence (int n, const Set& events, const TimeWindow& window,
                        const Condition& cond = Condition::True());
      int MultiCoincidence (int n, const Set& events, const TimeWindow& window,
                        const ConditionPtr& cond) {
         return MultiCoincidence (n, events, window, *cond); }
   
      /** Selects the events from the specified set which fullfill the 
          event multi coincidence condition and adds them to the 
          current set. Uses the time window of the set.
          If events e1, ... eN fullfill the event condition, a 
          multi event consisting of e1, ... en is added.
          @memo MultiCoincidence Add
          @param n Number of required coincidences
          @param events Set to pick events from
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int MultiCoincidenceAdd (int n, const Set& events,
                        const Condition& cond = Condition::True()) {
         return MultiCoincidenceAdd (n, events, mWindow, cond); }
      int MultiCoincidenceAdd (int n, const Set& events,
                        const ConditionPtr& cond) {
         return MultiCoincidenceAdd (n, events, mWindow, cond); }
      /** Selects the events from the specified set which fullfill the 
          event multi coincidence condition and adds them to the 
          current set. Specifies the time window explicitly.
          If events e1, ... eN fullfill the event condition, a 
          multi event consisting of e1, ... en is added.
          @memo MultiCoincidence Add
          @param n Number of required coincidences
          @param events Set to pick events from
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int MultiCoincidenceAdd (int n, const Set& events, 
                        const TimeWindow& window,
                        const Condition& cond = Condition::True());
      int MultiCoincidenceAdd (int n, const Set& events, 
                        const TimeWindow& window,
                        const ConditionPtr& cond) {
         return MultiCoincidenceAdd (n, events, window, *cond); }
   
      /** Selects the events which fullfill the event coincidence 
          condition. Uses the time window of the set.
          The (multi-)coincidence is formed between all chains
          of the set.
          If two events e1 and e2 fullfill the event condition, a 
          multi event consisting of e1 and e2 is added.
          @memo Coincidence
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int ChainCoincidence (const Condition& cond = Condition::True()) {
         return ChainCoincidence (mWindow, cond); }
      int ChainCoincidence (const ConditionPtr& cond) {
         return ChainCoincidence (mWindow, cond); }
      /** Selects the events which fullfill the event coincidence 
          condition. Specifies the time window explicitly.
          The (multi-)coincidence is formed between all chains
          of the set.
          If two events e1 and e2 fullfill the event condition, a 
          multi event consisting of e1 and e2 is added.
          @memo Coincidence
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int ChainCoincidence (const TimeWindow& window,
                        const Condition& cond = Condition::True());
      int ChainCoincidence (const TimeWindow& window,
                        const ConditionPtr& cond) {
         return ChainCoincidence (window, *cond); }
      /** Selects the events from the specified set which fullfill the 
          event coincidence condition. The destination set is cleared 
          first. Uses the time window of the set.
          The (multi-)coincidence is formed between all chains
          of the specified set.
          If two events e1 and e2 fullfill the event condition, a 
          multi event consisting of e1 and e2 is added.
          @memo Coincidence
          @param events Set to pick events from
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int ChainCoincidence (const Set& events, 
                        const Condition& cond = Condition::True()) {
         return ChainCoincidence (events, mWindow, cond); }
      int ChainCoincidence (const Set& events, 
                        const ConditionPtr& cond) {
         return ChainCoincidence (events, mWindow, cond); }
      /** Selects the events from the specified set which fullfill the 
          event coincidence condition. The destination set is cleared 
          first. Specifies the time window explicitly.
          The (multi-)coincidence is formed between all chains
          of the specified set.
          If two events e1 and e2 fullfill the event condition, a 
          multi event consisting of e1 and e2 is added.
          @memo Coincidence
          @param events Set to pick events from
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int ChainCoincidence (const Set& events, const TimeWindow& window,
                        const Condition& cond = Condition::True());
      int ChainCoincidence (const Set& events, const TimeWindow& window,
                        const ConditionPtr& cond) {
         return ChainCoincidence (events, window, *cond); }
      /** Selects the events from the specified set which fullfill the 
          event coincidence condition and adds them to the current set.
          Uses the time window of the set.
          The (multi-)coincidence is formed between all chains
          of the specified set.
          If two events e1 and e2 fullfill the event condition, a 
          multi event consisting of e1 and e2 is added.
          @memo Coincidence and add
          @param events Set to pick events from
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int ChainCoincidenceAdd (const Set& events,
                        const Condition& cond = Condition::True()) {
         return ChainCoincidenceAdd (events, mWindow, cond); }
      int ChainCoincidenceAdd (const Set& events, 
                        const ConditionPtr& cond) {
         return ChainCoincidenceAdd (events, mWindow, cond); }
      /** Selects the events from the specified set which fullfill the 
          event coincidence condition and adds them to the current set.
          Specifies the time window explicitly.
          The (multi-)coincidence is formed between all chains
          of the specified set.
          If two events e1 and e2 fullfill the event condition, a 
          multi event consisting of e1 and e2 is added.
          @memo Coincidence and add
          @param events Set to pick events from
          @param window Coincidence interval
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int ChainCoincidenceAdd (const Set& events, 
                        const TimeWindow& window,
                        const Condition& cond = Condition::True());   
      int ChainCoincidenceAdd (const Set& events, 
                        const TimeWindow& window,
                        const ConditionPtr& cond) {
         return ChainCoincidenceAdd (events, window, *cond); }    
   
      /** Selects the events which fullfill the event condition.
          Uses the time window of the set.
          @memo Select
   	  @param threshold Number of events to be considered a cluster
          @param cond Event condition
          @return Number of selected events
       ******************************************************************/
      int Clusters (int threshold, 
                   const Condition& cond = Condition::True()) {
         return Clusters (threshold, mWindow, cond); }
      int Clusters (int threshold, const ConditionPtr& cond) {
         return Clusters (threshold, mWindow, cond); }
   
      /** Selects the events which fullfill the event condition.
          Specifies the time window explicitly.
          @memo Select
   	  @param threshold Number of events to be considered a cluster
   	  @param window Duration of cluster window
          @param cond Event condition
          @return Number of selected events
       ******************************************************************/
      int Clusters (int threshold, const TimeWindow& window, 
                   const Condition& cond = Condition::True());
      int Clusters (int threshold, const TimeWindow& window, 
                   const ConditionPtr& cond) {
         return Clusters (threshold, window, *cond); }
      /** Selects the events from the specified set which fullfill the 
          event condition. The destination set is cleared first.
          Uses the time window of the set.
          @memo Select
          @param events Set to pick events from
   	  @param threshold Number of events to be considered a cluster
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int Clusters (const Set& events, int threshold, 
                   const Condition& cond = Condition::True()) {
         return Clusters (events, threshold, mWindow, cond); }
      int Clusters (const Set& events, int threshold, 
                   const ConditionPtr& cond) {
         return Clusters (events, threshold, mWindow, cond); }
      /** Selects the events from the specified set which fullfill the 
          event condition. The destination set is cleared first.
          Specifies the time window explicitly.
          @memo Select
          @param events Set to pick events from
   	  @param threshold Number of events to be considered a cluster
   	  @param window Duration of cluster window
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int Clusters (const Set& events, int threshold, 
                   const TimeWindow& window,
                   const Condition& cond = Condition::True());
      int Clusters (const Set& events, int threshold, 
                   const TimeWindow& window,
                   const ConditionPtr& cond) {
         return Clusters (events, threshold, window, *cond);}
      /** Selects the cluster events from the specified set which 
          fullfill the event condition and adds them to the current set.
          Uses the time window of the set.
          @memo Select and add
          @param events Set to pick events from
   	  @param threshold Number of events to be considered a cluster
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int ClusterAdd (const Set& events, int threshold, 
                     const Condition& cond = Condition::True()) {
         return ClusterAdd (events, threshold, mWindow, cond); }
      int ClusterAdd (const Set& events, int threshold, 
                     const ConditionPtr& cond) {
         return ClusterAdd (events, threshold, mWindow, cond); }
      /** Selects the cluster events from the specified set which 
          fullfill the event condition and adds them to the current set.
          Specifies the time window explicitly.
          @memo Select and add
          @param events Set to pick events from
   	  @param threshold Number of events to be considered a cluster
   	  @param window Duration of cluster window
          @param cond Event condition
          @return Number of added/selected events
       ******************************************************************/
      int ClusterAdd (const Set& events, int threshold, 
                     const TimeWindow& window,
                     const Condition& cond = Condition::True());
      int ClusterAdd (const Set& events, int threshold, 
                     const TimeWindow& window,
                     const ConditionPtr& cond) {
         return ClusterAdd (events, threshold, window, *cond); }
   
      /** Make a one-dimensional histogram. The specified event column 
          is used to add to the histogram all events fullfilling the
          event condition.
          @memo 1D histogram generation
          @param hist 1D histogram
          @param func Event function used for histogram axis
          @param cond Condition to select events
          @return Number of added events
       ******************************************************************/
      int Histogram (Histogram1& hist, const Function& func, 
                    const Condition& cond = Condition::True()) const {
         return Histogram (hist, func, mWindow, cond); }
      int Histogram (Histogram1& hist, const FunctionPtr& func, 
                    const ConditionPtr& cond) const {
         return Histogram (hist, func, mWindow, cond); }
      /** Make a one-dimensional histogram. The specified event column 
          is used to add to the histogram all events fullfilling the
          event condition.
          @memo 1D histogram generation
          @param hist 1D histogram
          @param func Event function used for histogram axis
   	  @param window Duration of time window
          @param cond Condition to select events
          @return Number of added events
       ******************************************************************/
      int Histogram (Histogram1& hist, const Function& func, 
                    const TimeWindow& window,
                    const Condition& cond = Condition::True()) const;
      int Histogram (Histogram1& hist, const FunctionPtr& func, 
                    const TimeWindow& window,
                    const ConditionPtr& cond) const {
         return Histogram (hist, *func, window, *cond); }
   
      /** Make a two-dimensional histogram. The specified event columns 
          are used to add to the histogram all events fullfilling the
          event condition.
          @memo 1D histogram generation
          @param hist 1D histogram
          @param f1 Event function used for first histogram axis
          @param f2 Event function used for second histogram axis
          @param cond Condition to select events
          @return Number of added events
       ******************************************************************/
      int Histogram (Histogram2& hist, const Function& f1, 
                    const Function& f2, 
                    const Condition& cond = Condition::True()) const {
         return Histogram (hist, f1, f2, mWindow, cond); }
      int Histogram (Histogram2& hist, const FunctionPtr& f1, 
                    const FunctionPtr& f2, 
                    const ConditionPtr& cond) const {
         return Histogram (hist, f1, f2, mWindow, cond); }
      /** Make a two-dimensional histogram. The specified event columns 
          are used to add to the histogram all events fullfilling the
          event condition.
          @memo 1D histogram generation
          @param hist 1D histogram
          @param f1 Event function used for first histogram axis
          @param f2 Event function used for second histogram axis
   	  @param window Duration of time window
          @param cond Condition to select events
          @return Number of added events
       ******************************************************************/
      int Histogram (Histogram2& hist, const Function& f1, 
                    const Function& f2, const TimeWindow& window,
                    const Condition& cond = Condition::True()) const;
      int Histogram (Histogram2& hist, const FunctionPtr& f1, 
                    const FunctionPtr& f2, const TimeWindow& window,
                    const ConditionPtr& cond) const {
         return Histogram (hist, *f1, *f2, *cond); }
   
      /** Make a time series. The event rate is plotted against time.
          @memo Time series generation
          @param ts Time series
          @param integrated If true plots the integrated event rate
          @param cond Condition to select events
          @return Number of events
       ******************************************************************/
      int TimeSeries (TSeries& ts, bool integrated = false,
                     const Condition& cond = Condition::True()) const {
         return TimeSeries (ts, mWindow, integrated, cond); }
      int TimeSeries (TSeries& ts, bool integrated,
                     const ConditionPtr& cond) const {
         return TimeSeries (ts, mWindow, integrated, cond); } 
      /** Make a time series. The event rate is plotted against time.
          @memo Time series generation
          @param ts Time series
   	  @param window Duration of time window
          @param integrated If true plots the integrated event rate
          @param cond Condition to select events
          @return Number of events
       ******************************************************************/
      int TimeSeries (TSeries& ts, const TimeWindow& window, 
                     bool integrated = false,
                     const Condition& cond = Condition::True()) const;
      int TimeSeries (TSeries& ts, const TimeWindow& window,
                     bool integrated, const ConditionPtr& cond) const {
         return TimeSeries (ts, window, integrated, *cond); } 
   
      /** Make a time series. The specified event column 
          is plotted against time.
          @memo Time series generation
          @param ts Time series
          @param func Event function used for Y axis
          @param cond Condition to select events
          @return Number of events
       ******************************************************************/
      int TimeSeries (TSeries& ts, const Function& func, 
                     const Condition& cond = Condition::True()) const;
      int TimeSeries (TSeries& ts, const FunctionPtr& func, 
                     const ConditionPtr& cond) const {
         return TimeSeries (ts, *func, *cond); } 
   
      /**  Dump Event data in the Set to the specified output stream
           @memo Dump Event data in the Set to the specified output stream
           @param os output stream
           @param num max number of events
   	******************************************************************/
      void Dump (std::ostream& os, int num = 0) const;
      /**  Dump Event data in the Set to the standard output stream
           @memo Dump Event data in the Set to the specified output stream
           @param os output stream
           @param num max number of events
   	******************************************************************/
      void Dump (int num = 0) const;
   
      /**  Dump column data in the Set to the specified output stream
           @memo Dump column data in the Set to the specified output stream
           @param name column name
           @param os output stream
           @param num max number of events
   	******************************************************************/	
      void DumpColumn (const char* name, std::ostream& os, 
                      int num = 0) const;
      /**  Dump column data in the Set to the standard output stream
           @memo Dump column data in the Set to the specified output stream
           @param name column name
           @param os output stream
           @param num max number of events
   	******************************************************************/	
      void DumpColumn (const char* name, int num = 0) const;
   
   protected:
      /**  Check that at least one chain and the default exists
           @memo Check for a valid chain
   	******************************************************************/	
      void CheckChain();
   
   private:
      /// List of event chains
      eventset		mSet;
      /// Default chain
      int		mDefault;
      /// Analysis window (in sec)
      TimeWindow	mWindow;
   };


}

#endif // _LIGO_EVENTSET_H
