//////////////////////////////////////////////////////////////////////////
//  									//
//  List								//
//  									//
//////////////////////////////////////////////////////////////////////////

#include <time.h>
#include <iostream>
#include <fstream>
#include "events/List.hh"
#include "events/Algorithm.hh"
#include "events/AlgorithmUtil.hh"


namespace events {

//______________________________________________________________________________
   class IteratorImpList : public IteratorImp {
   public:
      typedef List::eventlist::iterator list_iterator;
      // Create event list iterator implementation
      IteratorImpList (list_iterator i) 
      : mIter (i) {
      }
      // Copy
      virtual IteratorImp* Copy() const {
         return new IteratorImpList (*this); }
      // Get
      virtual Event* Get() const {
         return &*mIter; }
      // Increment
      virtual void Inc() {
         ++mIter; }
      // Decrement
      virtual void Dec() {
         --mIter; }
      list_iterator& GetListIterator() {
         return mIter; }
      const list_iterator& GetListIterator() const {
         return mIter; }
   private:
      list_iterator	mIter;
   };

//______________________________________________________________________________
   bool List::CheckOrder() const
   {
      return events::CheckOrder (Begin(), End());
   }

//______________________________________________________________________________
   void List::Swap (List& l)
   {
      mList.swap (l.mList);
      std::swap (mFilename, l.mFilename);
      std::swap (mState, l.mState);
   }

//______________________________________________________________________________
   List::iterator List::LowerBound (const Event& e)
   {
      return iterator (IteratorImpList 
                      (std::lower_bound (mList.begin(), mList.end(), e)));
   }

//______________________________________________________________________________
   List::const_iterator List::LowerBound (const Event& e) const
   {
      return const_iterator 
         (IteratorImpList 
         (std::lower_bound (((eventlist&)mList).begin(), 
                           ((eventlist&)mList).end(), e)));
   }

//______________________________________________________________________________
   List::iterator List::UpperBound (const Event& e)
   {
      return iterator (IteratorImpList 
                      (std::upper_bound (mList.begin(), mList.end(), e)));
   }

//______________________________________________________________________________
   List::const_iterator List::UpperBound (const Event& e) const
   {
      return const_iterator 
         (IteratorImpList 
         (std::upper_bound (((eventlist&)mList).begin(), 
                           ((eventlist&)mList).end(), e)));
   }

//______________________________________________________________________________
   void List::Sort ()
   {
      std::sort (mList.begin(), mList.end());
   }

//______________________________________________________________________________
   void List::Sort (const Function& func, bool ascending)
   {
      // create an array of sort elements
      sortarray sarr;
      for (eventlist::iterator i = mList.begin(); i != mList.end(); ++i) {
         sarr.push_back (SortElement (*i, func, ascending));
      }
      // sort it
      if (ascending) {
         std::sort (sarr.begin(), sarr.end());
      }
      else {
         std::sort (sarr.begin(), sarr.end(), 
                   std::greater<SortElement>());
      }
      // Now swap original events into correct order
      eventlist temp (sarr.size());
      eventlist::iterator i = temp.begin();
      for (sortarray::iterator j = sarr.begin(); j != sarr.end(); 
          ++j, ++i) {
         std::swap (*const_cast<Event*>(j->Get()), *i);
      }
      i = temp.begin();
      for (eventlist::iterator j = mList.begin(); j != mList.end(); ++j, ++i) {
         std::swap (*j, *i); 
      }
   }

//______________________________________________________________________________
   List::iterator List::Begin()
   {
      return iterator (IteratorImpList (mList.begin()));
   }

//______________________________________________________________________________
   List::iterator List::End()
   {
      return iterator (IteratorImpList (mList.end()));
   }

//______________________________________________________________________________
   List::const_iterator List::Begin() const
   {
      return const_iterator (IteratorImpList (((eventlist&)mList).begin()));
   }

//______________________________________________________________________________
   List::const_iterator List::End() const
   {
      return const_iterator (IteratorImpList (((eventlist&)mList).end()));
   }

//______________________________________________________________________________
   List::iterator List::Insert (const Event& event)
   {
      List::eventlist::iterator i = std::upper_bound 
         (mList.begin(), mList.end(), event);
      return iterator (IteratorImpList (mList.insert (i, event)));
   }

//______________________________________________________________________________
   List::iterator List::Insert (const iterator& pos, const Event& event)
   {
      const IteratorImpList* imp = 
         dynamic_cast<const IteratorImpList*> (pos.GetImplementation());
      if (imp == 0) {
         return End();
      }
      return iterator (IteratorImpList 
                      (mList.insert (imp->GetListIterator(), event)));
   }

//______________________________________________________________________________
   void List::Insert (const iterator& beg, const iterator& end)
   {
      for (iterator i = beg; i != end; ++i) {
         Insert (*i);
      }
   }

//______________________________________________________________________________
   List::iterator List::Erase (const iterator& pos)
   {
      const IteratorImpList* imp = 
         dynamic_cast<const IteratorImpList*> (pos.GetImplementation());
      if (imp == 0) {
         return End();
      }
      return iterator (IteratorImpList 
                      (mList.erase (imp->GetListIterator())));
   }

//______________________________________________________________________________
   List::iterator List::Erase (const iterator& beg, const iterator& end)
   {
      const IteratorImpList* imp1 = 
         dynamic_cast<const IteratorImpList*> (beg.GetImplementation());
      if (imp1 == 0) {
         return End();
      }
      const IteratorImpList* imp2 = 
         dynamic_cast<const IteratorImpList*> (end.GetImplementation());
      if (imp2 == 0) {
         return End();
      }
      return iterator (IteratorImpList 
                      (mList.erase (imp1->GetListIterator(),
                                   imp2->GetListIterator())));
   }

//______________________________________________________________________________
   List::state List::GetState() const
   {
      state s = mState;
      if ((s == kEmpty) && !mList.empty()) {
         s = kModified;
      }
      else if ((s != kEmpty) && mList.empty()) {
         s = kEmpty;
      }
      return s;
   }

//______________________________________________________________________________
   bool List::Load (bool keepforever)
   {
      Clear();
      std::ifstream inp (mFilename.c_str());
      if (!inp) {
         std::cerr << "Failed to open " << mFilename << std::endl;
         return false;
      }
      Read (inp, std::back_inserter (*this));
      if (!CheckOrder()) {
         Sort();
      }
      mState = kLocked;
      return true;
   }

//______________________________________________________________________________
   bool List::Unload()
   {
      if (mState == kLoaded) {
         Clear();
         mState = kEmpty;
      }
      return mState == kEmpty;
   }

//______________________________________________________________________________
   bool List::Save (const char* filename) const
   {
      std::ofstream out (filename);
      if (!out) {
         return false;
      }
      Write (out, Begin(), End());
      return !!out;
   }

}
