//////////////////////////////////////////////////////////////////////////
//  									//
//  Layout								//
//  									//
//////////////////////////////////////////////////////////////////////////

#include <time.h>
#include <iostream>
#include <stdio.h>
#include "events/Layout.hh"
#include "events/LayoutInfo.hh"
#include "events/Factory.hh"
#include <algorithm>


namespace events {

   using namespace std;

//______________________________________________________________________________
   Layout Layout::gSimple = Layout ();
   Layout Layout::gStandard = Layout ();

//______________________________________________________________________________
   Layout::Layout (const LayoutInfo* info)
   : mInfo (0) 
   {
      if (info && info->IsRegistered()) {
         mInfo = (LayoutInfo*)info;
      }
      else {
         mInfo = info ? info->Copy() : 0;
      }
      if (mInfo) mInfo->RefCount (true);
   }

//______________________________________________________________________________
   Layout::~Layout() 
   {
      if (mInfo) mInfo->RefCount (false);
      if (mInfo && !mInfo->IsRegistered()) {
         delete mInfo;
      }
   }

//______________________________________________________________________________
   Layout& Layout::operator= (const Layout& layout)
   {
      if (mInfo != layout.mInfo) {
         if (mInfo) mInfo->RefCount (false);
         if (mInfo && !mInfo->IsRegistered()) {
            delete mInfo;
         }
         if (layout.IsRegistered()) {
            mInfo = layout.mInfo;
         }
         else {
            mInfo = layout.mInfo ? layout.mInfo->Copy() : 0;
         }
         if (mInfo) mInfo->RefCount (true);
      }
      return *this;
   }

//______________________________________________________________________________
   bool Layout::operator== (const Layout& layout) const 
   {
      // case 1: mInfo is zero
      if (!mInfo) {
         return !layout.mInfo;
      }
      // case 2: layout.mInfo is zero (but Info is not)
      if (!layout.mInfo) {
         return false; 
      }
      // case 3: mInfo is registerd
      if (IsRegistered()) {
         return mInfo == layout.mInfo;
      }
      // case 4: layout.mInfo is registerd (but mInfo is not)
      if (layout.IsRegistered()) {
         return false;
      }
      // case 5: both layouts exist and are not registered
      return *mInfo == *layout.mInfo;
   }

//______________________________________________________________________________
   bool Layout::IsRegistered() const
   {
      return mInfo && mInfo->IsRegistered();
   }

//______________________________________________________________________________
   int Layout::GetRefCount() const
   {
      return mInfo ? mInfo->GetRefCount() : 0;
   }

//______________________________________________________________________________
   int Layout::GetColumNum (const_data_ptr data) const
   {
      return data ? ((Common_t*)data)->mColumns : 0;
   }

//______________________________________________________________________________
   Name Layout::GetName (const_data_ptr data) const
   {
      return Name (data ? ((Common_t*)data)->mName : 0);
   }

//______________________________________________________________________________
   void Layout::SetName (data_ptr data, const Name& name)
   {
      if (data) {
         ((Common_t*)data)->mName = name.GetId();
      }
   }

//______________________________________________________________________________
   Time Layout::GetTime (const_data_ptr data) const
   {
      return data ? ((Common_t*)data)->mTime : Time(0);
   }

//______________________________________________________________________________
   void Layout::SetTime (data_ptr data, const Time& time)
   {
      if (data) {
         ((Common_t*)data)->mTime = time;
      }
   }

//______________________________________________________________________________
   ColumnType::ColumnIfo_t Layout::GetIfo (const_data_ptr data) const
   {
      return data ? ((Common_t*)data)->mIfo : 0;
   }

//______________________________________________________________________________
   void Layout::SetIfo (data_ptr data, const ColumnType::ColumnIfo_t& ifo)
   {
      if (data) {
         ((Common_t*)data)->mIfo = ifo;
      }
   }

//______________________________________________________________________________
   string Layout::GetIfoStr (const_data_ptr data) const
   {
      string s;
      if (data) {
         IfoSet::GetTags (((Common_t*)data)->mIfo, s);
      }
      return s;
   }

//______________________________________________________________________________
   void Layout::SetIfoStr (data_ptr data, const char* ifo)
   {
      IfoSet ifoset (ifo);
      if (data) {
         ((Common_t*)data)->mIfo = ifoset.GetIfoSet();
      }
   }

//______________________________________________________________________________
   bool Layout::GetValue (const char* name, const_data_ptr data, 
                     Value& val) const
   {
      if (!mInfo || !data || !name) {
         val = Value();
         return false;
      }
      return mInfo->GetValue (name, data, val);
   }

//______________________________________________________________________________
   bool Layout::SetValue (const char* name, data_ptr& data, 
                     const Value& val)
   {
      if (!mInfo || !data || !name) {
         return false;
      }
      return mInfo->SetValue (name, data, val);
   }

//______________________________________________________________________________
   bool Layout::SetType (const Type& type)
   {
      const LayoutInfo* info = LayoutInfo::Lookup (type);
      if (info) {
         *this = Layout (info);
      }
      else {
         LayoutInfo newinfo (type);
         *this = Layout (&newinfo);
      }
      return (mInfo != 0);
   }

//______________________________________________________________________________
   bool Layout::GetType (Type& type) const
   {
      if (!mInfo) {
         type = Type();
         return false;
      }
      type = mInfo->GetType();
      return type.GetId() != 0;
   }

//______________________________________________________________________________
   bool Layout::IsCompatible (const Type& type) const
   {
      return (mInfo && (mInfo->GetType() == type));
   }

//______________________________________________________________________________
   const ColumnInfoList& Layout::GetColumnList() const
   {
      return mInfo->GetColumnList();
   }

//______________________________________________________________________________
   bool Layout::AddColumn (const ColumnInfo& col)
   {
      return mInfo ? mInfo->AddColumn (col) : false;
   }

//______________________________________________________________________________
   bool Layout::RemoveColumn (const char* name)
   {
      return mInfo ? mInfo->RemoveColumn (name) : false;
   }

//______________________________________________________________________________
   const ColumnInfo* Layout::GetColumn (const char* name) const
   {
      return mInfo ? mInfo->GetColumn (name) : 0;
   }

//______________________________________________________________________________
   int Layout::DataSize () const 
   {
      return mInfo ? mInfo->DataSize() : 0;
   }

//______________________________________________________________________________
   bool Layout::Construct (data_ptr data, const_data_ptr init)
   {
      return mInfo ? mInfo->Construct (data, init) : false;
   }

//______________________________________________________________________________
   bool Layout::Destruct (data_ptr data)
   {
      return mInfo ? mInfo->Destruct (data) : false;
   }

//______________________________________________________________________________
   bool Layout::Update (data_ptr& data)
   {
      return mInfo ? mInfo->Update (data) : false;
   }

//______________________________________________________________________________
   bool Layout::Compare (const_data_ptr d1, const_data_ptr d2) const
   {
      return mInfo ? mInfo->Compare (d1, d2) : false;
   }

//______________________________________________________________________________
   void Layout::Swap (Layout& l)
   {
      std::swap (mInfo, l.mInfo);
   }

//______________________________________________________________________________
   bool Layout::Register()
   {
      if (!mInfo) {
         return false;
      }
      const LayoutInfo* info = mInfo->Register();
      if (!info) {
         return false;
      }
      *this = Layout (info);
      return true;
   }

//______________________________________________________________________________
   void Layout::Dump (std::ostream& os) const        
   {
      mInfo->Dump(os);
   }

//______________________________________________________________________________
   void Layout::Dump () const        
   {
      mInfo->Dump (cout);
   }

//______________________________________________________________________________
   void Layout::DumpAll (std::ostream& os)
   {
      Factory::Get().DumpAllLayouts (os);
   } 

//______________________________________________________________________________
   void Layout::DumpAll()
   {
      Factory::Get().DumpAllLayouts (cout);
   } 

//______________________________________________________________________________
   const Layout& Layout::GetSimple()
   {
      if (gSimple.IsRegistered()) {
         gSimple = Layout (LayoutInfo::GetSimple());
      }
      return gSimple;
   }

//______________________________________________________________________________
   const Layout& Layout::GetStandard()
   {
      if (gStandard.IsRegistered()) {
         gStandard = Layout (LayoutInfo::GetStandard());
      }
      return gStandard;
   }

//______________________________________________________________________________
   Layout Layout::GetCoincidence (int order)
   {
      Layout coin (LayoutInfo::Coincidence(order));
      if (!coin.IsRegistered()) {
         coin.AddColumn ("Order", ColumnType::kInt);
         for (int j = 0; j < order; ++j) {
            char colname[256];
            sprintf (colname, "Event(%i)", j);
            coin.AddColumn (colname, ColumnType::kEvent);
         }
         coin.Register();
      }
      return coin;
   }

//______________________________________________________________________________
   Layout Layout::GetCluster (int eventnum)
   {
      Layout clust (LayoutInfo::Cluster (eventnum));
      if (!clust.IsRegistered()) {
         clust.AddColumn ("Order", ColumnType::kInt);
         for (int j = 0; j < eventnum; ++j) {
            char colname[256];
            sprintf (colname, "Event(%i)", j);
            clust.AddColumn (colname, ColumnType::kEvent);
         }
         clust.Register();
      }
      return clust;
   }

}
