#ifndef _LIGO_EVENTFILTER_H
#define _LIGO_EVENTFILTER_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Filter							*/
/*                                                         		*/
/* Module Description: Math operations with event columns		*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Filter.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <string>
#include "events/Condition.hh"
#include "events/Type.hh"
#include "events/Name.hh"
#include "events/Column.hh"


namespace events {


/** An event filter uses the type and the name information to select 
    events. It supports wildcards and an event index.

    Example:
    \begin{verbatim}
    Filter ("sngl_Burst::*")
    Filter ("*::glitchMon")
    Filter ("GDS_Trigger::glitchMon")
    \end{verbatim}
    The first example selects all burst events. The second example
    selects all events with name "glitchMon", whereas the third
    example selects events of type "GDS_Trigger" and name "glitchMon".

    If a filter is specified without an index it gets applied to all 
    event indices. Wildcards in form of "*" (asterix) and "?" 
    (question mark) can be used--even so they may translate into 
    inefficient string comparisons. Using just a "*" for one of
    the arguments is still efficient and no comparison is done at all.

    An event index can be explicitly specified as part of the string
    or as a separate argument.
    Examples:
    \begin{verbatim}
    Filter ("sngl_Burst::*[0]") && Filter ("GDS_Trigger::*", 1)
    \end{verbatim}
    selects burst events from the first event stream and DMT triggers
    from the second one.

    As with the column class a filter set can also be specified for
    events stored as event columns. 
    The same notation applies. For example:
    \begin{verbatim}
    Filter ("Event(1).sngl_Burst::*")
    Filter ("sngl_Burst::*(1)")
    \end{verbatim}
    selects events which contain a burst event at column "Event(1)".

    @memo Defines an event filter
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Filter : public Condition {
   public:
      /// matching criterium
      enum matchtype {
      /// Exact
      kExact = 0,
      /// All
      kAll,
      /// Wildcard
      kWildcard
      };
      /** Creates an event filter which rejects everything.
          @memo Constructs an empty event filter
       ******************************************************************/
      explicit Filter() : mIndex (-1) {
         SetPattern (0); }
      /** Creates a new event filter based on the specified filter string.
          @memo Constructors an event filter
          @param pattern Filter pattern
       ******************************************************************/
      explicit Filter (const char* pattern) : mIndex (-1) {
         SetPattern (pattern); }
      /** Creates a new event filter based on the specified filter string.
          @memo Constructors an event filter
          @param pattern Filter pattern
       ******************************************************************/
      explicit Filter (const char* pattern, int index) : mIndex (-1) {
         SetPattern (pattern); SetIndex (index); }
      /** Creates a new event filter based on the specified filter string.
          @memo Constructors an event filter
          @param pattern Filter pattern
       ******************************************************************/
      explicit Filter (const std::string& pattern, int index)
      : mIndex (-1) {
         SetPattern (pattern); SetIndex (index); }
      /** Returns a copy of the event condition. This method must be 
          overriden by all descendents.
          @memo Copy the event
          @return event copy
       ******************************************************************/
      virtual Filter* Copy() const {
         return new Filter (*this); }
      /** Checks if this is a valid filter.
          @memo Is valid?
          @return true if valid
       ******************************************************************/
      bool IsValid() const;
      /** Equality operator (case insensitive)
          @memo Equality operator
       ******************************************************************/
      bool operator== (const Filter& filter) const;
      /** Inequality operator.
          @memo Inequality operator
       ******************************************************************/
      bool operator!= (const Filter& filter) const {
         return !(*this == filter); }
   
      /** Evaluate the condition. This function must be overriden
          by descendents.
          @memo Evaluate
          @param arg Event argument list
          @param val Condition value (return)
          @return True if condition could be evaluated
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, bool& val) const;
   
      /** Set the event index.
          @memo Set event index
          @param index Event index
       ******************************************************************/
      void SetIndex (int index = -1) {
         mIndex = index; }
      /** Returns the event index.
          @memo Get event index
          @return Event index
       ******************************************************************/
      int GetIndex() const {
         return mIndex; }
      /** Returns the filter pattern.
          @memo Get event pattern
          @return Event filter name
       ******************************************************************/
      const char* GetPattern() const {
         return mPattern.c_str(); }
      /** Sets the filter pattern.
          @memo Set event pattern
          @param pattern Filter pattern
          @return true if valid name is supplied
       ******************************************************************/
      bool SetPattern (const char* pattern);
      /** Sets the filter pattern.
          @memo Set event pattern
          @param pattern Filter pattern
          @return true if valid name is supplied
       ******************************************************************/
      bool SetPattern (const std::string& pattern) {
         return SetPattern (pattern.c_str()); }
   
   protected:
      /** Match an event against the filter pattern.
          @memo Match event against pattern
          @param event Event to match
          @return true if filter matches event
       ******************************************************************/
      bool Match (const Event* event) const;
   
   private:
      /// Event index (-1 for all)
      int		mIndex;
      /// Filter name
      std::string	mPattern;
      /// Event access (used to resolve array indices)
      Column		mColumn;
      /// Type pattern
      std::string	mTypePat;
      /// Subtype pattern
      std::string	mNamePat;
      /// Type matching
      matchtype		mTypeMatch;
      /// Type matching
      matchtype		mNameMatch;
      /// Type to match (for exact matches)
      Type		mType;
      /// Name to match (for exact matches)
      Name		mName;
   };


}

#endif // _LIGO_EVENTFILTER_H
