//////////////////////////////////////////////////////////////////////////
//  									//
//  Factory								//
//  									//
//////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <fnmatch.h>
#include "events/Factory.hh"

   using namespace std;

namespace events {

//______________________________________________________________________________
   Factory* Factory::gFactory = 0;

//______________________________________________________________________________
   Factory& Factory::Get()
   {
      if (!gFactory) {
         gFactory = new (std::nothrow) Factory();
         gFactory->Init();
      }
      return *gFactory;
   }

//______________________________________________________________________________
   Factory::Factory() 
   : mLayoutInit (false), mLayoutAddColVers (0), mFixedColNum (0), 
   mNextColumnOfs (0), mInitFixedColumns (true)
   {
   }

//______________________________________________________________________________
   void Factory::Init() 
   {
      // Fill in default fixed columns
      AddFixedColumn (ColumnInfo 
                     (ColumnType::kColumnNumberName, ColumnType::kInt));
      AddFixedColumn (ColumnInfo 
                     (ColumnType::kColumnNameName, ColumnType::kInt));
      AddFixedColumn (ColumnInfo 
                     (ColumnType::kColumnTimeName, ColumnType::kTime));
      AddFixedColumn (ColumnInfo 
                     (ColumnType::kColumnIfoName, ColumnType::kInt));   
      mInitFixedColumns = false;
   
      // Add some interferometer tags
      IfoRegister ("H0"); // Hanford PEM
      IfoRegister ("H1"); // Hanford 4K
      IfoRegister ("H2"); // Hanford 2K
      IfoRegister ("L0"); // Livingston PEM
      IfoRegister ("L1"); // Livingston 2K
      IfoRegister ("G1"); // GEO
   }

//______________________________________________________________________________
   bool Factory::RegisterType (const char* name, TypeInfo*& type)
   {
      if (!mLayoutInit) {
         InitBasicLayouts();
      }
      if (!CheckTypeName (name)) {
         type = 0;
         return false;
      }
      type = GetType (name);
      // check if not registered
      if (!type) {
         // check if first; create dummy if yes
         if (mTypes.Empty()) {
            mTypes.Add (TypePtr (new TypeInfo (0, "__Invalid__")));
         }
         // create a new type
         int tid = mTypes.Size();
         mTypes.Add (TypePtr (new TypeInfo (tid, name)));
         if (tid >= mTypes.Size()) {
            return false;
         }
         type = mTypes[tid].get();
      }
      return true;
   }

//______________________________________________________________________________
   TypeInfo* Factory::GetType (const char* name)
   {
      if (!mLayoutInit) {
         InitBasicLayouts();
      }
      TypePtr* t = mTypes.Get (name);
      return t ? t->get() : 0;
   }

//______________________________________________________________________________
   bool Factory::MatchType (const Type& type, const char* pat)
   {
      if (!pat || (strlen (pat) == 0)) {
         return type.GetId() == 0;
      }
      // if no wildcards, compare strings
      if (strpbrk (pat, "[]*?") == 0) {
         return type.GetName() && 
            (strcasecmp (type.GetName(), pat) == 0);
      }
      // else use fnmatch
      else {
         return type.GetName() &&
            (fnmatch (pat, type.GetName(), 0) == 0);
      }
   }

//______________________________________________________________________________
   bool Factory::CheckTypeName (const char* name)
   {
      // must be at least one character
      if (!name || !*name) {
         return false;
      }
      // must not contain "[", "]", ".", "*", "?", spaces
      return strpbrk (name, "()[].*? \t\f\n\r\v") == 0;
   }

//______________________________________________________________________________
   bool Factory::RegisterName (const char* name, int& id)
   {
      if (mNames.Empty()) {
         mNames.Add ("__Invalid__"); // index zero is invalid
      }
      if (!CheckNameName (name)) {
         return false;
      }
      if ((id = GetNameId (name))) {
         return true;
      }
      mNames.Add (name);
      return ((id = GetNameId (name)));
   }

//______________________________________________________________________________
   int Factory::GetNameId (const char* name)
   {
      int i = mNames.GetPos (name) - mNames.begin();
      return (i < mNames.Size()) ? i : 0;
   }

//______________________________________________________________________________
   const char* Factory::GetNameStr (int id)
   {
      if ((id > 0) && (id < mNames.Size())) {
         return (mNames.begin() + id)->GetName();
      }
      else {
         return 0;
      }
   }

//______________________________________________________________________________
   bool Factory::MatchName (const Name& name, const char* pat)
   {
      if (!pat || (strlen (pat) == 0)) {
         return name.GetId() == 0;
      }
      // if no wildcards, compare strings
      if (strpbrk (pat, "[]*?") == 0) {
         return name.GetName() && 
            (strcasecmp (name.GetName(), pat) == 0);
      }
      // else use fnmatch
      else {
         return name.GetName() &&
            (fnmatch (pat, name.GetName(), 0) == 0);
      }
   }

//______________________________________________________________________________
   bool Factory::CheckNameName (const char* name)
   {
      return CheckTypeName (name);
   }

//______________________________________________________________________________
   void Factory::InitBasicLayouts()
   {
      if (mLayoutInit) {
         return;
      }
      mLayoutInit = true;
      // Simple
      Layout lsimple (LayoutInfo::Simple());
      lsimple.Register();
   
      // Standard
      Layout lstd (LayoutInfo::Standard());
      lstd.Register();
   
      // Coincidence
      for (int i = 2; i < 5; ++i) {
         Layout lcoin (LayoutInfo::Coincidence (i));
         lcoin.AddColumn ("Order", ColumnType::kInt);
         for (int j = 0; j < i; ++j) {
            char colname[256];
            sprintf (colname, "Event(%i)", j);
            lcoin.AddColumn (colname, ColumnType::kEvent);
         }
         lcoin.Register();
      }
   
      // Cluster
      for (int i = 0; i < 10; ++i) {
         Layout lclust (LayoutInfo::Cluster (i));
         lclust.AddColumn ("Order", ColumnType::kInt);
         for (int j = 0; j < i; ++j) {
            char colname[256];
            sprintf (colname, "Event(%i)", j);
            lclust.AddColumn (colname, ColumnType::kEvent);
         }
         lclust.Register();
      }
   
      // GDS trigger
      Layout lgdstrig (LayoutInfo::GdsTrigger());
      // lgdstrig.AddColumn("ROW",ColumnType::kInt);
      lgdstrig.AddColumn("CREATOR_DB",ColumnType::kInt);
      lgdstrig.AddColumn("PROCESS_ID",ColumnType::kString);
      lgdstrig.AddColumn("FILTER_ID",ColumnType::kString);
      lgdstrig.AddColumn("SUBTYPE",ColumnType::kString);
      lgdstrig.AddColumn("DURATION",ColumnType::kReal);
      lgdstrig.AddColumn("PRIORITY",ColumnType::kInt);
      lgdstrig.AddColumn("DISPOSITION",ColumnType::kInt);
      lgdstrig.AddColumn("SIZE",ColumnType::kReal);
      lgdstrig.AddColumn("SIGNIFICANCE",ColumnType::kReal);
      lgdstrig.AddColumn("FREQUENCY",ColumnType::kReal);
      lgdstrig.AddColumn("BANDWIDTH",ColumnType::kReal);
      lgdstrig.AddColumn("BINARYDATA",ColumnType::kString);
      lgdstrig.AddColumn("BINARYDATA_LENGTH",ColumnType::kInt);
      lgdstrig.AddColumn("EVENT_ID",ColumnType::kString);
      lgdstrig.Register();
   
      // Single Inspiral
      Layout lsnglinspiral (LayoutInfo::SnglInspiral());
      lsnglinspiral.AddColumn("CREATOR_DB",ColumnType::kInt);
      lsnglinspiral.AddColumn("PROCESS_ID",ColumnType::kString);
      lsnglinspiral.AddColumn("FILTER_ID",ColumnType::kString);
      lsnglinspiral.AddColumn("IMPULSE_TIME",ColumnType::kTime);
      lsnglinspiral.AddColumn("AMPLITUDE",ColumnType::kReal);
      lsnglinspiral.AddColumn("EFF_DISTANCE",ColumnType::kReal);
      lsnglinspiral.AddColumn("COA_PHASE",ColumnType::kReal);
      lsnglinspiral.AddColumn("MASS1",ColumnType::kReal);
      lsnglinspiral.AddColumn("MASS2",ColumnType::kReal);
      lsnglinspiral.AddColumn("MCHIRP",ColumnType::kReal);
      lsnglinspiral.AddColumn("ETA",ColumnType::kReal);
      lsnglinspiral.AddColumn("TAU0",ColumnType::kReal);
      lsnglinspiral.AddColumn("TAU2",ColumnType::kReal);
      lsnglinspiral.AddColumn("TAU3",ColumnType::kReal);
      lsnglinspiral.AddColumn("TAU4",ColumnType::kReal);
      lsnglinspiral.AddColumn("TAU5",ColumnType::kReal);
      lsnglinspiral.AddColumn("TTOTAL",ColumnType::kReal);
      lsnglinspiral.AddColumn("SNR",ColumnType::kReal);
      lsnglinspiral.AddColumn("CHISQ",ColumnType::kReal);
      lsnglinspiral.AddColumn("CHISQ_DOF",ColumnType::kInt);
      lsnglinspiral.AddColumn("SIGMASQ",ColumnType::kReal);
      lsnglinspiral.AddColumn("EVENT_ID",ColumnType::kString);
      lsnglinspiral.AddColumn("CHANNEL",ColumnType::kString);
      lsnglinspiral.Register();
   
   	// Single Burst
      Layout lsnglburst (LayoutInfo::SnglBurst());
      lsnglburst.AddColumn("CREATOR_DB",ColumnType::kInt);
      lsnglburst.AddColumn("PROCESS_ID",ColumnType::kString);
      lsnglburst.AddColumn("FILTER_ID",ColumnType::kString);
      lsnglburst.AddColumn("DURATION",ColumnType::kReal);
      lsnglburst.AddColumn("CENTRAL_FREQ",ColumnType::kReal);
      lsnglburst.AddColumn("BANDWIDTH",ColumnType::kReal);
      lsnglburst.AddColumn("AMPLITUDE",ColumnType::kReal);
      lsnglburst.AddColumn("SNR",ColumnType::kReal);
      lsnglburst.AddColumn("CONFIDENCE",ColumnType::kReal);
      lsnglburst.AddColumn("EVENT_ID",ColumnType::kString);
      lsnglburst.AddColumn("CHANNEL",ColumnType::kString);
      lsnglburst.Register();
   
   	// Single Ringdown
      Layout lsnglringdown (LayoutInfo::SnglRingdown());
      lsnglringdown.AddColumn("CREATOR_DB",ColumnType::kInt);
      lsnglringdown.AddColumn("PROCESS_ID",ColumnType::kString);
      lsnglringdown.AddColumn("FILTER_ID",ColumnType::kString);
      lsnglringdown.AddColumn("DURATION",ColumnType::kReal);
      lsnglringdown.AddColumn("AMPLITUDE",ColumnType::kReal);
      lsnglringdown.AddColumn("FREQUENCY",ColumnType::kReal);
      lsnglringdown.AddColumn("Q",ColumnType::kReal);
      lsnglringdown.AddColumn("MASS",ColumnType::kReal);
      lsnglringdown.AddColumn("CONFIDENCE",ColumnType::kReal);
      lsnglringdown.AddColumn("EVENT_ID",ColumnType::kString);
      lsnglringdown.AddColumn("CHANNEL",ColumnType::kString);
      lsnglringdown.Register();
   
   	// Single Unmodeled
      Layout lsnglunmodeled (LayoutInfo::SnglUnmodeled());
      lsnglunmodeled.AddColumn("CREATOR_DB",ColumnType::kInt);
      lsnglunmodeled.AddColumn("PROCESS_ID",ColumnType::kString);
      lsnglunmodeled.AddColumn("FILTER_ID",ColumnType::kString);
      lsnglunmodeled.AddColumn("DURATION",ColumnType::kReal);
      lsnglunmodeled.AddColumn("AMPLITUDE",ColumnType::kReal);
      lsnglunmodeled.AddColumn("SNR",ColumnType::kReal);
      lsnglunmodeled.AddColumn("CONFIDENCE",ColumnType::kReal);
      lsnglunmodeled.AddColumn("EVENT_ID",ColumnType::kString);
      lsnglunmodeled.AddColumn("CHANNEL",ColumnType::kString);
      lsnglunmodeled.Register();
   
   	// Single Directed Periodic
      Layout lsngldperiodic (LayoutInfo::SnglDPeriodic());
      lsngldperiodic.AddColumn("CREATOR_DB",ColumnType::kInt);
      lsngldperiodic.AddColumn("PROCESS_ID",ColumnType::kString);
      lsngldperiodic.AddColumn("FILTER_ID",ColumnType::kString);
      lsngldperiodic.AddColumn("END_TIME",ColumnType::kTime);
      lsngldperiodic.AddColumn("DURATION",ColumnType::kReal);
      lsngldperiodic.AddColumn("TARGET_NAME",ColumnType::kString);
      lsngldperiodic.AddColumn("SKY_RA",ColumnType::kReal);
      lsngldperiodic.AddColumn("SKY_DEC",ColumnType::kReal);
      lsngldperiodic.AddColumn("FREQUENCY",ColumnType::kReal);
      lsngldperiodic.AddColumn("AMPLITUDE",ColumnType::kReal);
      lsngldperiodic.AddColumn("PHASE",ColumnType::kReal);
      lsngldperiodic.AddColumn("SNR",ColumnType::kReal);
      lsngldperiodic.AddColumn("CONFIDENCE",ColumnType::kReal);
      lsngldperiodic.AddColumn("STAT_NAME",ColumnType::kString);
      lsngldperiodic.AddColumn("STAT_VALUE",ColumnType::kReal);
      lsngldperiodic.AddColumn("STAT_PROB",ColumnType::kReal);
      lsngldperiodic.AddColumn("EVENT_ID",ColumnType::kString);
      lsngldperiodic.AddColumn("CHANNEL",ColumnType::kString);
      lsngldperiodic.Register();
   
   	// Multi Inspiral
      Layout lmultiinspiral (LayoutInfo::MultiInspiral());
      lmultiinspiral.AddColumn("CREATOR_DB",ColumnType::kInt);
      lmultiinspiral.AddColumn("PROCESS_ID",ColumnType::kString);
      lmultiinspiral.AddColumn("FILTER_ID",ColumnType::kString);
      lmultiinspiral.AddColumn("IMPULSE_TIME",ColumnType::kTime);
      lmultiinspiral.AddColumn("AMPLITUDE",ColumnType::kReal);
      lmultiinspiral.AddColumn("EFF_DISTANCE",ColumnType::kReal);
      lmultiinspiral.AddColumn("COA_PHASE",ColumnType::kReal);
      lmultiinspiral.AddColumn("MASS1",ColumnType::kReal);
      lmultiinspiral.AddColumn("MASS2",ColumnType::kReal);
      lmultiinspiral.AddColumn("MCHIRP",ColumnType::kReal);
      lmultiinspiral.AddColumn("ETA",ColumnType::kReal);
      lmultiinspiral.AddColumn("TAU0",ColumnType::kReal);
      lmultiinspiral.AddColumn("TAU2",ColumnType::kReal);
      lmultiinspiral.AddColumn("TAU3",ColumnType::kReal);
      lmultiinspiral.AddColumn("TAU4",ColumnType::kReal);
      lmultiinspiral.AddColumn("TAU5",ColumnType::kReal);
      lmultiinspiral.AddColumn("TTOTAL",ColumnType::kReal);
      lmultiinspiral.AddColumn("SNR",ColumnType::kReal);
      lmultiinspiral.AddColumn("CHISQ",ColumnType::kReal);
      lmultiinspiral.AddColumn("CHISQ_DOF",ColumnType::kInt);
      lmultiinspiral.AddColumn("SIGMASQ",ColumnType::kReal);
      lmultiinspiral.AddColumn("LIGO_AXIS_RA",ColumnType::kReal);
      lmultiinspiral.AddColumn("LIGO_AXIS_DEC",ColumnType::kReal);
      lmultiinspiral.AddColumn("LIGO_ANGLE",ColumnType::kReal);
      lmultiinspiral.AddColumn("LIGO_ANGLE_SIG",ColumnType::kReal);
      lmultiinspiral.AddColumn("EVENT_ID",ColumnType::kString);
      lmultiinspiral.Register();
   
   	// Multi Burst
      Layout lmultiburst (LayoutInfo::MultiBurst());
      lmultiburst.AddColumn("CREATOR_DB",ColumnType::kInt);
      lmultiburst.AddColumn("PROCESS_ID",ColumnType::kString);
      lmultiburst.AddColumn("FILTER_ID",ColumnType::kString);
      lmultiburst.AddColumn("DURATION",ColumnType::kReal);
      lmultiburst.AddColumn("CENTRAL_FREQ",ColumnType::kReal);
      lmultiburst.AddColumn("BANDWIDTH",ColumnType::kReal);
      lmultiburst.AddColumn("AMPLITUDE",ColumnType::kReal);
      lmultiburst.AddColumn("SNR",ColumnType::kReal);
      lmultiburst.AddColumn("CONFIDENCE",ColumnType::kReal);
      lmultiburst.AddColumn("LIGO_AXIS_RA",ColumnType::kReal);
      lmultiburst.AddColumn("LIGO_AXIS_DEC",ColumnType::kReal);
      lmultiburst.AddColumn("LIGO_ANGLE",ColumnType::kReal);
      lmultiburst.AddColumn("LIGO_ANGLE_SIG",ColumnType::kReal);
      lmultiburst.AddColumn("EVENT_ID",ColumnType::kString);
      lmultiburst.Register();
   
   	// Segment
      Layout lsegment (LayoutInfo::Segment());
      lsegment.AddColumn("CREATOR_DB",ColumnType::kInt);
      lsegment.AddColumn("PROCESS_ID",ColumnType::kString);
      lsegment.AddColumn("SEGMENT_GROUP",ColumnType::kString);
      lsegment.AddColumn("VERSION",ColumnType::kInt);
      lsegment.AddColumn("END_TIME",ColumnType::kTime);
      lsegment.Register();
   
      mInitFixedColumns = false;
   }

//______________________________________________________________________________
   const LayoutInfo* Factory::RegisterLayout (const LayoutInfo& layout)
   {
      // if (!mLayoutInit) {
         // InitBasicLayouts();
      // }
      // Cannot register twice
      if (layout.IsRegistered()) {
         return 0;
      }
      // Need to have a valid type with layout
      const TypeInfo* tinfo = layout.GetType().GetTypeInfo();
      if (!tinfo || tinfo->GetLayout()){
         return 0;
      }
      // Now add layout to list
      mLayouts.push_back (LayoutPtr (new LayoutInfo (layout)));
      if (mLayouts.empty()) {
         return 0;
      }
      // Set registration flag
      LayoutInfo* reglayout = mLayouts.back().get();
      reglayout->SetRegistered();
      // Set layout pointer in type information
      tinfo->SetLayout (reglayout);
      mInitFixedColumns = false;
      return reglayout;
   }

//______________________________________________________________________________
   const LayoutInfo* Factory::LookupLayout (const Type& type)
   {
      if (!mLayoutInit) {
         InitBasicLayouts();
      }
      return type.GetLayout();
   }

//______________________________________________________________________________
   bool Factory::AddFixedColumn (const ColumnInfo& desc)
   {
      // still able to add fixed columns?
      if (!mInitFixedColumns) {
         return false;
      }
      // need name and type
      if ((desc.GetType() == ColumnType::kInvalid) ||
         (*desc.GetName() == 0)) {
         return false;
      }
      // Can only add once
      if (mFixedColumns.Get (desc.GetName()) != 0) {
         return false;
      }
      mFixedColumns.Add (desc);
      if (mFixedColumns.Empty()) {
         return false;
      }
      ColumnList::iterator col = mFixedColumns.end() - 1;
      col->SetFixed();
      // make sure alignment is correct
      switch (mFixedColNum) {
         case 0: 
            mNextColumnOfs = offsetof (ColumnType::Common_t, mColumns);
            break;
         case 1: 
            mNextColumnOfs = offsetof (ColumnType::Common_t, mName);
            break;
         case 2: 
            mNextColumnOfs = offsetof (ColumnType::Common_t, mTime);
            break;
         case 3: 
            mNextColumnOfs = offsetof (ColumnType::Common_t, mIfo);
            break;
         default:
            mNextColumnOfs = 
               ((mNextColumnOfs + col->GetTypeAlignment()) - 1) / 
               col->GetTypeAlignment() * col->GetTypeAlignment();
      }
      col->SetOffset (mNextColumnOfs);
      mNextColumnOfs += col->GetTypeSize();
      col->SetColumn (mFixedColNum++);
      return true;
   }

//______________________________________________________________________________
   const ColumnInfo* Factory::GetFixedColumn (const char* name)
   {
      return mFixedColumns.Get (name);
   }

//______________________________________________________________________________
   const ColumnInfoList& Factory::FixedColumns()
   {
      return mFixedColumns.List();
   }

//______________________________________________________________________________
   const Factory::ColumnList& Factory::FixedColumnIndex()
   {
      return mFixedColumns;
   }



//______________________________________________________________________________
   bool Factory::IfoString2Tag (const char* ifostring, IfoTag& tag)
   {
      if (!ifostring || !*ifostring) {
         return false;
      }
      // Get character
      char c = ::toupper (ifostring[0]);
      if (!isalpha (c)) {
         return false;
      }
      // get number
      int n = atoi (++ifostring);
      // check if all digits
      for (; *ifostring; ++ifostring) {
         if (!isdigit (*ifostring)) {
            return false;
         }
      }
      // make tag
      tag = std::make_pair (c, n);
      return true;
   }

//______________________________________________________________________________
   bool Factory::IfoString2Tags (const char* ifostring, IfoTagList& list)
   {
      // clear list
      list.clear();
      // valid string?
      if (!ifostring) {
         return true;
      }
      // parse string
      IfoTag tag;
      while (*ifostring) {
         // find end of ifo
         const char* p = ifostring + 1;
         while (*p && !isalpha (*p) && !isspace(*p)) ++p;
         // get tag
         std::string s (ifostring, p - ifostring);
         if (!IfoString2Tag (s.c_str(), tag)) {
            return false;
         }
         // add tag to list
         list.push_back (tag);
         while (*p && !isalpha (*p)) ++p;
         ifostring = p;
      }
      return true;
   }

//______________________________________________________________________________
   bool Factory::IfoTag2String (IfoTag& tag, std::string& ifostring)
   {
      if (!isalpha (tag.first) || (tag.second < 0)) {
         return false;
      }
      char buf[64];
      sprintf (buf, "%c%i", tag.first, tag.second);
      ifostring = buf;
      return true;
   }

//______________________________________________________________________________
   int Factory::IfoRegister (const char* ifostring)
   {
      // bit for new ifo tag available? valid tag?
      IfoTag tag;
      if (((int)mIfoList.size() >= IfoSet::Size()) ||
         !IfoString2Tag (ifostring, tag)) {
         return -1;
      }
      // check if already in list
      for (IfoTagList::iterator i = mIfoList.begin(); 
          i != mIfoList.end(); ++i) {
         if (tag == *i) {
            return -1;
         }
      }
      // add new tag
      mIfoList.push_back (tag);
      return mIfoList.size();
   }

//______________________________________________________________________________
   int Factory::IfoGetBit (const char* ifostring)
   {
      // valid tag?
      IfoTag tag;
      if (!IfoString2Tag (ifostring, tag)) {
         return -1;
      }
      // check if in list
      for (IfoTagList::iterator i = mIfoList.begin(); 
          i != mIfoList.end(); ++i) {
         if (tag == *i) {
            return i - mIfoList.begin();
         }
      }
      // not found
      return -1;
   }

//______________________________________________________________________________
   bool Factory::IfoGetBits (const char* ifostring, 
                     IfoSet::ifoset_type& ifoset)
   {
      // valid tags?
      IfoTagList list;
      if (!IfoString2Tags (ifostring, list)) {
         return false;
      }
      // go thru list
      ifoset = 0;
      for (IfoTagList::iterator i = list.begin(); 
          i != list.end(); ++i) {
         // check if in registered list
         bool found = false;
         for (IfoTagList::iterator j = mIfoList.begin(); 
             j != mIfoList.end(); ++j) {
            if (*i == *j) {
               found = true;
               ifoset |= 1 << (j - mIfoList.begin());
               break;
            }
         }
         if (!found) {
            return false;
         }
      }
      return true;
   }

//______________________________________________________________________________
   std::string Factory::IfoGetTag (int bit)
   {
      // valid bit?
      if ((bit < 0) || (bit >= (int)mIfoList.size())) {
         return "";
      }
      // get the interferometer name
      std::string s;
      if (!IfoTag2String (mIfoList[bit], s)) {
         return "";
      }
      else {
         return s;
      }
   }

//______________________________________________________________________________
   bool Factory::IfoGetTags (const IfoSet::ifoset_type& ifoset, 
                     std::string& ifostring)
   {
      // loop over all bits
      std::string s;
      int bit = 0;
      for (IfoSet::ifoset_type i = ifoset; i; i >>= 1, ++bit) {
         if (i & 1) {
            // get the interferometer name
            std::string ifo = IfoGetTag (bit);
            if (ifo.empty()) {
               return false;
            }
            s += ifo;
         }
      }
      ifostring = s;
      return true;
   }

//______________________________________________________________________________
   void Factory::DumpAllTypes (std::ostream& os) const 
   {
      for (TypeList::const_iterator i = mTypes.begin(); i != mTypes.end(); ++i) {
         os << i->GetName() << endl;
      }
   }

//______________________________________________________________________________
   void Factory::DumpAllNames (std::ostream& os) const 
   {
      for (NameList::const_index_iterator i = mNames.Index().begin(); 
          i != mNames.Index().end(); ++i) {
         os << i->first << endl;
      }
   }

//______________________________________________________________________________
   void Factory::DumpAllLayouts (std::ostream& os) const
   {
      if (!mLayoutInit) {
         ((Factory*)this)->InitBasicLayouts();
      }
      for (LayoutList::const_iterator i = mLayouts.begin(); i != mLayouts.end(); ++i) {
         i->get()->Dump(os);
      }
   }

}
