#ifndef _LIGO_EVENT_H
#define _LIGO_EVENT_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Event							*/
/*                                                         		*/
/* Module Description: Defines an event					*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Event.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "events/ColumnType.hh"
#include "events/Layout.hh"
#include "events/Value.hh"
#include <iosfwd>


namespace events {


/** An event is characterized by a type, a time stamp and a set of 
    parameters. This is the basic container to store a simple event.
   
    @memo Defines an event
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Event : public ColumnType {   
   public:
      /** Creates an empty event.
          @memo Default constructor
       ******************************************************************/
      Event() : mData (0) {
      }
      /** Creates an event which consists of minimum number of rows:
          type = "simple", type subid = 0, ifoset = "" and the
          specified time.
          @memo Constructor
       ******************************************************************/
      explicit Event (const Time& time) : mData (0) {
         Init (Layout::GetSimple()); SetTime (time); }
      /** Creates an event using a specified layout. The layout must be
          registered!
          @memo Constructor
       ******************************************************************/
      explicit Event (const Layout& layout) : mData (0) {
         Init (layout); }
      /** Creates an event using a specified type. The type must
          correspond to a registerd layout.
          @memo Constructor
       ******************************************************************/
      explicit Event (const Type& type) : mData (0) {
         Init (Layout (type)); }
      /** Creates an event using a specified type name. The type must
          exist and correspond to a registerd layout.
          @memo Constructor
       ******************************************************************/
      explicit Event (const char* name) : mData (0) {
         Init (Layout (name)); }
      /** Creates an event using a specified type name. The type must
          exist and correspond to a registerd layout.
          @memo Constructor
       ******************************************************************/
      explicit Event (const std::string& name) : mData (0) {
         Init (Layout (name)); }
      /** Copy constructor.
          @memo Copy constructor
       ******************************************************************/
      Event (const Event& event) : mData (0) {
         Init (event.mLayout, event.mData); }
      /** Destroys an event.
          @memo Destructor
       ******************************************************************/
      ~Event() {
         Destroy(); }
      /** Assignment operator.
          @memo Assignment operator
       ******************************************************************/
      Event& operator= (const Event& event) {
         if (this != &event) {
            Destroy(); Init (event.mLayout, event.mData); }
         return *this; }
   
      /** Is a valid event? A valid event must have a registered layout.
          @memo Valid event?
       ******************************************************************/
      bool IsValid() const {
         return mLayout.IsRegistered() && mData; }
   
      /** Returns a copy of the event. This method must be overriden 
          by all descendents.
          @memo Copy the event
          @return event copy
       ******************************************************************/
      Event* Copy() const {
         return new Event (*this); }
   
      /** Equality operator. Compares the layout and the full data.
          @memo Equality operator
       ******************************************************************/
      bool operator== (const Event& event) const;
      /** Inequality operator. Compares the layout and the full data.
          @memo Inequality operator
       ******************************************************************/
      bool operator!= (const Event& event) const {
         return !(*this == event); }
      /** Default sort order (by time).
          @memo Default sort order
       ******************************************************************/
      bool operator< (const Event& event) const {
         return GetTime() < event.GetTime(); }
      /** Inverse sort order (by time).
          @memo Inverse sort order
       ******************************************************************/
      bool operator> (const Event& event) const {
         return GetTime() > event.GetTime(); }
   
      /** Get the event layout.
          @memo Get the event layout
       ******************************************************************/
      const Layout& GetLayout() const {
         return mLayout; }
      /** Updates the event, if columns have been added. Fills additonal
          columns with default values.
          @memo Update the event
       ******************************************************************/
      bool Update() {
         return mLayout.Update (mData); }
      /**  Swap a event with the current
           @memo Swap an event
       ******************************************************************/
      void Swap (Event& e);
   
      /**  Returns the name of the event
           @memo Gets the event time
       ******************************************************************/
      Name GetName() const {
         return mLayout.GetName (mData); }
      /**  Sets the time of the event
           @memo Sets the event time
       ******************************************************************/
      void SetName (const Name& name) {
         mLayout.SetName (mData, name); }
      /**  Returns the time of the event
           @memo Gets the event time
       ******************************************************************/
      Time GetTime() const {
         return mLayout.GetTime (mData); }
      /**  Sets the time of the event
           @memo Sets the event time
       ******************************************************************/
      void SetTime (const time_type& time) {
         mLayout.SetTime (mData, time); }
      /**  Returns the ifo set of the event
           @memo Gets the ifo set
       ******************************************************************/
      ColumnType::ColumnIfo_t GetIfo() const {
         return mLayout.GetIfo (mData); }
      /**  Sets the ifo set of the event
           @memo Sets the event ifo set
       ******************************************************************/
      void SetIfo (const ColumnType::ColumnIfo_t& ifo) {
         mLayout.SetIfo (mData, ifo); }
      /**  Returns the ifo set string of the event
           @memo Gets the ifo set
       ******************************************************************/
      std::string GetIfoStr() const {
         return mLayout.GetIfoStr (mData); }
      /**  Sets the ifo set of the event
           @memo Sets the event ifo set
       ******************************************************************/
      void SetIfoStr (const char* ifo) {
         mLayout.SetIfoStr (mData, ifo); }
      /**  Returns the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Gets the column value of an event (Value)
           @param name Name of column
           @param val Column value (return)
           @return true if column exists and value could be read
       ******************************************************************/
      bool GetValue (const char* name, Value& val) const {
         return mLayout.GetValue (name, mData, val); }
      /**  Returns the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Gets the column value of an event (Real)
           @param name Name of column
           @param x Column value (return)
           @return true if column exists and value could be read
       ******************************************************************/
      bool GetValue (const char* name, ColumnType::Real& x) const {
         Value val;
         return GetValue (name, val) && val.Write (x); }
      /**  Returns the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Gets the column value of an event (Int)
           @param name Name of column
           @param i Column value (return)
           @return true if column exists and value could be read
       ******************************************************************/
      bool GetValue (const char* name, ColumnType::Int& i) const {
         Value val;
         return GetValue (name, val) && val.Write (i); }
      /**  Returns the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Gets the column value of an event (Complex)
           @param name Name of column
           @param c Column value (return)
           @return true if column exists and value could be read
       ******************************************************************/
      bool GetValue (const char* name, ColumnType::Complex& c) const {
         Value val;
         return GetValue (name, val) && val.Write (c); }
      /**  Returns the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Gets the column value of an event (Time)
           @param name Name of column
           @param t Column value (return)
           @return true if column exists and value could be read
       ******************************************************************/
      bool GetValue (const char* name, ColumnType::time_type& t) const {
         Value val;
         return GetValue (name, val) && val.Write (t); }
      /**  Returns the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Gets the column value of an event (string)
           @param name Name of column
           @param s Column value (return)
           @return true if column exists and value could be read
       ******************************************************************/
      bool GetValue (const char* name, std::string& s) const {
         Value val;
         return GetValue (name, val) && val.Write (s); }
      /**  Returns the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Gets the column value of an event (string)
           @param name Name of column
           @param p Column value (return)
           @param len Maximum length of buffer
           @return true if column exists and value could be read
       ******************************************************************/
      bool GetValue (const char* name, char* p, int len) const {
         Value val;
         return GetValue (name, val) && val.Write (p, len); }
      /**  Sets the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Sets the column value of an event (Value)
           @param name Name of column
           @param val New column value
           @return true if column exists and value could be read
       ******************************************************************/
      bool SetValue (const char* name, const Value& val) {
         return mLayout.SetValue (name, mData, val); }
      /**  Sets the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Sets the column value of an event (Real)
           @param name Name of column
           @param x New column value
           @return true if column exists and value could be read
       ******************************************************************/
      bool SetValue (const char* name, const ColumnType::Real& x) {
         return SetValue (name, Value (x)); }
      /**  Sets the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Sets the column value of an event (Int)
           @param name Name of column
           @param i New column value
           @return true if column exists and value could be read
       ******************************************************************/
      bool SetValue (const char* name, const ColumnType::Int& i) {
         return SetValue (name, Value (i)); }
      /**  Sets the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Sets the column value of an event (Complex)
           @param name Name of column
           @param c New column value
           @return true if column exists and value could be read
       ******************************************************************/
      bool SetValue (const char* name, const ColumnType::Complex& c) {
         return SetValue (name, Value (c)); }
      /**  Sets the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Sets the column value of an event (Time)
           @param name Name of column
           @param t New column value
           @return true if column exists and value could be read
       ******************************************************************/
      bool SetValue (const char* name, const ColumnType::time_type& t) {
         return SetValue (name, Value (t)); }
      /**  Sets the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Sets the column value of an event (string)
           @param name Name of column
           @param s New column value
           @return true if column exists and value could be read
       ******************************************************************/
      bool SetValue (const char* name, const std::string& s) {
         return SetValue (name, Value (s)); }
      /**  Sets the data value of the specified column. In general,
           use the column class to access an event column value.
           @memo Sets the column value of an event (string)
           @param name Name of column
           @param p New column value
           @param len length of buffer, or -1 for NULL terminated
           @return true if column exists and value could be read
       ******************************************************************/
      bool SetValue (const char* name, const char* p, int len = -1) {
         return SetValue (name, Value (p, len)); }
      /**  Get the data pointer
           @memo Get the data pointer
       ******************************************************************/
      data_ptr GetData() {
         return mData; }
      /**  Get the data pointer
           @memo Get the data pointer
       ******************************************************************/
      const data_ptr GetData() const {
         return mData; }
   
      /**  Dump Event data to the specified output stream
           @memo Dump Event data to the specified output stream
           @param os output stream
           @param indent indent level (number of tabs)
       ******************************************************************/
      void Dump (std::ostream& os, int indent = 0) const;
      /**  Dump Event data to the standard output stream
           @memo Dump Event data to the specified output stream
           @param indent indent level (number of tabs)
       ******************************************************************/
      void Dump (int indent = 0) const;
   
      /**  Dump column data to the specified output stream
           @memo Dump column data to the specified output stream
           @param name column name
           @param os output stream
       ******************************************************************/
      void DumpColumn (const char* name, std::ostream& os) const;
      /**  Dump column data to the standard output stream
           @memo Dump column data to the specified output stream
           @param name column name
       ******************************************************************/
      void DumpColumn(const char* name) const;
   
      /**  Gets a pointer to a default event
           @memo Get default event
       ******************************************************************/
      static const Event& Default();
   
   protected:
      /** Init method.
          @memo Initialize event
       ******************************************************************/
      bool Init (const Layout& layout, const_data_ptr init = 0);
      /** Destroy method.
          @memo Destroy event
       ******************************************************************/
      void Destroy ();
   
   private:
      /// Event Layout
      Layout		mLayout;
      /// Column data. The first word is the column number
      data_ptr		mData;
      /// Default event
      static Event*	gDefault;
   };

}

#endif // _LIGO_EVENT_H
