#ifndef _LIGO_EVENTCONDITIONPTR_H
#define _LIGO_EVENTCONDITIONPTR_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: ConditionPtr						*/
/*                                                         		*/
/* Module Description: Defines a smart pointer for an event list	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: ConditionPtr.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "events/Condition.hh"


namespace events {


/** A smart pointer for an event condition. When copying an event cond. 
    smart pointer, a new event condition is created which is an exact 
    copy     of the original.
   
    @memo Smart pointer for event conditions.
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class ConditionPtr {
   public:
      /** Creates a pointer to an event condition. The event conditions
          gets adopted.
          @memo Default constructor
       ******************************************************************/
      explicit ConditionPtr (Condition* ptr = 0) : mPtr (0) {
         Reset (ptr); }
      /** Creates a new copy of the event conditions. The event 
          condition is not adopted.
          @memo Constructor
       ******************************************************************/
      ConditionPtr (const Condition& cond) : mPtr (0) {
         Reset (cond); }
      /** Copies an event condition.
          @memo Copy constructor
       ******************************************************************/
      ConditionPtr (const ConditionPtr& ptr) : mPtr (0) {
         *this = ptr; }
      /** Destructs the event condition.
          @memo Destructor
       ******************************************************************/
      ~ConditionPtr () {
         Reset (0); }
      /** Assignment. Makes a new pointer with a true copy.
          @memo Assignment operator
       ******************************************************************/
      ConditionPtr& operator= (const ConditionPtr& ptr) {
         if (this != &ptr) Reset (ptr.Get() ? ptr->Copy() : 0); 
         return *this; }
      /** Assignment. Makes a new pointer with a true copy.
          @memo Assignment operator
       ******************************************************************/
      ConditionPtr& operator= (const Condition& cond) {
         Reset (cond.Copy()); 
         return *this; }
   #if 0
      /** Type converison into an event condition.
          @memo type conversion
       ******************************************************************/
      operator Condition& () {
         return *this; }
      /** Type converison into a constant event condition.
          @memo type conversion
       ******************************************************************/
      operator const Condition& () const {
         return *this; }
   #endif
      /** Get a pointer to the event condition.
          @memo Get
       ******************************************************************/
      Condition* Get() {
         return mPtr; }
      /** Get a pointer to the event condition.
          @memo Get
       ******************************************************************/
      const Condition* Get() const {
         return mPtr; }
      /** Dereference operator.
          @memo Dereference oprator
       ******************************************************************/
      Condition& operator*() {
         return *mPtr; }
      /** Dereference operator.
          @memo Dereference oprator
       ******************************************************************/
      const Condition& operator*() const {
         return *mPtr; }
      /** Member access oprator.
          @memo Member access oprator
       ******************************************************************/
      Condition* operator->() {
         return mPtr; }
      /** Member access oprator.
          @memo Member access oprator
       ******************************************************************/
      const Condition* operator->() const {
         return mPtr; }
      /** Release the event condition.
          @memo Release
       ******************************************************************/
      Condition* Release () {
         Condition* e = mPtr; mPtr = 0; 
         return e; }
      /** Resets the event condition. The new event condition
          gets adopted.
          @memo Reset
       ******************************************************************/
      void Reset (Condition* ptr = 0) {
         if (mPtr) delete mPtr; 
         mPtr = ptr; }
      /** Resets the event condition. The new event conditionsis copied.
          @memo Reset
       ******************************************************************/
      void Reset (const Condition& cond) {
         if (mPtr) delete mPtr; 
         mPtr = cond.Copy(); }
   
   private:
      /** Event pointer.
          @memo Pointer to event
       ******************************************************************/
      Condition* mPtr;
   };


}

#endif // _LIGO_EVENTCONDITIONPTR_H
