#ifndef _LIGO_EVENTCONDITION_H
#define _LIGO_EVENTCONDITION_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Condition						*/
/*                                                         		*/
/* Module Description: Defines an event	condition			*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Condition.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/


namespace events {


   class Argument;
   class Event;

/** Event condition. Event conditions can be evaluated on one, two 
    (coincidence) or three (triple-coincidence) events.
   
    @memo Defines an event condition
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Condition {
   public:
      /** Creates an event condition which is always false.
          @memo Default constructor
       ******************************************************************/
      Condition() {
      }
      /** Destructs the event condition.
          @memo Destructor
       ******************************************************************/
      virtual ~Condition() {
      }
      /** Returns a copy of the event condition. This method must be 
          overriden by all descendents.
          @memo Copy the event
          @return event copy
       ******************************************************************/
      virtual Condition* Copy() const = 0;
   
      /** Evaluate the condition. This function must be overriden
          by descendents.
          @memo Evaluate
          @param arg Event argument list
          @param val Condition value (return)
          @return True if condition could be evaluated
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, bool& val) const = 0;
      /** Condition call operator. Returns the same as evaluate 
          (should not be overriden).
          @memo () operator
          @param event Event upon condition is evaluated
          @param val Condition value (return)
          @return True if condition could be evaluated
       ******************************************************************/
      bool operator() (const Argument& arg, bool& val) const {
         return Evaluate (arg, val); }
      /** Condition call operator. Evaluates a condition on a single
          event (should not be overriden).
          @memo () operator
          @param arg Event argument list
          @param val Value (return)
          @return True if function could be evaluated
       ******************************************************************/
      bool operator() (const Event& event, bool& val) const;
   
      /** Returns a condition which yields always true.
          @memo True
       ******************************************************************/
      static const Condition& True();
      /** A condition which yields always false.
          @memo False
       ******************************************************************/
      static const Condition& False();
   
   protected:
      /** Copy constructor. This is almost never intended, so just
          disable it from the outside.
          @memo Copy constructor
       ******************************************************************/
      Condition (const Condition&) {
      }
      /** Assignment operator. This is almost never intended, so just
          disable it from the outside.
          @memo Assignment operator
       ******************************************************************/
      Condition& operator= (const Condition&) {
         return *this; }
   
   private:
   };


}

#endif // _LIGO_EVENTCONDITION_H
