#ifndef _LIGO_EVENTCOLUMNCACHE_H
#define _LIGO_EVENTCOLUMNCACHE_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: ColumnCache						*/
/*                                                         		*/
/* Module Description: Defines fast column access methods		*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: ColumnCache.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <map>
#include "events/ColumnType.hh"
#include "events/Type.hh"
#include "events/Event.hh"


namespace events {

   class Value;


/** Event column cache. To avoid looking up the column name for every
    processed event, the column cache will store data offset values of
    the most recently accesses events. The column cache is used
    by the Column class. For event array indeices the column cache
    will maintain a linked list of ColumnCache
   
    @memo Defines fast column access methods
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class ColumnCache {
   public:
      /** Creates an column cache.
          @memo Constructor
          @param name full name
       ******************************************************************/
      explicit ColumnCache (const char* name) : mColCache (0) {
         SetName (name); }
      /** Copy constructor.
          @memo Copy constructor
       ******************************************************************/
      ColumnCache (const ColumnCache& cc) : mColCache (0) {
         *this = cc; }
      /** Destructs an event column.
          @memo Denstructor
       ******************************************************************/
      ~ColumnCache() {
         if (mColCache) delete mColCache; }
      /** Assignment operator.
          @memo Assignment operator
       ******************************************************************/
      ColumnCache& operator= (const ColumnCache& cc);
      /** Returns a copy of the event column cache.
          @memo Copy the event column
          @return event copy
       ******************************************************************/
      ColumnCache* Copy() const {
         return new ColumnCache (*this); }
      /** Checks if this is a valid column.
          @memo Is valid?
          @return true if valid
       ******************************************************************/
      bool IsValid() const {
         return mValid; }   
      /** Returns the column value of the event.
          @memo Get column value
          @param events Event to pick column value from
          @param val Column value (return)
          @return true if column exists
       ******************************************************************/
      bool Get (const Event& event, Value& val) const;
      /** Sets the column value of the event.
          @memo Set column value
          @param events Event for which to set column value
          @param val New column value
          @return true if column exists and value could be set
       ******************************************************************/
      bool Set (Event& event, const Value& val);
      /** Returns a pointer to the last event. This method specifies
          the first event.
          @memo Get column event
          @param 
          @return Event if exist
       ******************************************************************/
      Event* GetEvent (Event& event);
      /** Returns a pointer to the last event. This method specifies
          the first event.
          @memo Get column event
          @param 
          @return Event if exist
       ******************************************************************/
      const Event* GetEvent (const Event& event) const;
   
      /** Set the name of the column.
          @memo Set column name
       ******************************************************************/
      void SetName (const char* name);
      /** Get the name of the column in its canonical representation.
          @memo Get column name
          @return Name
       ******************************************************************/
      std::string GetName() const;
      /** Resets the column caches.
          @memo Resets the column caches
       ******************************************************************/
      void Reset() const;
   
   protected:
      /** Clear the cache.
          @memo Clear cache
       ******************************************************************/
      void CacheClear() const;
      /** Initialize the cache.
          @memo Initialize cache
       ******************************************************************/
      void CacheInit();
      /** Lookup a column. Takes an events as the argument and returns
          a pointer to the data and its type. Returns true if the
          column exists.
          @memo Lookup column
          @param event Event
          @param data Pointer to column value (return)
          @param type Type of column value
          @return True if column exists
       ******************************************************************/
      bool CacheLookup (Event& event, ColumnType::data_ptr& data, 
                       ColumnType::Enum& type) const;
   
   public:
      /** Cache line. Caches offset and data type for a column name
          and event type pair.
       ******************************************************************/
      class CacheLine {
      public:
         /// Cache line status
         enum cachestatus {
         /// Uninitialzed
         kNeedInit = 0,
         /// Column not found
         kNotFound,
         /// Valid entry
         kValid
         };
         /// Default constructor (invalid cache line)
         CacheLine () : mStatus (kNeedInit), mColIndex (0), mOffset (0), 
         mDataType (ColumnType::kInvalid) {
         }
         /// Cache access
         bool Lookup (ColumnType::data_ptr& data, ColumnType::Enum& type, 
                     const Event& event, const std::string& name);
         /// Equality operator
         bool operator== (const CacheLine& cl) const;
         /// Less than operator
         bool operator< (const CacheLine& cl) const;
      private:
         /// Status
         cachestatus	mStatus;
         /// Column index
         int		mColIndex;
         /// Data offset
         int		mOffset;
         /// Data type
         ColumnType::Enum mDataType;
      };
      /// Map for column name cache (hash map?)
      typedef std::map <Type, CacheLine> Cache;
   
   private:
      /// Is valid?
      bool		mValid;
      /// Column name
      std::string	mName;
      /// Column cache for further indices
      ColumnCache*	mColCache;
      /// Cache a fixed column?
      bool		mColIsFixed;
      /// Fixed column offset cache
      int		mFixColOfs;
      /// Fixed column type cache
      ColumnType::Enum	mFixColType;
      /// Column name cache
      mutable Cache	mCache;
      /// Chache version (Layout add column modification version)
      mutable int	mCacheVers;
   };

}

#endif // _LIGO_EVENTCOLUMNCACHE_H
