//////////////////////////////////////////////////////////////////////////
//  									//
//  Algorithm implementation of templated code				//
//  									//
//////////////////////////////////////////////////////////////////////////
#if !defined (__GNUG__) || defined (_NEED_EVENTALGORITHM_TEMPL)
#include "PConfig.h"
#include <cstdio>
#include <queue>
#include <vector>
#include <functional>
#include <algorithm>
#include "events/Algorithm.hh"
#include "events/AlgorithmUtil.hh"
#include "events/Event.hh"
#include "events/Iterator.hh"
#include "events/WindowIterator.hh"
#include "events/Function.hh"
#include "events/FunctionPtr.hh"
#include "events/Condition.hh"
#include "events/ConditionPtr.hh"
#include "events/Argument.hh"
#include "events/List.hh"
#include "events/Chain.hh"
#include "events/Set.hh"
#include "events/Value.hh"
#include "events/Column.hh"
#include "events/Cluster.hh"
#include "Histogram1.hh"
#include "Histogram2.hh"
#include "events/XsilEvent.hh"


namespace events {

//______________________________________________________________________________
template <class DestIterator>
   DestIterator Select (
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd, 
                     DestIterator dest, 
                     const Condition& cond, 
                     const TimeWindow& window)
   {
      // create window iterators
      ConstWindowIterator i (sourceBegin, sourceEnd, 1, window);
      ConstWindowIterator end (sourceEnd, sourceEnd, 1, window);
      // loop over events
      for (; i != end; ++i) {
         // evaluate condition
         bool ret;
         if (cond.Evaluate (const_cast<Window&>(*i), ret) && ret) {
            // found a event which fullfills the condition
            *dest = i->Current();
            ++dest; 
         }
      }
      return dest;
   }

//______________________________________________________________________________
template <class DestIterator>
   DestIterator Coincidence (
                     const ConstWindowIterator& beg, 
                     const ConstWindowIterator& end,
                     DestIterator dest, 
                     const Condition& cond)
   {
      ConstWindowIterator i = beg;
      // use a time ordered priority queue to guarantee that coincidence 
      // events are added in the correct time order
      typedef std::priority_queue <Event, std::vector<Event>, 
         std::greater<Event> > eventqueue;
      eventqueue queue;
      // get order
      int order = beg.GetOrder();
      // create column objects
      Column* cols = new Column [order];
      for (int j = 0; j < order; ++j) {
         char buf[64];
         sprintf (buf, "Event(%i)", j);
         cols[j].SetName (buf);
      }
      Column ord ("Order");
      Column ifo ("Ifo");
      // get coincidence layout
      Layout clayout = Layout::GetCoincidence (order);
      if (!clayout.IsRegistered()) {
         return dest;
      }
      // loop over coincidences
      for (; i != end; ++i) {
         // check if we can remove events form the queue
         Time start = i->GetStartTime();
         while (!queue.empty() && (queue.top().GetTime() < start)) {
            *dest = queue.top();
            queue.pop();
            ++dest;
         }
         // check condition
         bool ret;
         if (cond.Evaluate (const_cast<Window&>(*i), ret) && ret) {
            // found a event which fullfills the coincidence condition
            Event ne (clayout);
            // fill in column values
            if (ne.IsValid()) {
               ne.SetTime (i->GetTime()); // set time
               ord.Set (ne, order);       // set order
               Value ifo0, ifo1;
               for (int j = 0; j < order; ++j) {
                  cols[j].Set (ne, i->Current(j)); // set (sub)event
                  if (j == 0) {
                     ifo.Get (i->Current(0), ifo0);
                  }
                  else {
                     ifo.Get (i->Current(0), ifo1);
                     ifo0 |= ifo1;
                  }
               }
               ifo.Set (ne, ifo0);        // set ifo set
               // Add new event to priority queue
               queue.push (ne);
            }
         }
      }
      delete [] cols;
      // remove remaining events form the queue
      while (!queue.empty()) {
         *dest = queue.top();
         queue.pop();
         ++dest;
      }
      return dest;
   }

//______________________________________________________________________________
template <class DestIterator>
   DestIterator Coincidence (
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     int order,
                     DestIterator dest, 
                     const TimeWindow& window,
                     const Condition& cond)
   {
      // use a time ordered priority queue to guarantee that coincidence 
      // events are added in the correct time order
      typedef std::priority_queue <Event, std::vector<Event>, 
         std::greater<Event> > eventqueue;
      eventqueue queue;
      // create window iterators
      ConstWindowIterator i (sourceBegin, sourceEnd, order, window);
      ConstWindowIterator end (sourceEnd, sourceEnd, order, window);
      // create column objects
      Column* cols = new Column [order];
      for (int j = 0; j < order; ++j) {
         char buf[64];
         sprintf (buf, "Event(%i)", j);
         cols[j].SetName (buf);
      }
      Column ord ("Order");
      Column ifo ("Ifo");
      // get coincidence layout
      Layout clayout = Layout::GetCoincidence (order);
      if (!clayout.IsRegistered()) {
         return dest;
      }
      // loop over coincidences
      for (; i != end; ++i) {
         // check if we can remove events form the queue
         Time start = i->GetStartTime();
         while (!queue.empty() && (queue.top().GetTime() < start)) {
            *dest = queue.top();
            queue.pop();
            ++dest;
         }
         // check condition
         bool ret;
         if (cond.Evaluate (const_cast<Window&>(*i), ret) && ret) {
            // found a event which fullfills the coincidence condition
            Event ne (clayout);
            // fill in column values
            if (ne.IsValid()) {
               ne.SetTime (i->GetTime()); // set time
               ord.Set (ne, order);       // set order
               Value ifo0, ifo1;
               for (int j = 0; j < order; ++j) {
                  cols[j].Set (ne, i->Current(j)); // set (sub)event
                  if (j == 0) {
                     ifo.Get (i->Current(0), ifo0);
                  }
                  else {
                     ifo.Get (i->Current(0), ifo1);
                     ifo0 |= ifo1;
                  }
               }
               ifo.Set (ne, ifo0);        // set ifo set
               // Add new event to priority queue
               queue.push (ne);
            }
         }
      }
      delete [] cols;
      // remove remaining events form the queue
      while (!queue.empty()) {
         *dest = queue.top();
         queue.pop();
         ++dest;
      }
      return dest;
   }

//______________________________________________________________________________
template <class DestIterator>
   DestIterator Coincidence (
                     const ConstIterator& source1Begin, 
                     const ConstIterator& source1End, 
                     const ConstIterator& source2Begin, 
                     const ConstIterator& source2End, 
                     DestIterator dest, 
                     const TimeWindow& window,
                     const Condition& cond)
   {
      // create window iterators
      ConstWindowIterator beg (source1Begin, source1End, 
                           source2Begin, source2End, window);
      ConstWindowIterator end (source1End, source1End, 
                           source2End, source2End, window);
      return Coincidence (beg, end, dest, cond);
   }

//______________________________________________________________________________
template <class DestIterator>
   DestIterator Coincidence (
                     const ConstIterator& source1Begin, 
                     const ConstIterator& source1End, 
                     const ConstIterator& source2Begin, 
                     const ConstIterator& source2End, 
                     const ConstIterator& source3Begin, 
                     const ConstIterator& source3End, 
                     DestIterator dest, 
                     const TimeWindow& window,
                     const Condition& cond)
   {
      // create window iterators
      ConstWindowIterator beg (source1Begin, source1End, 
                           source2Begin, source2End, 
                           source3Begin, source3End, window);
      ConstWindowIterator end (source1End, source1End, 
                           source2End, source2End, 
                           source3End, source3End, window);
      return Coincidence (beg, end, dest, cond);
   }

//______________________________________________________________________________
template <class DestIterator>
   DestIterator Coincidence (
                     const WindowIterator::InputStateList& state, 
                     DestIterator dest, 
                     const TimeWindow& window,
                     const Condition& cond)
   {
      // create window iterators
      ConstWindowIterator beg (state, window);
      WindowIterator::InputStateList estate = state;
      for (WindowIterator::InputStateList::iterator i = estate.begin();
          i != estate.end(); ++i) {
         i->mBegin = i->mEnd;
      }
      ConstWindowIterator end (estate, window);
      return Coincidence (beg, end, dest, cond);
   }


//______________________________________________________________________________
template <class DestIterator>
   DestIterator SelectClusters (
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd, 
                     DestIterator dest, 
                     int threshold, const TimeWindow& window, 
                     const Condition& cond)
   
   {
      // create window iterators
      ConstWindowIterator i (sourceBegin, sourceEnd, 1, window);
      ConstWindowIterator end (sourceEnd, sourceEnd, 1, window);
      // Cluster condition
      Cluster ccond (cond, threshold, window);
      // loop over events
      Column ord ("Order");
      Column ifo ("Ifo");
      Cluster::eventlist list;
      for (; i != end; ++i) {
         if (ccond.GetClusters (const_cast<Window&>(*i), list) >= threshold) {
            // found a cluster: get cluster layout
            Layout clayout = Layout::GetCluster (list.size());
            if (!clayout.IsRegistered()) {
               return dest;
            }
            // make a new cluster event
            Event ne (clayout);
            // fill in column values
            if (ne.IsValid()) {
               ne.SetTime (i->Current(0).GetTime()); // set time
               Value valord ((int)list.size());
               ord.Set (ne, valord);                 // set order
               Value ifo0;
               ifo.Get (i->Current(0), ifo0);
               ifo.Set (ne, ifo0);                   // set ifo
               for (int j = 0; j < (int)list.size(); ++j) {
                  char colname[64];
                  sprintf (colname, "Event(%i)", j);
                  ne.SetValue (colname, *list[j]);   // set (sub)event
               }
               // Add new event to destination
               *dest = ne;
               ++dest; 
            }
         }
      }
      return dest;
   }

//______________________________________________________________________________
// NOTE: the 2nd list has to be disjoint!!!
template <class DestIterator1, class DestIterator2, 
class DestIterator3, class DestIterator4>
   bool VetoGate (const ConstIterator& beg1, const ConstIterator& end1,
                 const ConstIterator& beg2, const ConstIterator& end2,
                 DestIterator1 out1p, DestIterator2 out1f,
                 DestIterator3 out2p, DestIterator4 out2f,
                 Interval offset)
   {  
      bool b;
      bool rb=true;
      events::ConstIterator itr1 = beg1;
      events::ConstIterator itr2 = beg2;
      events::ConstIterator vetoi = beg2;
      events::ConstIterator i;
      //n1p=n1f=n2p=n2f=0;
      //Int_t vetoC=0;
      events::Column colDur1("Duration");
      events::Column colDur2("Duration");
      Time START1; double DUR1;
      Time START2; double DUR2;  
      for (; itr1 != end1; ++itr1) {
         START1=itr1->GetTime();
         if (!colDur1.Get(*itr1,DUR1)) {
            fprintf (stderr, "Duration not readable!\n"); 
            return false; 
         } // Duration was not readable
         // loop thru all too early vetos
         if(itr2!=end2  &&  (rb=colDur2.Get(*itr2,DUR2))) { 
            START2=itr2->GetTime(); 
            b=(START2+DUR2+offset <= START1); 
         } 
         else { 
            b=false; 
         }
         while(b) {
            if(vetoi!=itr2) { // Veto is successful (failed)
               *out2f=*itr2;
               ++out2f;
            } 
            else { // Veto not successful (passed)
               *out2p=*itr2;
               ++out2p;	    
               ++vetoi;
            }
            ++itr2;
            if (itr2!=end2  &&  (rb=colDur2.Get(*itr2,DUR2))) { 
               START2=itr2->GetTime(); 
               b=(START2+DUR2+offset <= START1); 
            } 
            else { 
               b=false; 
            }
         }
         if(!rb) {
            fprintf (stderr, "Duration not readable!\n"); 
            return false; 
         } // Duration was not readable
         if(itr2!=end2 && (START1+DUR1)>(START2+offset)) { // signal vetoed (failed)
            *out1f=*itr1;
            ++out1f;
            // and veto successful
            if(vetoi==itr2) { 
               ++vetoi;
            }
            // check whether this signal is in coincidence with another veto
            if(vetoi!=end2  &&  (rb=colDur2.Get(*vetoi,DUR2))) { 
               START2=vetoi->GetTime(); 
               b=((START1+DUR1)>(START2+offset)); 
            } 
            else { 
               b=false; 
            }
            while(b) {
               ++vetoi;
               if(vetoi!=end2  &&  (rb=colDur2.Get(*vetoi,DUR2))) {
                  START2=vetoi->GetTime(); 
                  b=((START1+DUR1)>(START2+offset)); 
               } 
               else { 
                  b=false; 
               }	    
            }
            if(!rb) { 
               fprintf (stderr, "Duration not readable!\n"); 
               return false; 
            } // Duration was not readable
         } 
         else { // signal NOT vetoed (passed)
            *out1p=*itr1;
            ++out1p;
         }
      }
   
      // store the remaining vetoes
      for(;itr2!=end2; ++itr2) {
         if(vetoi!=itr2) { // Veto is successful (failed)
            *out2f=*itr2;
            ++out2f;
         } 
         else { // Veto not successful (passed)
            *out2p=*itr2;
            ++out2p;	    
            ++vetoi;
         }
      }
      return true;
   }

//______________________________________________________________________________
template <class DestIterator>
   DestIterator Sort (const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     DestIterator destBegin, 
                     const Function& func, 
                     bool ascending, int n)
   {
      // create an array of sort elements
      sortarray sarr;
      for (ConstIterator i = sourceBegin; i != sourceEnd; ++i) {
         sarr.push_back (SortElement (*i, func, ascending));
      }
      // sort it
      if ((n <= 0) || (n > (int)sarr.size() / 2)) {
         if (ascending) {
            std::sort (sarr.begin(), sarr.end());
         }
         else {
            std::sort (sarr.begin(), sarr.end(), 
                      std::greater<SortElement>());
         }
      }
      else {
         if (ascending) {
            std::partial_sort (sarr.begin(), sarr.end(),
                              sarr.begin() + n);
         }
         else {
            std::partial_sort (sarr.begin(), sarr.end(), 
                              sarr.begin() + n,
                              std::greater<SortElement>());
         }
      }
      // Now add sorted event to destination
      sortarray::iterator sortEnd = sarr.end();
      if ((n > 0) && (n < (int)sarr.size())) {
         sortEnd = sarr.begin() + n;
      }
      return std::copy (sarr.begin(), sortEnd, destBegin);
   }

//______________________________________________________________________________
template <class DestIterator>
   DestIterator Read (std::istream& inp, 
                     DestIterator insert, int max)
   {
      //Factory::Get().InitBasicLayouts(); // register basic layouts.
      Layout::GetSimple(); // register basic layouts.
      xml::xsilHandlerQueryEvent<DestIterator> query (insert, max);
      xml::xsilParser parser;
      parser.AddHandler (query);
      parser.Parse (inp);
      return insert;
   }

//______________________________________________________________________________
template <class SrcIterator> 
   SrcIterator Write (std::ostream& out, 
                     SrcIterator beg, SrcIterator end, int max)
   {
      events::ConstIterator b = 
         events::ConstIterator (events::StdIteratorImp<SrcIterator>(beg));
      events::ConstIterator e = 
         events::ConstIterator (events::StdIteratorImp<SrcIterator>(end));
      out << xml::xsilHeader() << std::endl;
      out << xml::xsilTime ("CreationTime", Now(), 1) << std::endl;
      xml::xsilEvent xsilevent ("Events", b, e, max);
      out << xsilevent << std::endl;
      out << xml::xsilTrailer() << std::endl;
      const events::StdIteratorImp<SrcIterator>* imp = 
         dynamic_cast<const events::StdIteratorImp<SrcIterator>*> (
                           xsilevent.get().GetImplementation());
      return imp ? imp->GetIterator() : end;
   }

}
#endif
