/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "SBTrigWriter.hh"
#include "TrigBase.hh"
#include "TrigProc.hh"
#include "SBTable.hh"
#include "xsil/Xwriter.hh"
#include "xsil/ligolw.hh"
#include "lmsg/ErrorList.hh"
#include <fstream>
#include <sstream>
#include <algorithm>
#include <stdexcept>

using namespace std;
using namespace trig;

//====================================  Sngl_burst writer constructor
SBTrigWriter::SBTrigWriter(void) 
    : mNPrcWrt(0)
{
}

//====================================  sngl_burst writer destructor
SBTrigWriter::~SBTrigWriter(void) {
}

//====================================  Add a trigger to the list
lmsg::error_type 
SBTrigWriter::addSegment(const Segment& s, const TrigProc& p) {
    throw std::runtime_error("SBTrigWriter doesn't handle segments");
    return lmsg::OK;
}


//====================================  Add a trigger to the list
lmsg::error_type 
SBTrigWriter::addTrigger(const TrigBase& t, const TrigProc& p) {
    proc_iter pit = insert_proc(p);
    if (!mTrgList.empty() && mTrgList.back() == t) {
	cerr << "Trigger repeated. ID: " << t.getID() 
	     << " SubID: " << t.getSubID()  << endl;
	return lmsg::Invalid;
    }
    mTrgList.push_back(t);
    mTrgList.back().setProcess(pit->getProcessID());
    if (getDebug() > 1) cout << "Trigger " << t.getID() << ":" << t.getSubID()
			     << " inserted at " << t.getTime() << endl;
    return lmsg::OK;
}

//======================================  Clear temporary tables
void 
SBTrigWriter::clear(const Time& start, const Time& end) {
    // mPrcList.clear();
    mNPrcWrt = mPrcList.size();
 
    //----------------------------------  Clear the used triggers
    for (trig_iter i=mTrgList.begin(); i!=mTrgList.end(); ) {
	trig_iter j = i++;
	if (!end || j->getTime() < end) {
	    mTrgList.erase(j);
	}
    }
}

//======================================  Get earliest time
Time
SBTrigWriter::getEarly(void) const {
    Time t(0);
    for (const_trig_iter i=mTrgList.begin(); i!=mTrgList.end(); ++i) {
	if (!t || i->getTime() < t) t =  i->getTime();
    }
    return t;
}

//======================================  Data size method
int
SBTrigWriter::getNTrigs(const Time& t) const {
    if (!t) return mTrgList.size();
    int N = 0;
    for (const_trig_iter i=mTrgList.begin(); i!=mTrgList.end(); ++i) {
	if (i->getTime() < t) N++;
    }
    return N;
}

//======================================  Add a process to the list
lmsg::error_type 
SBTrigWriter::setProcess(const TrigProc& p) {
    refProcess()  = p;
    // proc_iter pit = 
    insert_proc(refProcess());
    return lmsg::OK;
}

//======================================  Write out tables as xml.
lmsg::error_type 
SBTrigWriter::write(const string& file, const Time& start, 
		    const Time& end) const {
    if (mTrgList.empty()) return lmsg::OK;
    ofstream out(file.c_str());
    if (!out.is_open()) {
	cerr << "SBTrigWriter is unable to open file: " << file << endl;
	return lmsg::Failure;
    }
    xsil::Xwriter xw(out);
    xw.setDocType("SYSTEM \"http://gateway/doc/ligolwAPI/html/ligolw_dtd.txt\"");
    xsil::ligolw lwfile("ligo:ldas:file");
 
    //----------------------------------  Fill in the (new) processes
    if (int(mPrcList.size()) > mNPrcWrt) {
	const_proc_iter pStart = mPrcList.begin();
	for (int i=0; i<mNPrcWrt; ++i) ++pStart;

	ProcTable* pTab = new ProcTable(true);
	for (const_proc_iter i=pStart; i!=mPrcList.end(); ++i) {
	    pTab->addRow(*i);
	    //mNPrcWrt++;
	}
	lwfile.addObject(pTab);
    }
 
    //----------------------------------  Fill in the triggers
    int nTrgs = 0;
    SBTable* tTab = new SBTable(true);
    for (const_trig_iter i=mTrgList.begin(); i!=mTrgList.end(); ++i) {
	if (!end || i->getTime() < end) {
	    tTab->addRow(*i);
	    nTrgs++;
	}
    }
    if (nTrgs) lwfile.addObject(tTab);
    else       delete tTab;

    //----------------------------------  Spew the file.
    lwfile.Spew(xw);
    if (out.fail()) {
	cerr << "SBTrigWriter: Writing triggers to file: " << file 
	     << " failed." << endl;
	return lmsg::Failure;
    } else if (getDebug()) {
	cout << "Wrote " << nTrgs << " triggers to file " << file << endl;
    }
    return lmsg::OK;
}

//======================================  Add/find process in list
SBTrigWriter::proc_iter 
SBTrigWriter::insert_proc(const TrigProc& p) {
    proc_iter pit = find(mPrcList.begin(), mPrcList.end(), p);
    if (pit == mPrcList.end()) {
	int pid = mPrcList.size();
	mPrcList.push_back(p);
	pit = mPrcList.end(); --pit;
	if ( ! *(pit->getProcessID())) {
	    ostringstream id;
	    id << "process:process_id:" << pid;
	    pit->setProcessID(id.str());
	}
	if (getDebug()) {
	    cerr << "SBTrigWriter: Added process: " << pit->getName()
		 << " pID: " << pit->getProcessID() << endl;
	}
    } else if (getDebug() > 1) {
	cerr << "SBTrigWriter: Found process: " << pit->getName()
	     << " pID: " << pit->getProcessID() << endl;
    }
    return pit;
}
