/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_NAMEDATA_HH
#define LMSG_NAMEDATA_HH

#include <string>

#ifndef __CINT__
#include "lmsg/MsgAddr.hh"
#endif

namespace lmsg {
#ifndef __CINT__
    class TransInput;
    class TransOutput;
#endif

    /**  Process type enumeration.
      *  @memo Process types.
      */
    enum NameProcs {
        p_Any,
	p_Server,
	p_Monitor,
	p_Tunnel
    };

    /**  The name data class contains the data served by the name server.
      *  @memo Name data
      *  @author J. Zweizig
      *  @version 1.1; Modified 14-Jul-2000
      */
    class NameData {
    public:
        typedef unsigned int size_t;

    public:
        /**  Construct a NameData object and clear all fields.
	  *  @memo default constructor
	  */
        NameData(void);

        /**  Construct a NameData object and set all fields as specified.
	  *  @memo Data constructor.
	  *  @param Name  Server name string.
	  *  @param Addr  IP address, port and sub-process of the server
	  *  @param PType Process type of the server.
	  */
        NameData(const std::string& Name, const MsgAddr& Addr, 
		 NameProcs PType=p_Any);

        /**  Destroy an object.
	  *  @memo Destructor.
	  */
        virtual ~NameData(void);

        /**  Get a pointer to a character array containing the server
	  *  name. The pointer is valid only as long as the object exists.
	  *  @memo Get the server name.
	  *  @return Pointer to a character array containing the server name.
	  */
        const char* getName(void) const;

        /**  Return a constant reference to the server address. The reference
	  *  is valid only as long as the object exists.
	  *  @memo get the server address.
	  *  @return Reference to the server address.
	  */
        const MsgAddr& getAddr(void) const;

        /**  Set the server address.
	  *  @memo set server address.
	  *  @param addr Address to which the NameData address field is set.
	  */
        void  setAddr(const MsgAddr& addr);

	/**  Resolve the ip-address in the NameData address field if it 
	  *  is set to "IP_ANY".
	  *  @memo Fix ip address.
	  *  @param from IP address of process that created the NameData.
	  */
        void  fixAddr(const MsgAddr& from);

        /**  Get the server process type.
	  *  @memo Get process type.
	  *  @return Type of process described by NameData.
	  */
        NameProcs getPType(void) const;

        /**  Get a refernce to a constant std::string containing the server
	  *  name. The referecne is valid only as long as the object exists.
	  *  @memo Get the server name.
	  *  @return Constant reference to a std::string eith the server name.
	  */
	const std::string& refName(void) const;

        /**  Get the NameData object size in bytes.
	  *  @memo Get the size.
	  *  @return size of the namedata object.
	  */
        size_type size(void) const;

    private:
        /**  Server name
	 */
        std::string mName;

        /**  Server address
	 */
        MsgAddr     mAddr;

        /**  Server process type
	 */
        NameProcs   mPType;
    };

    //==================================  Inline methodsS
    inline const char* 
    NameData::getName(void) const {
	return mName.c_str();
    }

    inline const MsgAddr& 
    NameData::getAddr(void) const {
	return mAddr;
    }

    inline NameProcs 
    NameData::getPType(void) const {
	return mPType;
    }

    inline const std::string& 
    NameData::refName(void) const {
	return mName;
    }

}

#endif  // LMSG_NAMEDATA_HH
