/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef MONSERVER_HH
#define MONSERVER_HH

#ifndef DMTOFFLINE
#include "AppServer.hh"
#endif
#include <map>
#include <string>
#include <stdexcept>

class DataDesc;
class TSeries;
class FSeries;
class FSpectrum;
class Histogram1;
namespace containers {
    class ASD;
    class DFT;
    class PSD;
}


/**  The monitor data server API allows the user to specify data elements 
  *  to be served on demand to clients. The monitor server maintains a 
  *  list of the data elements and their names.
  *  @memo Monitor data server.
  *  @author John Zweizig
  *  @version 1.1; Modified April 20, 2000
  */
class MonServer 
#ifndef DMTOFFLINE
: private lmsg::AppServer 
#endif
{
public:
  typedef std::map<std::string, DataDesc> Dictionary;
  typedef Dictionary::iterator dict_iter;
  typedef Dictionary::const_iterator const_dict_iter;

public:
  /**  Construct a monitor data base and register the server with the 
    *  name server.
    */
  MonServer(const char* Process=0);

  /**  Close the server port and unregister its name.
    *  @memo Close the server.
    */
  virtual ~MonServer(void);

  /**  Test whether data have been defined for service.
   */
  bool dataServed(void) const;

  /**  Process a message from the monitor input port.
   */
  void processMsg(void);

  /**  Handle the attention interrupt delivered to a monitor.
    *  @memo Monitor attention handler.
    */
  void Attention(void);

  /**  Specify a power spectrum to be served on demand. The power
    *  spectrum data are converted to Single-precision float numbers
    *  before being sent to the client.
    *  @memo Serve a power spectrum.
    *  @param Name Data name
    *  @param fs   Pointer to data to be served.
    *  @param Comm Comment supplied with index.
    */
#if __SUNPRO_CC == 0x500
  void serveData(const char* Name, const void*    iVal, const char* Comm=0);
  void serveData(const char* Name, const long*    iVal, const char* Comm=0);
  void serveData(const char* Name, const double*  iVal, const char* Comm=0);
  void serveData(const char* Name, const std::string* iVal, const char* Comm=0);
  void serveData(const char* Name, const TSeries* iVal, const char* Comm=0);
  void serveData(const char* Name, const FSeries* iVal, const char* Comm=0);
  void serveData(const char* Name, const FSpectrum* iVal, const char* Comm=0);
  void serveData(const char* Name, const Histogram1* iVal, const char* Comm=0);
#else
  template<class T> 
  void serveData(const char* Name, const T* iVal, const char* Comm=0);
#endif

  /**  Set the server name.
   */
  void setServerName(const char* Name);

  /**  Get the debug flag valus
   */
  int getDebug(void) const;

  /**  Set the debug flag
   */
  void setDebug(int lvl);

  /**  Return a reference to the served dictionary data.
    */
  const Dictionary& refDictionary(void) const;

private:
  Dictionary mDict;
#ifdef DMTOFFLINE
  int  mDebug;
#endif
};

//======================================  Data Descriptor class
class DataDesc {
public:
    enum DataType {
	t_void,
	t_long,
	t_double,
	t_string,
	t_TSeries,
	t_FSeries,
	t_FSpectrum,
	t_Histogram1,
	t_asd,
	t_dft,
	t_psd
    };
    DataDesc(void);
    DataDesc(const long*        addr, const char* Comm=0);
    DataDesc(const double*      addr, const char* Comm=0);
    DataDesc(const std::string* addr, const char* Comm=0);
    DataDesc(const TSeries*     addr, const char* Comm=0);
    DataDesc(const FSeries*     addr, const char* Comm=0);
    DataDesc(const FSpectrum*   addr, const char* Comm=0);
    DataDesc(const Histogram1*  addr, const char* Comm=0);
    DataDesc(const containers::ASD* addr, const char* Comm=0);
    DataDesc(const containers::DFT* addr, const char* Comm=0);
    DataDesc(const containers::PSD* addr, const char* Comm=0);
    ~DataDesc(void);
    const void* getAddr() const;
    const char* getComment() const;
    DataType getType() const;
private:
  DataType    mType;
  const void* mAddr;
  std::string mComment;
#ifdef DMTOFFLINE
  int         mDebug;
#endif
};

//======================================  MonServer inline functions.
#if __SUNPRO_CC == 0x500
inline void MonServer::serveData(const char* Name, const long* data, 
				 const char* Comm) {
    mDict[Name] = DataDesc(data, Comm);
}
inline void MonServer::serveData(const char* Name, const double* data, 
				 const char* Comm){
    mDict[Name] = DataDesc(data, Comm);
}
inline void MonServer::serveData(const char* Name, const std::string* data, 
				 const char* Comm) {
    mDict[Name] = DataDesc(data, Comm);
}
inline void MonServer::serveData(const char* Name, const TSeries* data, 
				 const char* Comm) {
    mDict[Name] = DataDesc(data, Comm);
}
inline void MonServer::serveData(const char* Name, const FSeries* data, 
				 const char* Comm) {
    mDict[Name] = DataDesc(data, Comm);
}
inline void MonServer::serveData(const char* Name, const FSpectrum* data, 
				 const char* Comm) {
    mDict[Name] = DataDesc(data, Comm);
}
inline void MonServer::serveData(const char* Name, const Histogram1* data,
				 const char* Comm) {
    mDict[Name] = DataDesc(data, Comm);
}
#else
template<class T>
inline void
MonServer::serveData(const char* Name, const T* data, const char* Comm) {
    if (!data) throw std::runtime_error("serveData: Invalid data address.");
    if (!Name) throw std::runtime_error("serveData: Invalid name string.");
    mDict[Name] = DataDesc(data, Comm);
}
#endif

inline bool
MonServer::dataServed(void) const {
    return !mDict.empty();
}

inline const MonServer::Dictionary&
MonServer::refDictionary(void) const {
    return mDict;
}

#ifdef DMTOFFLINE
inline void 
MonServer::setDebug(int lvl) {
    mDebug = lvl;
}

inline int  
MonServer::getDebug(void) const {
    return mDebug;
}
#else
inline void 
MonServer::setDebug(int lvl) {
    AppServer::setDebug(lvl);
}

inline int  
MonServer::getDebug(void) const {
    return AppServer::getDebug();
}
#endif

//======================================  MonServer inline functions.
inline const void* 
DataDesc::getAddr(void) const {
    return mAddr;
}

inline const char* 
DataDesc::getComment(void) const {
    return mComment.c_str();
}

inline DataDesc::DataType 
DataDesc::getType(void) const {
    return mType;
}

#endif  //  MONSERVER_HH
