//
//    DPUSHRT is the application to push the data from a file directory 
//    into the shared memory partition in frame format. 
//
//    Syntax:
//
//       DpushRT infile <file> [partition <pname>] [lbuf <nbytes>] [nbuf <n>]
//             [log <logfile>] [time <ticks>]
//
//       file    Wildcard list of files to be pushed.
//       pname   Shared memory partition name
//       nbytes  Length of shared memory buffers
//       n       Number of shared memory buffers
//       logfile Log file.
//       ticks   Watchdog (statistics summary) timeout interval
//
//    Notes:
//      1) Partitions aren't currently created by the program.
//
//    Revision History:
//
//      14-Apr-2000  J. Zweizig
//          Preliminary version.
//
//------- A few definitions
//
//    DEFNDSIP the default NDS server IP address.
//    DEFPART  is the default partition name that is used if the "partition" 
//             command line argument is not specified.
//    CDSDATA  receive data in CDS proprietary format.
//
// #define _POSIX_C_SOURCES
#if defined(sun) && !defined(__EXTENSIONS__)
#define __EXTENSIONS__
#endif

//  Use CDSDATA to disable frame checking.
#define CDSDATA 1
#define DEFPART "LIGO_Offline"

//------- Include till you drop.
#include "PConfig.h"
#include <iostream>
#include <fstream>
#ifdef __GNU_STDC_OLD
#include <strstream.h>
#else
#include <sstream> 
#endif
#include <stdlib.h>
#include <unistd.h>
#include <string>
#include <time.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/time.h>
#include "SigFlag.hh"
#include "SysError.hh"
#include "Time.hh"
#include "lsmp_prod.hh"
#include "framedir.hh"
#include "CheckFrame.hh"

using namespace std;

//------- Statistics class.
class Stats {
public:
  typedef unsigned long ulong_t;
  Stats();
  ~Stats();
  void Log(ostream& out);
  void Reset(void);
  void Start(ulong_t Secs);
  bool Test(void);
  void Update(ulong_t FrameID, ulong_t GPS, ostream& out);
private:
  ulong_t nSecs;
  SigFlag watchdog;
  time_t  Stamp;
  ulong_t GPSstart;
  ulong_t GPSlast;
  ulong_t RecIDstart;
  ulong_t RecIDlast;
  ulong_t FrameCount;
  ulong_t FrameLost;
  ulong_t FrameUnsent;
};

//---->  Shared memory output device
static LSMP_PROD* OPart=0;

//---->  Termination signal flag pointer
static SigFlag term;
static bool    active;
static bool    debug(false);

//---->  Watchdog statistics timer.
static Stats pgmStat;

//---->  Error log control;
static bool LogOn;
static ofstream logfi;

//---->  Time control
static timeval  startTime;
static unsigned long FrameID;
static double compress(1.0);
static FrameDir fDir;
static FrameDir::file_iterator pFrame;

//----> Internal entry points.
//
//    Dpsh_Init     Set up the input and output functions
//    Dpsh_Halt     Set down.
//    Dpsh_Read     Read routine.
//    Dpsh_Check    Check a frame is consistent.
//    stamp         Generate a time stamp.
static bool Dpsh_Init(int argc, const char **argv);
static void Dpsh_Halt(void);
static void Dpsh_Read(void);
static bool Dpsh_Check(unsigned int length, const char* addr);
static string stamp(time_t now);
//
//    main function. 
//      * Set up. This includes initializing the input and output 
//        files and the exception handling.
//      * Loop over records until the ceiling falls in on us
//      * Call Dpsh_Read to process a frame
//      * When the roof caves in set-down the application
//
int main(int argc, const char **argv) {

    active = !Dpsh_Init(argc, argv);

    while (active && !term) {
        if (LogOn && pgmStat.Test()) pgmStat.Log(logfi);
        Dpsh_Read();
    }
    Dpsh_Halt();
    return 0;}

//
//    Dpsh_Init - set up the internal environment for pushing data.
//       * Parse any command line parameters.
//       * Create a shared memory partition for data distribution.
//       * Make a connection to the CDS Network Data Server. Register 
//         notification of all frames.
//       * Set up the termination signal handling.
//
static bool Dpsh_Init(int argc, const char **argv) {
    int   nbuf = 1, lbuf = 1000000, Secs=1200;
    LogOn = false;
    compress = 1.0;

    //---------------------------------  Define default parameters
    const char*   ParName = getenv("LIGOSMPART");
    if (!ParName) ParName = DEFPART;

    //---------------------------------  Parse the arguments
    for (int i=1 ; i<argc ; i++) {
        string argi = argv[i];
        if (argi == "partition") {
	    ParName  = argv[++i];
	} else if (argi == "infile") {
	    fDir.add(argv[++i]);
	} else if (argi == "lbuf") {
	    lbuf = strtol(argv[++i], 0, 0);
	} else if (argi == "nbuf") {
	    nbuf = strtol(argv[++i], 0, 0);
	} else if (argi == "time") {
	    Secs = strtol(argv[++i], 0, 0);
	} else if (argi == "compress") {
	    compress = strtod(argv[++i], 0);
	} else if (argi == "debug") {
	    debug = true;
	} else if (argi == "log") {
	    LogOn = true;
	    i++;
	    string lfile;
	    if (i >= argc || !*argv[i]) {
	        char logName[128];
		LocalStr(Now(), 
			 logName, 
			 "/logs/DpushRT-%4Y-%02m-%02d-%02H-%02N.log");
		lfile  = getenv("HOME");
		lfile += logName;
	    } else {
	        lfile  = argv[i];
	    }
	    logfi.open(lfile.c_str(), ios::out);
	    if (logfi.good()) {
	        logfi << stamp(0) << argv[0] << " Starting..." 
		     << endl;
	    } else {
		LogOn = false;
	    }
	} else {
	    cerr << argv[0] << ": Invalid argument name " << argv[i] 
		 << endl;
	    cerr << "Command syntax:" << endl << endl;
	    cerr << argv[0] << " infile <file> [partition <partition>] "
		 << "[debug] \\" << endl;
	    cerr << "       [nbuf <#buffers>] [lbuf <buffer-length>] "
		 << "[log <log-file>] [time <stat-timer>] [compress <rate>]" 
		 << endl;
	    return true;
	}
    }

    //---------------------------------  Open the output partition.
    if (debug) cout << "Opening partition " << ParName << " for output." 
		    << endl;
    OPart = new LSMP_PROD(ParName, nbuf, lbuf);
    if (!OPart->valid()) {
        cerr << "Unable to access partition " << ParName << ": " 
	     << OPart->Error() << "." << endl;
	if (LogOn) logfi << stamp(0) << "Unable to access partition " 
		      << ParName << ": " << OPart->Error() << "." 
		      << endl;
	return true;
    }

    //---------------------------------  Set up the statistics task
    if (LogOn) pgmStat.Start(Secs);

    //---------------------------------  Set up the termination signals
    term.add(SIGINT);
    term.add(SIGTERM);
    term.add(SIGHUP);

    //---------------------------------  Get the start time;
    gettimeofday(&startTime, 0);
    FrameID = 0;
    pFrame  = fDir.begin();

    //---------------------------------  Done, quit.
    return false;
}

//
//    Dpsh_Halt - set down.
//       * Release the shared memory partition.
//       * Close the connection to the CDS Network Data Server.
//       * Cancel the termination signal handling.
//
static void Dpsh_Halt(void) {

    //---------------------------------  Disable the signal
    term.zero();

    //---------------------------------  Delete the partition
    if (OPart) delete OPart;

    //---------------------------------  Final statistics
    pgmStat.Log(logfi);

    //---------------------------------  Close the error log
    if (LogOn) {
        LogOn = false;
	logfi << stamp(0) << "Dpush terminating. Error log closed" 
	     << endl;
        logfi.close();
    }
}

//
//    Dpsh_Read - read the data.
//      *  Wait for a frame to arrive.
//      *  Read in the data
//      *  Pack data into a frame
//      *  Write frame to shared memory.
//
static void Dpsh_Read(void) {
    //---------------------------------  Wait for a data message
    if (pFrame == fDir.end()) {
        cout << "Stopping after last file" << endl;
        active = false;
	return;
    }

    //---------------------------------  Open the file
    FrameDir::gps_t GPS = pFrame->getStartGPS();
    string fName = pFrame->getFile();
    int fd = open(fName.c_str(), O_RDONLY);
    if (fd < 0) {
        cerr << "Error opening file: " << fName << endl;
        active = false;
	return;
    } else if (debug) {
        cout << "Opened frame file: " << fName << endl;
    }

    //---------------------------------  Get the file length - check buffer
    struct stat statBuf;
    fstat(fd, &statBuf);
    long lFile = statBuf.st_size;

    if (lFile > OPart->getBufferLength()) {
        active = false;
	cerr << "Frame longer than shared memory buffer." << endl;
	if (LogOn) logfi << stamp(0) << "Frame longer than shared memory buffer"
			 << endl;
	return;
    }

#if defined(P__SOLARIS)
    directio(fd, DIRECTIO_ON);
#endif

    //---------------------------------  Get a buffer
    char* buffer = reinterpret_cast<char*>(0);
    while (!buffer) {
        try {
            buffer = OPart->get_buffer();
	} catch (exception& e) {
	    active = false;
	    cerr << "Error allocating buffer:" << e.what() << endl;
	    if (LogOn) logfi << stamp(0) << "Error allocating buffer:" 
			     << e.what() << endl;
	    return;
	}	    
    }

    //---------------------------------  Wait for data.
    timeval now, diff;
    gettimeofday(&now, 0);
    double cTime = compress*FrameID;
    int    nSec  = int(cTime);
    int    nUSec = int((cTime - double(nSec))*1000000.0);
    diff.tv_sec  = startTime.tv_sec  + nSec;
    diff.tv_usec = startTime.tv_usec + nUSec;
    if (diff.tv_usec >= 1000000) {
        diff.tv_usec -= 1000000;
        diff.tv_sec++;
    }
    if (diff.tv_sec >= now.tv_sec) {
        diff.tv_sec -= now.tv_sec;
        if (diff.tv_usec >= now.tv_usec) {
	    diff.tv_usec -= now.tv_usec;
	} else if (diff.tv_sec > 0) {
	    diff.tv_sec--;
	    diff.tv_usec += 1000000 - now.tv_usec;
	} else {
	    diff.tv_usec = 0;
	}
	unsigned long ttot = diff.tv_sec*1000000 + diff.tv_usec;
	if (ttot > 0) {
	    if (debug) cout << "Waiting " << ttot << "usec." << endl;
	    usleep(ttot);
	}
    }

    //---------------------------------  Copy the data
    long length = read(fd, buffer, lFile);
    if (debug) cout << "Read frame: file-length/rc " << lFile
		    << " / " << length << endl;
    close(fd);
    pFrame++;
    FrameID++;

    if (length == 0) {
        OPart->return_buffer();
	cerr << "Read error." << endl;
	if (LogOn) logfi << stamp(0) << "Read error." << endl;
	active = false;
    } else if (!Dpsh_Check(length, buffer)) {
        OPart->return_buffer();
	if (LogOn) {
	    logfi << stamp(0) << "Data check failed on frame " << FrameID 
		 << endl;
	} else {
	    cerr << "Data check failed on frame " << FrameID << endl;
	}
    } else {
        OPart->SetID(GPS);
        OPart->release(length);
    }

    //----------------------------------  Check for discontinuities
    if (LogOn) pgmStat.Update(FrameID, GPS, logfi);
    return;
}

//=======================     Frame Check Code      =======================

//
//  Dpsh_Check  Checks the incoming frame data structure headers for 
//              consistency.
//
//  The structure headers must meet the following requirements
//  1) The length must be greater than 8 (the length of the header) and 
//     less than the length of the Frame.
//  2) The structure type must be 1 or 2 or have been defined in a previous
//     SH structure. The Structure ID must be in the range 0 < ID < MAXID 
//     where MAXID is arbitrarily set to 256 (for now).
//  3) the structure instance numbers must increase by one with each new 
//     instance and start at either 0 or 1.
//
//  In addition the SH structures are unpacked and must be internally 
//  consistent (see procSH).
//
//  Dpsh_Check returns true if the frame passes the requirements.
//
class DPCheck : public CheckFrame {
  public:
    DPCheck(istream& s);
    void ErrorOut(const string& msg);
};

DPCheck::DPCheck(istream& s) 
  :  CheckFrame(s) {
}

void 
DPCheck::ErrorOut(const string& s) {
    if (LogOn) logfi << stamp(0) << s << endl;
    cerr << s << endl;
}

static bool
Dpsh_Check(unsigned int length, const char* addr) {
#ifndef CDSDATA
#ifdef __GNU_STDC_OLD
    istrstream s(addr, length);
#else
    istringstream s(string(addr, length));
#endif
    DPCheck c(s);
    try {
        c.Check();
    } catch (exception& e) {
        if (LogOn) logfi << stamp(0) << e.what() << endl;
	cerr << e.what() << endl;
	return false;
    }
#endif  // !def(CDSDATA)
    return true;
}

static string stamp(time_t now) {
    if (now == 0) now = time(0);
    string datel(::ctime(&now)+4, 15);
    datel += " ";
    return datel;
}

//--------------------------------------  Statistics class methods
Stats::Stats(void) 
  : GPSlast(0), RecIDlast(0)
{}

Stats::~Stats() {
}

bool 
Stats::Test() {
  return watchdog;
}

void 
Stats::Log(ostream& out) {
    time_t now = time(0);
    float Rate;
    if (now == Stamp) Rate = 1.0;
    else              Rate = float(GPSlast - GPSstart) / float(now-Stamp);
    if (FrameLost || FrameUnsent || Rate > 1.02 || Rate < 0.98) {
        out << stamp(now) << " Statistics - Frames read: " 
	    << FrameCount << " Lost: " << FrameLost << " Unsent: " 
	    << FrameUnsent << " Rate: " << Rate << endl;
    }
    Reset();
    watchdog.clear();
}

void 
Stats::Reset(void) {
    GPSstart   = GPSlast;
    RecIDstart = RecIDlast;
    FrameCount = 0;
    FrameLost  = 0;
    FrameUnsent = 0;
    ::time(&Stamp);
}

void 
Stats::Start(ulong_t Secs) {
    struct itimerval ticks;

    ticks.it_value.tv_sec  = Secs;
    ticks.it_value.tv_usec  = 0;
    ticks.it_interval.tv_sec  = Secs;
    ticks.it_interval.tv_usec  = 0;
    nSecs = Secs;
    setitimer(ITIMER_REAL, &ticks, 0);
    watchdog.setMode(SigFlag::kBlock);
    watchdog.add(SIGALRM);
    time(&Stamp);
}

void 
Stats::Update(ulong_t FrameID, ulong_t GPS, ostream& log) {
    if (!GPSlast) {
        log << stamp(0) << "First frame read, GPS: " << GPS 
	    << " Frame: " << FrameID << endl;
        GPSlast   = GPS;
	RecIDlast = FrameID;
	Reset();
    } else if (FrameID <= RecIDlast) {
        log << stamp(0) << "Frame received out of order, Frame=" << FrameID
	    << " expected " << RecIDlast+1 << endl;
    } else if (GPS <= GPSlast) {
        log << stamp(0) << "Frame received out of order, GPS=" << GPS 
	    << " expected " << GPSlast+1 << endl;
    } else {
	ulong_t lost = FrameID - RecIDlast - 1;
        FrameLost   += lost;
	FrameUnsent += GPS - GPSlast - 1 - lost;
    }
    FrameCount++;
    GPSlast   = GPS;
    RecIDlast = FrameID;
}
