/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    File BitTest.cc
//
//    Bit test monitor class. Test that all the low order bits of a 
//    data series are seen in both '1' and '0' states. Count the longest
//    sequence of repeated data for each channel.
//
#include <iostream>
#include <iomanip>
#include <fstream>
#include "dvTest.hh"
#include "Dacc.hh"
#include "Time.hh"
#include "TrigRslt.hh"
#include "framecpp/FrameH.hh"
#include "framecpp/FrAdcData.hh"
#include "framecpp/FrVect.hh"
#include "ParseLine.hh"

using namespace std;

//-->  The next three lines are needed if you are going to generate triggers.
//     The descriptive title in PIDTITLE should the monitor function.
#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/dvTest/dvTest.cc 7126 2014-06-27 09:14:07Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "Channel "
#include "ProcIdent.hh"

//======================================  Generate the main function.
EXECDMT(dvTest)

//======================================  Constructor
dvTest::dvTest(int argc, const char *argv[])
    : DMTBase(argc, argv), mConfig("dvTest.conf"), mGblLog(false), 
      mGblTrig(false), mRawData(false), mLastPrint(0)
{
    //----------------------------------  Parse the arguments
    mConfig  = "dvTest.conf";
    for (int i=1 ; i<argc ; i++) {
	string argi = argv[i];
        if (argi == "-conf") {
	    mConfig = argv[++i];
        } else if (argi == "-log") {
	    mGblLog = true;
        } else if (argi == "-raw") {
	    mRawData = true;
        } else if (argi == "-trig") {
	    mGblTrig = true;
        } else if (isEnvArg(argv[i])) {
	    ++i;
        } else {
	    cerr << "Invalid argument: " << argi << endl;
	    cerr << "Command syntax is :" << endl;
	    cerr << "  " << *argv << " [-conf <config>] [-raw] [-trig]" << endl;
	    finish();
	    return;
	}
    }
    getDacc().setIgnoreMissingChannel(true);

    //----------------------------------  Read the configuration file
    if (ReadConfig()) {
        finish();
    }

    const char* htmloutenv = "DMTHTMLOUT";
    mStatusFile.clear();
    if (getenv(htmloutenv)) mStatusFile = getenv(htmloutenv);
    mStatusFile += "/dvTest.txt";

    cout << "Starting dvTest $Id: dvTest.cc 7126 2014-06-27 09:14:07Z john.zweizig@LIGO.ORG $" << endl;
}

//======================================  Read in the configuration
bool
dvTest::ReadConfig(void) {

    //----------------------------------  Open the configuration file
    ParseLine fd(mConfig.c_str());
    if (!fd.isOpen()) {
        cerr << "Unable to open dvTest configuration file: " 
	     << mConfig << endl;
        finish();
        return true;
    }
    
    //----------------------------------  Read in the configuration file
    bool syntax   = false;
    while (fd.getLine() >= 0) {
	int nArg = fd.getCount();
	if (!nArg) continue;
	bool trig     = mGblTrig;
	bool log      = mGblLog;
	bool procdata = false;
	for (int i=1; i<nArg; ++i) {
	    string argi = fd[i];
	    if (argi == "-adc") {
		procdata = false;
	    } else if (argi == "-log") {
		log = true;
	    } else if (argi == "-nolog") {
		log = false;
	    } else if (argi == "-notrigger") {
		trig = false;
	    } else if (argi == "-procdata") {
		procdata = true;
	    } else if (argi == "-trigger") {
		trig = true;
	    } else {
		cerr << "Undefined configuration argument: " << argi << endl;
		syntax = true;
		break;
	    }
	}
	if (procdata && !mRawData) {
	    mProcChan.push_back(ChanDV(fd[0], mTrig));
	    mProcChan.back().setTrigEnable(trig);
	    mProcChan.back().setLog(log);
	} else {
	    mAdcChan.push_back(ChanDV(fd[0], mTrig));
	    mAdcChan.back().setTrigEnable(trig);
	    mAdcChan.back().setLog(log);
	}
    }
    return syntax;
}

//======================================  Process one frame (FrameCPP).
void
dvTest::ProcessFrame(DMTBase::frame_ptr_type frame) {
    Time tFrame(frame->GetGTime().getSec(), frame->GetGTime().getNSec());

    //----------------------------------  Allocate local data storage
    for (channel_iter itr=mAdcChan.begin(); itr!=mAdcChan.end(); itr++) {
        const char* ChName = itr->getChannel();
	channel_iter tent = itr;

        //------------------------------  Get the adc pointer.
	if (Debug() > 2) cerr << "Look for channel: " << ChName << endl;
	fradcdata_pointer adc;
	int order = getDacc().findAdcOrdered(ChName, adc);
	if (!order) {
	    if (Debug()) cout << "Channel: " << ChName 
			      << " not found." << endl;
	    continue;
	} else if (order < 0 && itr != mAdcChan.begin()) {
	    channel_iter save = tent;
	    --itr;
	    if (Debug() > 2) cerr << "Reorder channels: " 
				  << itr->getChannel() << " <-> "
				  << save->getChannel() << endl;
	    tent = mAdcChan.insert(itr, *save);
	    mAdcChan.erase(save);
	}
	if (Debug() > 1) cerr << "Scan AdcData: " << ChName << endl;
	itr->Scan(adc, tFrame);
    }

    //----------------------------------  Allocate local data storage
    for (channel_iter itr=mProcChan.begin(); itr!=mProcChan.end(); ++itr) {
        const char* ChName = itr->getChannel();
	channel_iter tent = itr;

        //------------------------------  Get the proc data pointer.
	if (Debug() > 2) cout << "Look for channel: " << ChName << endl;
	frprocdata_pointer proc;
	int order = getDacc().findProcOrdered(ChName, proc);
	if (!order) {
	    if (Debug()) cout << "Channel: " << ChName 
			      << " not found." << endl;
	    continue;
	} else if (order < 0 && itr != mProcChan.begin()) {
	    channel_iter save = tent;
	    --itr;
	    if (Debug() > 2) cout << "Reorder channels: " 
				  << itr->getChannel() << " <-> "
				  << save->getChannel() << endl;
	    tent = mProcChan.insert(itr, *save);
	    mProcChan.erase(save);
	}

	if (Debug() > 1) cerr << "Scan ProcData: " << ChName << endl;
	itr->Scan(proc, tFrame);
    }

    Time now = Now();
    if (now > mLastPrint + Interval(300.0)) {
	ofstream out(mStatusFile.c_str());
	if (out.is_open()) PrintStatus(out);
	mLastPrint = now;
    }
}

//======================================  Print final Results
dvTest::~dvTest(void) {
    //----------------------------------  Flush any remaining segments
    for (channel_iter itr=mAdcChan.begin(); itr!=mAdcChan.end(); itr++) {
	itr->writeSegment();
    }

    for (channel_iter itr=mProcChan.begin(); itr!=mProcChan.end(); itr++) {
	itr->writeSegment();
    }
}

void 
dvTest::PrintStatus(std::ostream& out) const {
    ChanDV::PrintHeader(out);

    for (const_chan_iter itr=mAdcChan.begin(); itr!=mAdcChan.end(); itr++) {
	itr->PrintStatus(out);
    }

    for (const_chan_iter itr=mProcChan.begin(); itr!=mProcChan.end(); itr++) {
	itr->PrintStatus(out);
    }
}

//======================================  Channel constructor
ChanDV::ChanDV(const char *name, TrigClient& tc) 
  : mChanName(name), mTrigEnable(true), mLogEnable(false), mTrig(tc),
    mCurVal(0), mStart(0), mCurrent(0), mCount(0), mLast(0)
{
}

//======================================  Channel destructor
ChanDV::~ChanDV() {
    writeSegment();
}

//======================================  Channel destructor
void 
ChanDV::PrintHeader(std::ostream& out) {
    char stamp[64];
    out << "dvTest Channel statistics: " 
	<< TimeStr(Now(), stamp, "%Y-%02m-%02d %2H:%02N:%02S")
	<< endl;
    out << endl;
    out << "Channel                          Errors    Last" << endl;
}

//======================================  Channel destructor
void 
ChanDV::PrintStatus(std::ostream& out) const {
    char stamp[64];
    std::streamsize wsav = out.width();
    out << setw(32) << left  << mChanName.c_str() 
	<< setw(10) << right << mCount;
    if (!mLast) out << " --" << endl;
    else        out << TimeStr(mLast, stamp, " %Y-%02m-%02d %2H:%02N:%02S")
		    << endl;
    out.width(wsav);
}

//======================================  Check everything
void
ChanDV::Scan(fradcdata_pointer adc, const Time& t0) {
    //----------------------------------  Check the dataValid
    short dvValue = adc->GetDataValid();
    if (!dvValue) return;

    //----------------------------------  Find the datavalid Aux data.
#ifndef FCPP_SHARED_PTRS
    frvect_pointer dvVect = 0;
#else
    frvect_pointer dvVect;
#endif
    int nAux = adc->RefAux().size();
    for (int i=0; i<nAux; ++i) {
	dvVect = adc->RefAux()[i];
	if (dvVect && dvVect->GetName() == "dataValid") break;
#ifndef FCPP_SHARED_PTRS
	dvVect = 0;
#else
	dvVect.reset();
#endif
    }

    //----------------------------------  Generate a frame-wide segment if 
    //                                    Aux data are not specified.
    if (!dvVect) {
	dvVect = adc->RefData()[0];
	FrameCPP::Dimension dim = dvVect->GetDim(0);
	Time tStart = t0 + Interval(dim.GetStartX());
	Interval dT(dim.GetDx() * dim.GetNx());
	segment(tStart, dT, dvValue);
	return;
    }

    //-----------------------------------  Scan the dataValid vector
    Scan(dvVect, t0);
}

//======================================  Check everything
void
ChanDV::Scan(frprocdata_pointer proc, const Time& t0) {

    //----------------------------------  Find the datavalid Aux data.
#ifndef FCPP_SHARED_PTRS
    frvect_pointer dvVect = 0;
#else
    frvect_pointer dvVect;
#endif
    int nAux = proc->RefAux().size();
    for (int i=0; i<nAux; ++i) {
	dvVect = proc->RefAux()[i];
	if (dvVect && dvVect->GetName() == "dataValid") break;
#ifndef FCPP_SHARED_PTRS
    dvVect = 0;
#else
    dvVect.reset();
#endif
    }

    //-----------------------------------  Scan the dataValid vector
    Scan(dvVect, t0);
}

//======================================  Check everything
void
ChanDV::Scan(frvect_pointer vect, const Time& t0) {
    if (!vect) return;

    //-----------------------------------  Uncompress the vector
    vect->Uncompress();
    int N = vect->GetNData();
    if (N<=0) return;

    //-----------------------------------  Get the vector binning info.
    FrameCPP::Dimension dim=vect->GetDim(0);
    Time tBase = t0 + Interval(dim.GetStartX());
    Interval dT = dim.GetDx();

    //-----------------------------------  Scan the dataValid vector
#ifndef FCPP_SHARED_PTRS
    const short* p = reinterpret_cast<const short*>(vect->GetData());
#else
    const short* p = reinterpret_cast<const short*>(vect->GetData().get());
#endif
    short val = *p;
    int start = 0;
    for (int i=1; i<N; ++i) {
	if ( p[i] != val ) {
	    segment(tBase+double(start)*dT, dT*double(i-start), val);
	    start=i;
	    val = p[i];
	}
    }
    segment(tBase+double(start)*dT, dT*double(N-start), val);
}

//======================================  Check everything
void
ChanDV::segment(const Time& t0, Interval dT, short val) {
    if (t0 > mCurrent || mCurVal != val) writeSegment();
    if (!mStart) {
	mStart   = t0;
	mCurVal  = val;	
    }
    mCurrent = t0 + dT;
}

//======================================  Check everything
void
ChanDV::writeSegment(void) {
    if (mCurVal && mStart != Time(0) && mStart < mCurrent) {
	if (mLogEnable) {
	    cout << "Chan: " << mChanName << " Val: " << mCurVal 
		 << " time: " << mStart << "-" << mCurrent << endl;
	}
	if (mTrigEnable) {
	    Interval dT = mCurrent - mStart;
	    trig::TrigBase t("dvTest", mChanName.c_str(), mStart, dT);
	    mTrig.sendTrigger(t);
	}
	mCount++;
	mLast = mStart;
    }
    mStart  = Time(0);
    mCurVal = 0;
}

//======================================  Set trigger enable
void 
ChanDV::setTrigEnable(bool te) {
    mTrigEnable = te;
}

//======================================  Set log enable
void 
ChanDV::setLog(bool le) {
    mLogEnable = le;
}
