/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef DQ_MODULE_HH
#define DQ_MODULE_HH 1

#include "MultiStream.hh"
#include "DQ_Bit.hh"
#include "frame_name.hh"
#include "FrWriter.hh"
#include "TSeries.hh"
#include <vector>
#include <string>
#include <iosfwd>

/**  The %DQ_Module class implements the %DQ_Module monitor. This montor calls 
  *  a boolean-valued c++ function of one or more time series and uses the 
  *  result value to set a specified bit in a uint32_t bit-mask channel. The 
  *  monitor loads a plugin.
  */
class DQ_Module : public MultiStream {
public:
   /**  %DQ_odule monitor constructor. The monitor requires a single json
     *  configuration file path, and also parses the usual MultiStream 
     *  command line arguments.
     */
   DQ_Module(int argc, const char* argv[]);
   /**  %DQ_Module destructor.
     */
   ~DQ_Module(void);

   /**  Collect all the channel names requested as input to the dq bit 
     *  evaluation functions and the channels requested to be copied to
     *  the output frame.
     */ 
   void book_channels(void);

   /**  Call the init methods for each of the defined flag instances.
     *  \brief Initialize flag evaluation.
     */
   void init_all(void);

   /**  Create output channels for all defined flags. This method also 
     *  builds a mask for each bit that will be used.
     *  \brief Initialize output channels.
     */
   void init_output_channels(void);
   
   /**  Parse the json configuration file.
     */
   void config(const std::string& cfg_file);

   /**  Process a stride of input data.
     */
   void ProcessData(void);
   void put_config(std::ostream& out) const;
   void set_flag(DQ_bit* dq_def, const Time& t0, Interval dt);
   void write_frame(const Time& t0, Interval dT);
 private:
   double stride;            ///<  computation stride (frame length)
   double flag_step;         ///<  mask calculation stride length
   int    parity_bit;        ///< Parity bit number or no parity if \< 0
   bool   invert;
   frame_name frame_path;
   std::string compression;
   std::string config_file;
   std::string config_version;
   uint32_t    config_crc32;
   std::vector<std::string> copy_channels;
   std::vector<std::string> copy_history;
   std::vector<DQ_bit*> flag_vect;
   FrWriter fr_writer;
   Time     output_start;
   Time     current;
   
   //-----------------------------------  define the flag map data type
   class dq_bitmask {
   public:
      dq_bitmask(void);
      ~dq_bitmask();

      /**  Add the numbered but to the enable mask (i.e. the mask of all
        *  bits set by the defined dq flags.
	*/
      void enable(int bitno);

      /**  Preset the bit time series to all trigger bits OK. The evaluated 
        *  flags are then "OR"ed with the bit-masks.
	*  \brief Preset the time series.
	*  \param t0 Start time for the current stride.
	*  \param dT flag sample interval
	*  \param N  Number of time-series samples in this stride.
	*/
      void preset(const Time& t0, Interval dT, size_t N);

      /**  Return a constant reference to the time series for this channel.
        *  \brief DQ channel time series reference.
        */
      const TSeries& ref_series(void) const {return _series;}

      /**  Return a constant reference to the time series for this channel.
        *  \brief DQ channel time series reference.
        */
      TSeries& ref_series(void) {return _series;}
   private:
      TSeries  _series;
      uint32_t _enable_mask;
   };
   typedef std::map<std::string, dq_bitmask> flag_map_type;
   typedef flag_map_type::iterator flag_map_iter;
   typedef flag_map_type::const_iterator const_flag_map_iter;
   flag_map_type flag_map;
   size_t   frames_out;
};

//======================================  Inline methods
inline void
DQ_Module::dq_bitmask::enable(int bitno) {
   _enable_mask |= (1<<bitno);
}



#endif
