/* -*- mode: c++; c-basic-offset: 4; indent-tabs-mode: nil -*- */
/*
 * Calibration Lines Monitor.
 *
 * Jordi Burguet-Castell.
 * January 2010.
 */

// $Id: callineMon.cc 7899 2017-06-16 13:44:56Z ed.maros@LIGO.ORG $

#include "callineMon.hh"


#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/callineMon/callineMon.cc 7899 2017-06-16 13:44:56Z ed.maros@LIGO.ORG $"
#define PIDTITLE  "Generate gamma factors for calibration lines"
#include "ProcIdent.hh"

#include "ParseLine.hh"

#include "Dacc.hh"
#include <iostream>
#include <string>
#include <vector>
#include <cstdlib>
#include <complex>
#include <fftw3.h>


using namespace std;


// Macro for log output.
#define LOG(LEVEL) if (Debug() >= LEVEL) cout << "callineMon: "

// Macro for common checks.
#define CHECK(COND)                                                     \
    do {                                                                \
        if (COND != 0) {                                                \
            cerr << "Error in " << __FILE__ << ":" << __LINE__ << endl; \
            finish();  return;                                          \
        }                                                               \
    } while (0)


// Generate main routine.
EXECDAT(callineMon)


/*
 * Constructor.
 */
callineMon::callineMon(int argc, const char *argv[])
  : DatEnv(argc, argv), mStride(60.0), mNGammaFT(60), mDt(1.0), mAlpha(0.206),
    mDoSpectrum(false), mLastWasZero(true), mWriteFrames(false),
    mDCTRLChanName("Xn:LSC-DARM_CTRL"),
    mEXCChanName("Xn:LSC-DARM_CTRL_EXC_DAQ"),
    mSVChanName("Xn:IFO-SV_STATE_VECTOR"),
    mOutputFileName("calline-%s-60.gwf")
{
    LOG(5) << "Constructor called\n";

    int err = 0;  /* error code */
    err = initialize_input_data(argc, argv);
    CHECK(err);

    // Specify DMT Viewer channel names
    setServerName("callineMon");
    for (size_t i = 0; i < mGammaChanNames.size(); i++)
        serveData((mGammaChanNames[i]+"_mean").c_str(), mHistory[i], 0);
    if (mDoSpectrum)
        for (size_t i = 0; i < mPowChanNames.size(); i++) {
            serveData(mPowChanNames[i].c_str(), mPow[i], 0);
            serveData(mPowAvgChanNames[i].c_str(), mPowAvg[i], 0);
        }

    // Set the trender.
    mTrend.setName("callineMon");    // trend frame name
    mTrend.setFrameCount(1);         // frames per file
    mTrend.setType(Trend::kMinute);  // trend type (minute trends)
    mTrend.setAutoUpdate(false);     // avoid automatic screwup
    for (size_t i = 0; i < mGammaChanNames.size(); i++)
        mTrend.addChannel((mGammaChanNames[i]+"_mean").c_str());

    // Set the data accessor.
    getDacc().setStride(mStride);  // we'll read chunks of mStride seconds

    getDacc().addChannel(mDCTRLChanName.c_str());  // add DARM_CTRL
    getDacc().addChannel(mEXCChanName.c_str());    // add DARM_CTRL_EXC_DAQ
    getDacc().addChannel(mSVChanName.c_str());     // add STATE_VECTOR

    // Trend dataviewer index file to generate.
    string filename;
    const char* dmtout = getenv("DMTOUTPUT");
    if (dmtout) {
        filename = dmtout;
        filename += "/";
    }
    filename += "channel.cfg";
    mTrend.writeIndex(filename.c_str());
}



/*
 * Destructor.
 */
callineMon::~callineMon()
{
    LOG(5) << "Destructor called\n";

    for (size_t i = 0; i < mHistory.size(); i++)
        delete mHistory[i];

    if (mDoSpectrum)
        for (size_t i = 0; i < mPowChanNames.size(); i++) {
            if (mPow[i] != NULL)
                delete mPow[i];
            if (mPowAvg[i] != NULL)
                delete mPowAvg[i];
        }
}



/*
 * Handle messages.
 */
void callineMon::Attention(void)
{
    LOG(5) << "Attention\n";

    MonServer::Attention();
}




/* ------------------------- Main functions ---------------------------- */


/*
 * Process one data stride. This function gets called each time a new
 * frame is ready to be processed.
 */
void callineMon::ProcessData()
{
    LOG(4) << "Process data\n";

    vector<TSeries*> gammas;  // will contain the gammas
    vector<FSpectrum*> powers;  // will contain the power spectra

    TSeries* dctrl = getDacc().refData(mDCTRLChanName.c_str());
    TSeries* exc   = getDacc().refData(mEXCChanName.c_str());
    TSeries* sv    = getDacc().refData(mSVChanName.c_str());

    /* See if we are UP */
    bool sv_up = true;
    {  // new block to avoid polluting variable names
        float* s = (float*) sv->refData();
        int n = sv->getNSample();
        for (int i = 0; i < n; i++)
            if ((int(s[i]) & (1 << 2)) == 0)
                sv_up = false;
    }

    /* Reserve memory for DARM_CTRL and the excitation channels */
    int ntotal = dctrl->getNSample();
    float* dc  = new float[ntotal];     // DARM_CTRL
    float* l   = new float[ntotal];     // DARM_CTRL_EXC_DAQ

    /* Extract data from the DMT objects into the "dc" and "l" arrays */
    dctrl->getData(ntotal, dc);
    exc->getData(ntotal, l);

    // For convenience, get the initial time and duration of the interval
    const Time& t0 = dctrl->getStartTime();
    const double t_total = dctrl->getInterval().GetSecs();

    // number of samples in the gamma channel
    int ngamma = int(t_total / mDt + 0.5);

    if (mDoSpectrum && ngamma < mNGammaFT) {
        LOG(1) << "WARNING: Not enough data to compute the power spectra. "
               << mNGammaFT << " points requested, but only "
               << ngamma << " available (use a longer -stride or a "
               << "smaller -ngamma-ft to compute it)\n";
        mDoSpectrum = false;
    }

    int naverage = ntotal / ngamma;  // number of samples in the average
    
    /* Create Hann window */
    const double pi = 4*atan(1);
    const complex<double> I(0, 1);  // imaginary unit
    double* hann = new double[naverage];  // Hann window
    for (int i = 0; i < naverage; i++)
        hann[i] = 0.5 * ( 1 - cos(2*pi*i/(naverage-1)) );

    /* Get the gammas for all the frequencies */
    for (size_t ifreq = 0; ifreq < mFreqs.size(); ifreq++) {
        LOG(5) << ifreq << " - Process freq " << mFreqs[ifreq] << endl;

        /* Allocate memory for the gamma timeseries */
        float* GAMMA_RE = new float[ngamma];
        float* GAMMA_IM = new float[ngamma];

        /* Prepare for Fourier Transforming data */
        double dt = exc->getTStep().GetSecs();

        complex<double> z = exp(-2*pi * I * mFreqs[ifreq] * dt);


        /* Main loop: window the data, fourier-transform it, compute the
         * gamma factors */
        LOG(5) << "Fourier transform and compute gammas\n";
        for (int i = 0; i < ngamma; i++) {
            /* Window the data */
            for (int j = 0; j < naverage; j++) {
                dc[i*naverage + j] *= 2.0 * hann[j];
                l[i*naverage + j]  *= 2.0 * hann[j];
            }

            /* Fourier transform it */
            complex<double> ft_dc = 0.0;  // fourier transform of DARM_CTRL
            complex<double> ft_l  = 0.0;  // fourier transform of EXC

            complex<double> acc = 1;  // accumulator

            for (int j = 0; j < naverage; j++) {
                ft_dc += acc * (double)dc[i*naverage + j];
                ft_l  += acc * (double)l[i*naverage + j];
                acc *= z;
            }

            /* Next things I am copying from ComputeCalibrationFactors.c
             * without understanding... */
            ft_dc *= dctrl->getTStep().GetSecs();
            ft_dc *= mW0[ifreq];  // dewhitening...

            ft_l *= exc->getTStep().GetSecs();
            /* Ok, that was bad. TODO: comment properly what's going on. */

            /* Get the gamma factor and put it into the real and imag arrays */
            complex<double> gamma = (ft_l / ft_dc - 1.0) / mG0[ifreq];

            GAMMA_RE[i] = gamma.real();
            GAMMA_IM[i] = gamma.imag();

            // Actually, put them to 0 if we are not UP
            if (! sv_up) {
                GAMMA_RE[i] = 0.0;
                GAMMA_IM[i] = 0.0;
            }
        }

        LOG(5) << "Create the gamma TSeries\n";
        gammas.push_back(new TSeries(t0, t_total/ngamma, ngamma, GAMMA_RE));
        gammas.push_back(new TSeries(t0, t_total/ngamma, ngamma, GAMMA_IM));
        
        /* Also record the power spectra (normally every 1 min) */
        if (mDoSpectrum) {
            powers.push_back(compute_spectrum(GAMMA_RE, GAMMA_IM, t0));
        }

        delete[] GAMMA_IM;
        delete[] GAMMA_RE;

        /* For trends */
        LOG(4) << "Make trends\n";
        mTrend.trendData((mGammaChanNames[2*ifreq]+"_mean").c_str(),
                         t0, gammas[2*ifreq]->getAverage());
        mTrend.trendData((mGammaChanNames[2*ifreq+1]+"_mean").c_str(),
                         t0, gammas[2*ifreq+1]->getAverage());
    }


    /* For visualization */
    LOG(4) << "Visualize\n";
    for (size_t i = 0; i < mGammaChanNames.size(); i++) {
        float mean = gammas[i]->getAverage();
        mHistory[i]->fixedAppend(t0, t_total, &mean);
    }

    if (mDoSpectrum)
        for (size_t i = 0; i < mPowChanNames.size(); i++) {
            /* Instantaneous spectrum */
            if (mPow[i] != NULL)
                delete mPow[i];
            mPow[i] = new FSpectrum(*powers[i]);

            /* Exponentially averaged spectrum, a better PSD estimation */
            if (mLastWasZero) {
                if (mPowAvg[i] != NULL)
                    delete mPowAvg[i];
                mPowAvg[i] = new FSpectrum(*powers[i]);  // start with current
            }
            else {
                // mPowAvg[i] <-- mAlpha * mPowAvg[i] + (1-mAlpha) * mPow[i];
                int n = mPowAvg[i]->getNStep()+1;  // for short notation
                float* current = mPow[i]->refData();
                float* old_avg = mPowAvg[i]->refData();
                float* avg = new float[n];  // array that will have the new avg
                for (int m = 0; m < n; m++)
                    avg[m] = mAlpha * current[m] + (1-mAlpha) * old_avg[m];
                mPowAvg[i]->setData(n, avg);
                delete[] avg;
            }
        }

    /* Write the frames */
    if (mWriteFrames) {
        write_frames(gammas, mPow, mPowAvg);
    }

    /* Update the trends */
    mTrend.Update();

    /* Leave a clean memory */
    LOG(5) << "Clean memory\n";
    delete[] hann;
    delete[] l;
    delete[] dc;

    for (size_t i = 0; i < gammas.size(); i++)
        delete gammas[i];

    for (size_t i = 0; i < powers.size(); i++)
        delete powers[i];

    mLastWasZero = ! sv_up;  // if not UP now, we will have all zeros
}



/*
 * Return the power spectrum of the GAMMA_RE+i*GAMMA_IM time series.
 */
FSpectrum* callineMon::compute_spectrum(float* GAMMA_RE, float* GAMMA_IM,
                                        const Time& t0)
{
    LOG(5) << "Compute spectrum\n";

    if (mNGammaFT % 2 != 0) {
        LOG(1) << "Warning: number of gammas used for the spectrum not even. "
               << "I will use " << mNGammaFT - 1 << " instead." << endl;
        mNGammaFT -= 1;
    }

    fftw_complex* in;
    fftw_complex* out;
    fftw_plan p;
    
    in = (fftw_complex*) fftw_malloc(sizeof(fftw_complex) * mNGammaFT);
    out = (fftw_complex*) fftw_malloc(sizeof(fftw_complex) * mNGammaFT);
    
    p = fftw_plan_dft_1d(mNGammaFT, in, out, FFTW_FORWARD, FFTW_ESTIMATE);
    
    /* Fill the input for the FFT */
    for (int i = 0; i < mNGammaFT; i++) {
        in[i][0] = GAMMA_RE[i];
        in[i][1] = GAMMA_IM[i];
    }
    
    fftw_execute(p);  // FFT

    int n = mNGammaFT;  float n2 = n*n;  // short notation
    float* POWER = new float[n/2+1];
    POWER[0]   = (out[0][0]   * out[0][0]   + out[0][1]   * out[0][1])   / n2;
    POWER[n/2] = (out[n/2][0] * out[n/2][0] + out[n/2][1] * out[n/2][1]) / n2;
    for (int i = 1; i < n/2; i++) {  // watch out: starts at 1
        float x = 0.5 * (out[i][0] + out[n-i][0]);  // mean of z[i] and z[n-1]*
        float y = 0.5 * (out[i][1] - out[n-i][1]);  // taking the conjugate
        // this "mean" is equivalent to FT only the real part of the gammas
        
        POWER[i] = 2 * (x*x + y*y) / n2;
        // see http://www.mathworks.com/support/tech-notes/1700/1702.html
    }
    
    FSpectrum* spectrum = new FSpectrum(0, 1.0/(mDt*n), t0,
                                        mDt*n, n/2+1, POWER);
    // for some strange reason, it should be n/2+3 for the graphs to
    // appear in
    // http://marble.ligo-wa.caltech.edu:9991/
    fftw_destroy_plan(p);
    fftw_free(in);
    fftw_free(out);
    delete[] POWER;

    return spectrum;
}




/* -------------------------- Utility functions -------------------------- */


/*
 * Write the frames to disk.
 */
int callineMon::write_frames(vector<TSeries*> gammas,
                             vector<FSpectrum*> powers,
                             vector<FSpectrum*> powavgs)
{
    LOG(3) << "Write frame\n";

    mFrameWriter.buildFrame(gammas[0]->getStartTime(), Interval(mStride));

    // Gamma real and imaginary channels, and power channels if available
    for (size_t i = 0; i < mFreqs.size(); i++) {
        mFrameWriter.addProcSeries(mGammaChanNames[2*i].c_str(), *gammas[2*i]);
        mFrameWriter.addProcSeries(mGammaChanNames[2*i+1].c_str(), *gammas[2*i+1]);
        if (powers.size() > 0) {
            mFrameWriter.addSpectrum(mPowChanNames[i].c_str(), *powers[i]);
            mFrameWriter.addSpectrum(mPowAvgChanNames[i].c_str(), *powavgs[i]);
        }
    }

    // History (meta-info in the header)
    mFrameWriter.addHistory("callineMon", Now(),
                            "callineMon: calibration lines factors monitor");
    mFrameWriter.addHistory("callineMon", Now(), mHeader);
    mFrameWriter.addWriterHistory();

    char fname[256];  // buffer for the file name
    TimeStr(gammas[0]->getStartTime(),
            fname, mOutputFileName.c_str());  // allow a nice name for the file

    mFrameWriter.open(fname, true);
    mFrameWriter.writeFrame();
    mFrameWriter.close();

    return 0;
}



/*
 * Set the names of all channels to their default values with the name
 * of the interferometer as its initial part.
 */
void callineMon::set_channel_names(string ifo)
{
    LOG(5) << "Set channel names\n";

    /* Check that the IFO name is 2 letters long */
    if (ifo.length() != 2) {
        cerr << "Warning: IFO name (" << ifo << ") doesn't have "
             <<" just 2 characters! I won't change the channel names.\n";
        return;
    }

    /* Change name of input channels */
    mDCTRLChanName = ifo + mDCTRLChanName.substr(2);
    mEXCChanName   = ifo + mEXCChanName.substr(2);
    mSVChanName    = ifo + mSVChanName.substr(2);

    /* Set name of output channels */
    for (size_t i = 0; i < mFreqs.size(); i++) {
        char buffer[1000];
        snprintf(buffer, sizeof(buffer), ":DMT-CLIN_GAMMA_%.1f", mFreqs[i]);
        mGammaChanNames.push_back(ifo + buffer + "_Re");
        mGammaChanNames.push_back(ifo + buffer + "_Im");
        snprintf(buffer, sizeof(buffer), ":DMT-CLIN_POWER_%.1f", mFreqs[i]);
        mPowChanNames.push_back(ifo + buffer);
        snprintf(buffer, sizeof(buffer), ":DMT-CLIN_POWER_AVG_%.1f", mFreqs[i]);
        mPowAvgChanNames.push_back(ifo + buffer);
    }
}



/*
 * Add a comma-separated list of numbers to the vector v.
 */
void callineMon::append_nums(vector<double>& v, string nums,
                             const char* separators)
{
    int last_pos = -1;  // last position
    while (true) {
        // position of the next separator
        size_t new_pos = nums.find_first_of(separators, last_pos+1);
        if (new_pos == string::npos) {
            v.push_back(strtod(nums.substr(last_pos+1).c_str(), 0));
            break;
        }
        v.push_back(strtod(nums.substr(last_pos+1,
                                       new_pos-last_pos).c_str(), 0));
        last_pos = new_pos;
    }
}




/* ------------------ Initialization, allocation et al ------------------- */


/*
 * Read command line arguments and initialize internal variables from
 * it. Set all the non-changing input variables (arguments passed to
 * the program). To be called in the constructor.
 */
int callineMon::initialize_input_data(int argc, const char *argv[])
{
    LOG(5) << "Read input arguments\n";

    char usage[] =
"Command syntax:\n"
"callineMon [-help] [-version] [-debug <level>]"
" [-inlist <value>]"
" [-stride <value>]"
" [-ifo <value>]"
" [-write-frames]"
" [-o <value>]"
" [-freqs <values>]"
" [-olg-re <values>] [-olg-im <values>]"
" [-whitener-re <values>] [-whitener-im <values>]"
" [-factors-time <time>]"
" [-spectrum]"
" [-ngamma-ft <value>]"
" [-alpha <value>]"
" [<DatEnv-args>]\n";

    char help[] =
"Arguments are:\n"
"  -help          FLAG      This message.\n"
"  -version       FLAG      Show version number and exit.\n"
"  -debug         INT       Debug level.\n"
"  -inlist        STRING    File with the paths to the input frame files.\n"
"  -stride        FLOAT     Length in seconds of the data stride read from "
                           "the DMT.\n"
"  -ifo           STRING    Interferometer name (eg, H1).\n"
"  -write-frames  FLAG      Write all gamma factors in an output frame file.\n"
"  -o             STRING    Output file name (eg, cal_freqlines-%s-60.gwf).\n"
"  -freqs         FLOAT(s)  List of calibration line frequencies in Hz. "
                           "(eg, 46.7,393.1,1144.3) \n"
"  -olg-re        FLOAT(s)  List of real parts of the open loop gain at the "
                           "calibration lines frequencies.\n"
"  -olg-im        FLOAT(s)  List of imaginary parts of the open loop gain at "
                           "the calibration lines frequencies.\n"
"  -whitener-re   FLOAT(s)  List of real parts of the whitening filter at the "
                           "calibration lines frequencies.\n"
"  -whitener-im   FLOAT(s)  List of imaginary parts of the whitening filter "
                           "at the calibration lines frequencies.\n"
"  -factors-time  FLOAT     Factors integration time in seconds.\n"
"  -spectrum      FLAG      Compute the PSDs of the gamma factors.\n"
"  -ngamma-ft     INT       Number of gamma samples used in the spectra.\n"
"  -alpha         FLOAT     Weight of the last spectrum in the exponentially "
                            "smoothed estimation of the PSD.\n";

    /* Define the next vectors for convenience: we will fill a complex
     * mG0 and mW0 later with their values. */
    vector<double> mG0_re, mG0_im;
    vector<double> mW0_re, mW0_im;

    /* Temporary string with ifo name, to use in channel names later */
    const char* ifo = 0;

    /* Read arguments */
    for (int i = 1; i < argc; i++) {
        string s = argv[i];

        if (s == "-help") {
            cout << usage << endl << help;
            finish();  return 0;
        }
        else if (s == "-version") {
            cout << argv[0] << " version 0.98" << endl;
            finish();  return 0;
        }
        else if (s == "-stride")       { mStride = strtod(argv[++i], 0); }
        else if (s == "-ifo")          { ifo = argv[++i]; }
        else if (s == "-write-frames") { mWriteFrames = true; }
        else if (s == "-o")            { mOutputFileName = argv[++i]; }
        else if (s == "-freqs")        { append_nums(mFreqs, argv[++i]); }
        else if (s == "-olg-re")       { append_nums(mG0_re, argv[++i]); }
        else if (s == "-olg-im")       { append_nums(mG0_im, argv[++i]); }
        else if (s == "-whitener-re")  { append_nums(mW0_re, argv[++i]); }
        else if (s == "-whitener-im")  { append_nums(mW0_im, argv[++i]); }
        else if (s == "-factors-time") { mDt = strtod(argv[++i], 0); }
        else if (s == "-spectrum")     { mDoSpectrum = true; }
        else if (s == "-ngamma-ft")    { mNGammaFT = atoi(argv[++i]); }
        else if (s == "-alpha")        { mAlpha = strtod(argv[++i], 0); }
        else if (isDatEnvArg(argv[i])) {
            i++;
        }
        else {
            cerr << "Unrecognized argument: " << s << endl
                 << usage;
            return -3;
        }
    }

    /* Check that all the needed arguments are there */
    string needed_args[] = {"-ifo", "-freqs", "-olg-re", "-olg-im",
                            "-whitener-re", "-whitener-im"};

    bool missing_arg = false;
    int iarg;
    for (iarg = 0; iarg < 6; iarg++) {
        bool current_arg_present = false;
        for (int j = 0; j < argc; j++) {
            if (needed_args[iarg] == argv[j]) {
                current_arg_present = true;
                break;
            }
        }
        if (! current_arg_present) {
            missing_arg = true;
            break;
        }
    }

    if (missing_arg) {
        cerr << "Missing necessary argument: " << needed_args[iarg] << endl
             << usage;
        return -2;
    }

    /* Check that the number of elements is consistent */
    if (mFreqs.size() != mG0_re.size() || mFreqs.size() != mG0_im.size() ||
        mFreqs.size() != mW0_re.size() || mFreqs.size() != mW0_im.size()) {
        cerr << "Number of gains and whitening points must correspond to "
             << "the number of frequencies.\n";
        return -4;
    }
    
    /* Fill complex mG0 and mW0 with real and imaginary parts */
    for (size_t i = 0; i < mFreqs.size(); i++) {
        mG0.push_back(complex<double>(mG0_re[i], mG0_im[i]));
        mW0.push_back(complex<double>(mW0_re[i], mW0_im[i]));
    }

    /* Set the name of all the channels */
    set_channel_names(ifo);

    /* Set the histories (for the DMT Viewer) */
    for (size_t i = 0; i < mGammaChanNames.size(); i++)
        mHistory.push_back(new FixedLenTS(43200));

    /* Put placeholeders for the spectra and their averages */
    if (mDoSpectrum)
        for (size_t i = 0; i < mPowChanNames.size(); i++) {
            mPow.push_back(new FSpectrum());
            mPowAvg.push_back(new FSpectrum());
        }

    /* Get command line and all parameters as a string for the header
     * of the frame file */
    mHeader += "\nCommand line run: ";
    for (int i = 0; i < argc; i++) {
        mHeader += argv[i];
        mHeader += " ";
    }

    setStrideAlignment(long(mStride), 0.0);

    return 0;
}
