/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef APERTURE_HH
#define APERTURE_HH
#include "DatEnv.hh"
#include "OperStateCondList.hh"
#include "TSeries.hh"
#include <vector>
#include <string>
#include <iosfwd>

namespace xsil {
   class MetaIO;
}

class channels {
public:
   //-----------------------------------  1d statistics structure
   struct chan_stat {
      chan_stat(size_t id);
      bool operator< (const chan_stat& rhs) const;
      size_t channel_id;
      size_t min_index;
      size_t trigs_found;
      double trigs_expected;
      double min_likelihood;
  };

public:
   channels(const std::string& chan, size_t N=32768);
   ~channels(void);
   void append(const TSeries& ts);
   std::string bin_id(size_t id) const; 
   bool empty(void) const;
   const std::string& name(void) const;
   const TSeries& ref_values(void) const;
   const TSeries& ref_rank(void) const;
   int  reject(void) const;
   void set_limits(size_t M=16);
   void set_reject(int rej);
   bool test1D(const TSeries& ts, double CL, chan_stat& s);

private:
   size_t      _alloc;
   std::string _name;
   TSeries     _value;
   std::vector<double> _limits;
   TSeries     _rank;
   int         _reject;
};

inline bool 
channels::empty(void) const {
   return _value.empty();
}

inline int
channels::reject(void) const {
   return _reject;
}



class aperture : public DatEnv {
private:

   //-----------------------------------   Trigger selection structure.
   struct trig_select {
      void critter(std::ostream& out) const;
      std::string field;
      std::string string;
      double      number;
      enum  {
	 s_pass,
	 s_string,
	 s_num_ge,
	 s_num_lt
      } type;
   };

   //-----------------------------------  Trigger veto structure
   struct trig_veto {
      std::string channel;
      double      val_min;
      double      val_max;
   };

   //-----------------------------------  2d statistics structure
   struct stat_2d {
      stat_2d(size_t i, size_t j);
      bool operator< (const stat_2d& rhs) const;
      size_t channel_a;
      size_t channel_b;
      size_t min_index;
      size_t trigs_found;
      double trigs_expected;
      double correlation;
      double min_likelihood;
   };
   

public:
   typedef Time::ulong_t gps_t;
public:
   aperture(int argc, const char* argv[]);
   ~aperture(void);
   void configure(const std::string& conf);
   void calculate(void);
   void calc_matrices(size_t i, size_t j, size_t M, int* trig,
		      int* counts) const;
   void calc_stats(size_t M, const int* trig, const int* counts,
		   stat_2d& stat) const;
   void make_page(void) const;
   void ProcessData(void);
   void read_triggers(const std::string& file, gps_t start, gps_t stop);
   bool select_trigger(const xsil::MetaIO& mio) const;
   std::string select_list(void) const;
   void veto_triggers(void);

private:
   std::string _config;
   double      _logCL;
   double      _logCL1D;
   double      _corr_limit;
   
   //-------------------------------------  OSC parameters
   std::string _oscfile;
   std::string _osc_cond;

   //-------------------------------------  Trigger file and selection
   std::string _trig_path;
   std::string _trig_table_name;
   std::string _trig_time_field;
   std::string _trig_time_ns_field;

   //-------------------------------------  html parameters
   std::string _html_directory;

   //-------------------------------------  Trigger data
   std::vector<trig_select> _trig_select_list;
   std::vector<trig_veto> _trig_veto_list;
   TSeries     _trig_series;
   size_t      _trig_total;
   size_t      _nEnt;
   double      _trigAvg;
   
   //------------------------------------  Channels
   typedef std::vector<channels> chan_vect;
   chan_vect   _channel;
   OperStateCondList _osc;

   //------------------------------------  Statistics vector
   typedef std::vector<channels::chan_stat> stat1dvec_type;
   typedef stat1dvec_type::iterator stat1dvec_iter;
   stat1dvec_type _stat1d_vector;

   //------------------------------------  Statistics vector
   typedef std::vector<stat_2d> statvec_type;
   typedef statvec_type::iterator statvec_iter;
   statvec_type _stat_vector;
};

//======================================  inline methods
const std::string&
channels::name(void) const {
   return _name;
}

const TSeries&
channels::ref_values(void) const {
   return _value;
}

const TSeries&
channels::ref_rank(void) const {
   return _rank;
}

aperture::stat_2d::stat_2d(size_t i, size_t j)
   : channel_a(i), channel_b(j), min_index(0), correlation(0),
     min_likelihood(0)
{}

bool 
aperture::stat_2d::operator< (const stat_2d& rhs) const {
   //   return min_likelihood < rhs.min_likelihood;
   return trigs_found > rhs.trigs_found;
}

channels::chan_stat::chan_stat(size_t i)
   : channel_id(i), min_index(0), min_likelihood(0)
{}

bool 
channels::chan_stat::operator< (const chan_stat& rhs) const {
   //   return min_likelihood < rhs.min_likelihood;
   return trigs_found > rhs.trigs_found;
}

#endif // !defined(APERTURE_HH)
