/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "aperture.hh"
#include "lcl_array.hh"
#include "DVecType.hh"
#include "framedir.hh"
#include "jsonStack.hh"
#include "xsil/MetaIO.hh"
#include "GDSPlot.hh"
#include "html/document.hh"
#include "html/image.hh"
#include "html/Table2C.hh"
#include "html/writer.hh"
#include "TCanvas.h"
#include <cmath>
#include <cstring>
#include <algorithm>
#include <iostream>
#include <sstream>

using namespace std;

EXECDAT(aperture)

//======================================  constructor
aperture::aperture(int argc, const char* argv[]) 
: DatEnv(argc, argv), _logCL(0), _logCL1D(0), _corr_limit(0.5), _trig_total(0), 
   _osc(getDacc())
{
   _config = argv[argc-1];
   configure(_config);
   getDacc().setStride(16.0);
}

//======================================  Destructor
aperture::~aperture(void) {
   cout << "Starting post-processing" << endl;
   gps_t start = _channel[0].ref_values().getStartTime().getS();
   gps_t stop  = _channel[0].ref_values().getEndTime().getS();
   Interval dt = _channel[0].ref_values().getTStep();
   size_t nBin = double(stop - start)/double(dt);
   cout << "reading triggers from: " << _trig_path 
        << " gps range: " << start << "-" << stop << endl;
   unique_ptr<DVectI> dvi(new DVectI(nBin));
   dvi->replace_with_zeros(0, nBin, nBin);
   _trig_series.setData(Time(start), dt, dvi.release());
   cout << "Trigger selection criteria: " << endl;
   for (size_t i=0; i<_trig_select_list.size(); i++) {
      _trig_select_list[i].critter(cout);
   }
   read_triggers(_trig_path, start, stop);
   cout << "Vetoing triggers" << endl;
   veto_triggers();
   cout << "Starting channel pair test" << endl;
   calculate();
}

//======================================  Poisson function
//  this function returns the probability that a randomly (poisson) distributed
//  sample with mean x will have >=i events.
static double
poissonge(double x, int i) {
   double res = 1.0;
   if (i > 0) {
      double val = exp(-x);
      res -= val;
      for (size_t n=1; n<i; n++) {
	 val *= x / double(n);
	 res -= val;
      }
   }
   return res;
}

//======================================  Calculate everything.
void
aperture::calculate(void) {
   //-----------------------------------  Get limits perform 1D tests
   size_t N = _channel.size();
   for (size_t i=0; i<N; i++) {
      if (_channel[i].empty()) {
	 cout << _channel[i].name() << " is empty!" << endl;
	 _channel[i].set_reject(1);
         continue;
      }
      _channel[i].set_limits();

      //--------------------------------  Find worst entry
      channels::chan_stat stat(i);
      bool fail = _channel[i].test1D(_trig_series, _logCL1D, stat);
      if (fail) {
         _stat1d_vector.push_back(stat);
      }
   }
   std::sort(_stat1d_vector.begin(), _stat1d_vector.end());

   //-----------------------------------  get the expected number per cell.
   size_t M=16;

   cout << "start 2D tests..." << endl;
   lcl_array<int> kArray(M*M);
   lcl_array<int> tArray(M*M);
   for (size_t i=1; i<N; i++) {
      if (_channel[i].reject()) continue;
      DVectI dvi(*_channel[i].ref_rank().refDVect());
      _nEnt = dvi.size();
      _trigAvg = double(_trig_total) / double(_nEnt);
      for (size_t j=0; j<i; j++) {
         if (_channel[j].reject()) continue;
	 //cout << "computing {" << _channel[i].name() << ", "
	 //     << _channel[j].name() << "}" << endl;

	 //-----------------------------  calculate matrices
	 calc_matrices(i, j, M, tArray.get(), kArray.get());
	 
	 //-----------------------------  flag stats
	 stat_2d stat(i, j);
	 calc_stats(M, tArray.get(), kArray.get(), stat);

         if (stat.correlation < _corr_limit && stat.min_likelihood < _logCL) {
	    size_t savij = stat.min_index;
            size_t kj = savij % M;
            size_t ki = savij / M;
            cout << _channel[i].name() << "[" << ki << "] " 
                 << _channel[j].name() << "[" << kj << "] N="
		 << tArray[savij]
	         << " expect=" << kArray[savij]*_trigAvg 
                 << " pois=" << stat.min_likelihood << endl;
	    cout << "correlation = " << stat.correlation << endl;
	    _stat_vector.push_back(stat);
         }
      }
   }

   //------------------------------------  sort results
   std::sort(_stat_vector.begin(), _stat_vector.end());
   lcl_array<bool> used(N);
   for (size_t i=0; i<N; i++) used[i] = false;
   size_t inxok = 0;
   for (size_t i=0; i<_stat_vector.size(); i++) {
      size_t inxa = _stat_vector[i].channel_a;
      size_t inxb = _stat_vector[i].channel_b;
      if (used[inxa] || used[inxb]) continue;
      if (i != inxok)  _stat_vector[inxok] =  _stat_vector[i];
      inxok++;
      used[inxa] = true;
      used[inxb] = true;
   }
   make_page();
}

//======================================  Formatting utility
inline string
CLstring(double logCL) {
   ostringstream msg;
   msg << logCL << " (log " << exp(logCL) << ")";
   return msg.str();
}

//======================================  Read the configuration file
void
aperture::configure(const std::string& cfile) {
   jsonStack js(cfile);
   double CL = 0, CL1D = 0;

   //-----------------------------------  Numeric parameters
   js.fetch_data("confidence", CL);
   js.fetch_data("log-confidence", _logCL);
   js.fetch_data("confidence-1d", CL1D);
   js.fetch_data("log-confidence-1d", _logCL1D);
   js.fetch_data("correlation-limit", _corr_limit);

   //-----------------------------------  OSC info
   js.fetch_data("osc-file", _oscfile);
   js.fetch_data("osc-cond", _osc_cond);

   //-----------------------------------  html directory
   js.fetch_data("html-directory", _html_directory);   

   //-----------------------------------  Trigger selections.
   js.fetch_data("trigger-path", _trig_path);
   js.fetch_data("trigger-table-name", _trig_table_name);
   js.fetch_data("trigger-time-field", _trig_time_field);
   js.fetch_data("trigger-time-ns-field", _trig_time_ns_field);
   if (js.push_element(string("trigger-select"))) {
      while (js.iterate()) {
         trig_select ts;
         js.fetch_data("field", ts.field);
         if (js.fetch_data("string-eq", ts.string)) {
	    ts.type = trig_select::s_string;
         } else if (js.fetch_data("num-ge", ts.number)) {
	    ts.type = trig_select::s_num_ge;
         } else if (js.fetch_data("num-lt", ts.number)) {
	    ts.type = trig_select::s_num_lt;
         } else {
	    ts.type = trig_select::s_pass;
         }
         _trig_select_list.push_back(ts);
         js.pop();
      }
      js.pop();
   }

   if (js.push_element(string("vetos"))) {
      while (js.iterate()) {
         trig_veto tv;
	 tv.val_min = -1.0/0.0;
	 tv.val_max =  1.0/0.0;
         js.fetch_data("channel", tv.channel);
         js.fetch_data("val-min", tv.val_min);
         js.fetch_data("val-max", tv.val_max);
	 _trig_veto_list.push_back(tv);
         js.pop();
      }
      js.pop();
   }

   //-----------------------------------  Channel list
   std::vector<std::string> chan_vect;
   size_t nChan = js.fetch_data("channels", chan_vect);
   if (!nChan) throw runtime_error("aperture: Error reading channel list");
   for (size_t i=0; i<nChan; i++) {
      if (chan_vect[i].empty()) continue;
      _channel.push_back(channels(chan_vect[i]));
      getDacc().addChannel(chan_vect[i]);
   }

   //------------------------------------  Set stuff up
   if (!_oscfile.empty()) _osc.readConfig(_oscfile);
   if (_logCL == 0) {
      if (CL <= 0) CL = 0.002;
      _logCL = log(CL);
   }
   if (_logCL1D == 0) {
      if (CL1D <= 0) CL1D = 0.0002;
      _logCL1D = log(CL1D);
   }

   //------------------------------------  Print
   cout << "confidence:            "   << CLstring(_logCL) << endl;
   cout << "confidence-1d:         "   << CLstring(_logCL1D) << endl;
   cout << "html-directory:        \"" << _html_directory << "\"" << endl;
   cout << "osc-file:              \"" << _oscfile << "\"" << endl;
   cout << "osc-cond:              \"" << _osc_cond << "\"" << endl;
   cout << "trigger-path           \"" << _trig_path << "\"" << endl;
   cout << "trigger-table-name     \"" << _trig_table_name << "\"" << endl;
   cout << "trigger-time-field     \"" << _trig_time_field << "\"" << endl;
   cout << "trigger-time-ns-field  \"" << _trig_time_ns_field << "\"" << endl;
   cout << "trigger selections:    "   << _trig_select_list.size() << endl;
   cout << "trigger vetoes:        "   << _trig_veto_list.size() << endl;
   cout << "number of channels:    "   << _channel.size() << endl;
}

//======================================  Calculate matrices
void 
aperture::calc_matrices(size_t i, size_t j, size_t M,
			int* trig, int* counts) const {
   //-----------------------------------  Zero statistics arrays
   memset(counts, 0, M*M*sizeof(int));
   memset(trig,   0, M*M*sizeof(int));

   //-----------------------------------  fill matrices
   const DVectI dvi(*_channel[i].ref_rank().refDVect());
   const DVectI dvj(*_channel[j].ref_rank().refDVect());
   const DVectI& dvt = dynamic_cast<const DVectI&>(*_trig_series.refDVect());
   size_t nEnt = dvi.size();
   for (size_t k=0; k<nEnt; k++) {
      size_t ivsj = dvi[k] * M + dvj[k];
      counts[ivsj]++;
      trig[ivsj] += dvt[k];
   }
}
   
//======================================  Calculate statistics
void 
aperture::calc_stats(size_t M, const int* trig, const int* counts,
		     stat_2d& s) const {
   int savij = -1;
   double wx = 0;
   double ssq=0;
   for (size_t inxij=0; inxij<M*M; inxij++) {
      ssq += pow(counts[inxij], 2);
      double xpect = counts[inxij]*_trigAvg;
      double pois  = log(poissonge(xpect, trig[inxij]));
      if (pois < wx) {
	 wx = pois;
	 savij = inxij;
      }
   }
   size_t nEnt = _trig_series.getNSample();
   s.min_likelihood = wx;
   if (savij >= 0) {
      s.correlation = sqrt(M * ssq / (nEnt *nEnt));
      s.min_index   = savij;
      s.trigs_found = trig[savij];
      s.trigs_expected = counts[savij] * _trigAvg;
   }
}
   
//======================================  Make the html output
void
aperture::make_page(void) const {
   if (_html_directory.empty()) return;
   size_t M = 16;
   //-----------------------------------  set up bin edges.
   lcl_array<double> xybins(M+1);
   for (size_t i=0; i <= M; i++) {
      xybins[i] = i;
   }

   //------------------------------------  Start html document
   html::document ape_page("Ape Page");

   //------------------------------------  Build run parameter table
   html::Table2C* argtb = new html::Table2C("Global Parameters", 
					    "Parameter", "Value");
   ape_page.addObject(argtb);
   argtb->addRow("confidence             ", CLstring(_logCL));
   argtb->addRow("confidence-1d          ", CLstring(_logCL1D));
   argtb->addRow("html-directory         ", _html_directory);
   argtb->addRow("osc-file               ", _oscfile);
   argtb->addRow("osc-cond               ", _osc_cond);
   argtb->addRow("trigger-path           ", _trig_path);
   argtb->addRow("trigger-table-name     ", _trig_table_name);
   argtb->addRow("trigger-time-field     ", _trig_time_field);
   argtb->addRow("trigger-time-ns-field  ", _trig_time_ns_field);
   argtb->addRow("trigger selections:    ", select_list());
   argtb->addRow("trigger vetoes:        ", _trig_veto_list.size());
   argtb->addRow("number of channels:    ", _channel.size());
   argtb->addRow("start time:            ", _trig_series.getStartTime().getS());
   argtb->addRow("duration:              ", _trig_series.getInterval());
   argtb->addRow("improbable channels    ", _stat1d_vector.size());
   argtb->addRow("number triggers:       ", _trig_total);

   //-------------------------------------  Build 1d results table
   html::table* tab1d = new html::table("Rejected channels");
   ape_page.addObject(tab1d);
   tab1d->addColumn("Channel");
   tab1d->addColumn("Bin");
   tab1d->addColumn("#/expected");
   tab1d->refHeader(2).setAlign("center");
   tab1d->addColumn("log(Likelihood)");
   tab1d->refHeader(2).setAlign("right");
   for (size_t i=0; i < _stat1d_vector.size(); i++) {
      const channels::chan_stat& s(_stat1d_vector[i]);
      int id = s.channel_id;
      size_t row = tab1d->addRow();
      tab1d->insertData(row, 0, _channel[id].name());
      tab1d->insertData(row, 1, _channel[id].bin_id(s.min_index));
      ostringstream n_navg;
      n_navg << s.trigs_found << "/" << s.trigs_expected;
      tab1d->insertData(row, 2, n_navg.str());
      //tab1d->refCell(row, 2).setAlign("center");
      tab1d->insertData(row, 3, s.min_likelihood);
   }

   //------------------------------------- Build 2d results table
   html::table* tab = new html::table("Statistics");
   ape_page.addObject(tab);
   tab->addColumn("statistics");
   tab->addColumn("plots");

   //-----------------------------------  Loop over plots.
   TCanvas* canvas = new TCanvas("GDSPlot", "Ape Plots");
   GDSPlot gplot(canvas);
   gplot.set_palette("viridis");
   lcl_array<int> counts(M*M);
   lcl_array<int> trig(M*M);
   size_t N = _stat_vector.size();
   if (N > 50) N = 50;
   for (size_t i=0; i<N; i++) {
      const stat_2d& stat = _stat_vector[i];
      size_t chan_a = stat.channel_a;
      size_t chan_b = stat.channel_b;
      html::Table2C partbl("", "", "");
      partbl.addRow("channel name A    ", _channel[chan_a].name());
      partbl.addRow("channel A bin     ", 
                    _channel[chan_a].bin_id(stat.min_index/M));
      partbl.addRow("channel name B    ", _channel[chan_b].name());
      partbl.addRow("channel B bin     ", 
                    _channel[chan_b].bin_id(stat.min_index%M));
      partbl.addRow("correlation       ", stat.correlation);
      partbl.addRow("Triggers found    ", stat.trigs_found);
      partbl.addRow("Triggers expected ", stat.trigs_expected);
      partbl.addRow("log-likelihood    ", stat.min_likelihood);
      tab->addRow();
      tab->insertData(i, 0, partbl);
      gplot.new_plot();
      calc_matrices(chan_a, chan_b, M, trig, counts);
      double maxz = 0.0;
      lcl_array<double> data(M*M);
      for (size_t j=0; j<M*M; j++) {
	 double xpect = counts[j] * _trigAvg;
	 if (xpect != 0) data[j] = double(trig[j]) / xpect;
	 else            data[j] = 0;
	 if (data[j] > maxz) maxz = data[j];
      }
      gplot.set_zrange(0.0, maxz);
      gplot.xlabel(_channel[chan_a].name());
      gplot.ylabel(_channel[chan_b].name());
      gplot.surf(M, xybins, M, xybins, data);
      ostringstream plotname;
      plotname << "aperture_" << chan_a << "_" << chan_b << ".png";
      gplot.print(_html_directory+"/"+plotname.str());
      html::image img(plotname.str());
      tab->insertData(i, 1, img);
   }
   ofstream out((_html_directory + "/index.html").c_str());
   html::writer wout(out);
   ape_page.write(wout);
}

//======================================  Collect data from specified channels
void
aperture::ProcessData(void) {
   cout << "read data data at: " << getDacc().getFillTime() << endl;
   if (!_osc_cond.empty() && !_osc.satisfied(_osc_cond)) return;
   size_t N = _channel.size();
   for (size_t i=0; i<N; i++) {
      const TSeries* ts = getDacc().refData(_channel[i].name());
      if (!ts) {
         cerr << "aperture: no data for channel: " << _channel[i].name() <<endl;
         continue;
      }
      _channel[i].append(*ts);
   }
}

//=======================================  Read trigger data
void
aperture::read_triggers(const std::string& file, gps_t start, gps_t stop) {
   size_t nRead(0), nTime(0), nSelect(0);
   FrameDir fd;
   fd.add(file.c_str(), true);
   FrameDir::file_iterator endfile = fd.find(Time(stop));
   DVectI& dvi = dynamic_cast<DVectI&>(*_trig_series.refDVect()); 
   if (Debug()) cout << "first trigger file: "<< fd.find(Time(start))->getFile()
		     << " last file: " << endfile->getFile() << endl;
   for (FrameDir::file_iterator i=fd.find(Time(start)); i != endfile; ++i) {
      if (Debug()) cout << "Read trigger file: " << i->getFile() << endl;
      xsil::MetaIO mio(i->getFile().c_str(), _trig_table_name.c_str());
      if (!mio.is_open()) continue;
      while (mio.getRow()) {
	 nRead++;
	 Time::ulong_t gps_s  = mio.getInt(_trig_time_field.c_str());
	 if (gps_s < start || gps_s >= stop) continue;
         nTime++;
	 Time::ulong_t gps_ns = mio.getInt(_trig_time_ns_field.c_str());
	 if (select_trigger(mio)) {
	    nSelect++;
	    size_t ibin = _trig_series.getBin(Time(gps_s, gps_ns));
	    dvi[ibin]++;
            _trig_total++;
	 }
      }
   }
   cout << "triggers read:       " << nRead << endl;
   cout << "triggers in time:    " << nTime << endl;
   cout << "triggers selected:   " << nSelect << endl;
}

//======================================  List trigger selection criteria
std::string
aperture::select_list(void) const {
   size_t N = _trig_select_list.size();
   if (!N) return string("none");
   //-----------------------------------  Build selection string
   ostringstream crit;
   for (size_t i=0; i<N; i++) {
      const trig_select& tsi = _trig_select_list[i];
      if (tsi.type  == trig_select::s_pass) continue;
      if (i) crit << " && ";
      switch (tsi.type) {
      case trig_select::s_string:
	 crit << tsi.field << " == " << tsi.string;
	 break;
      case trig_select::s_num_ge:
	 crit << tsi.field << " >= " << tsi.number;
	 break;
      case trig_select::s_num_lt:
	 crit << tsi.field << " < " << tsi.number;
	 break;
      default:
	 break;
      }
   }
   return crit.str();
}


//=======================================  Test trigger selection criteria
bool
aperture::select_trigger(const xsil::MetaIO& mio) const {
   size_t N = _trig_select_list.size();
   for (size_t i=0; i<N; i++) {
      const trig_select& tsi = _trig_select_list[i];
      switch (tsi.type) {
      case trig_select::s_string:
	 if (mio.getString(tsi.field.c_str()) != tsi.string) return false;
	 break;
      case trig_select::s_num_ge:
	 if (mio.getFloat(tsi.field.c_str()) < tsi.number) return false;
	 break;
      case trig_select::s_num_lt:
	 if (mio.getFloat(tsi.field.c_str()) >= tsi.number) return false;
	 break;
      default:
	 break;
      }
   }
   return true;
}

//=======================================  Veto triggers
void
aperture::veto_triggers(void) {
   size_t nVetoed = 0;
   size_t Nc = _channel.size();
   size_t Nv = _trig_veto_list.size();
   cout << setprecision(3) << fixed;
   for (size_t iv=0; iv<Nv; iv++) {
      int ic = -1;
      for (size_t i=0; i<Nc; i++) {
	 if (_channel[i].name() == _trig_veto_list[iv].channel) {
	    ic = i;
	    break;
	 }
      }
      if (ic < 0) continue;
      double veto_min = _trig_veto_list[iv].val_min;
      double veto_max = _trig_veto_list[iv].val_max;
      cout << "veto triggers with " << veto_min << " <= "
	   << _trig_veto_list[iv].channel  << " < " << veto_max << endl;
      size_t jCol = 0;
      size_t Ne = _channel[ic].ref_values().getNSample();
      DVectI& trigDv(*dynamic_cast<DVectI*>(_trig_series.refDVect()));
      const TSeries& data = _channel[ic].ref_values();
      for (size_t i=0; i<Ne; i++) {
	 if (!trigDv[i]) continue;
	 double x = data.getDouble(i);
	 if (x >= veto_min && x < veto_max) {
	    if (!jCol) cout << "vetoed:";
	    cout << " " << data.getBinT(i).totalS();
	    if (++jCol == 5) {
	       cout << endl;
	       jCol = 0;
	    }
	    nVetoed += trigDv[i];
	    trigDv[i] = 0;
	 }
      }
      if (jCol) cout << endl;
   }
   cout.unsetf(ios::fixed);
   cout << "triggers vetoed:     " << nVetoed << endl;
   _trig_total -= nVetoed;
   cout << "total triggers:      " << _trig_total << endl;
}

//======================================  Print trigger selection rule
void 
aperture::trig_select::critter(std::ostream& out) const {
    out << "field: " << field;
    switch (type) {
    case s_pass:
       out << " pass";
       break;
    case s_string:
       out << " equal to \"" << string;
       break;
    case s_num_ge:
       out << " >= " << number;
       break;
    default:
       out << " undefined operation.";
    }
    out << endl;
}

//======================================  Channel constructor
channels::channels(const std::string& chan, size_t N)
   : _alloc(N), _name(chan), _reject(0)
{
}

//=======================================  Destructor
channels::~channels(void) {
}

//=======================================  print bin id and limits
std::string
channels::bin_id(size_t id) const {
   ostringstream binid;
   binid << id << " {" << _limits[id] << ", " << _limits[id+1] << "}";
   return binid.str();
} 

//=======================================  Append data
void 
channels::append(const TSeries& ts) {
   if (ts.empty()) return;
   if (_value.empty()) {
      DVector* dv = ts.refDVect()->clone();
      dv->reserve(_alloc);
      _value.setData(ts.getStartTime(), ts.getTStep(), dv);
   } else {
      _value.Append(ts);
   }
}
   
//=======================================  Set the limits
void 
channels::set_limits(size_t M) {
   DVectF cdata(*_value.refDVect());
   DVectF data = cdata;
   size_t N=data.size();
   std::sort(data.refTData(), data.refTData()+N);
   _limits.resize(M+1, 0);

   //-----------------------------------  Pick up the rank limit values.
   _limits[0] = data[0];
   _limits[M] = data[N-1];
   size_t div = N / M;
   size_t ndup = 0;
   for (size_t i=1; i<M; i++) {
      size_t inx = i * div;
      _limits[i] = data[inx];
      if (_limits[i] == data[inx-1]) ndup++;
   }
   if (ndup) {
      cout << "warning - " << ndup << " slices of channel: " << name()
           << " may share values." << endl;
      set_reject(2);
   }

   //-------------------------------------  Set up the rank Series.
   DVectI rvect(N);
   size_t rinx = 0;
   for (size_t i=0; i<N; i++) {
      float cdi = cdata[i];
      while (cdi < _limits[rinx] || cdi >= _limits[rinx+1]) {
	 if (cdi < _limits[rinx]) {
	    if (!rinx) break;
	    rinx--;
	 } else if (cdi >= _limits[rinx+1]) {
	    if (rinx == M-1) break;
	    rinx++;
	 }
      }
      rvect[i] = rinx;
   }
   _rank.setData(_value.getStartTime(), _value.getTStep(), rvect.clone());
}

//=====================================  Test the trigger rates per rank
bool
channels::test1D(const TSeries& ts, double logCL, chan_stat& stat) {
   size_t N = _rank.getNSample();
   double nTrig  = ts.refDVect()->VSum(0, N);
   double perbin = nTrig/double(N);
   size_t M = 16;
   bool   fail = false;

   //----------------------------------  Set up count arrays
   lcl_array<size_t> kval(M);
   lcl_array<size_t> tval(M);
   memset(kval.get(), 0, M*sizeof(size_t));
   memset(tval.get(), 0, M*sizeof(size_t));

   //----------------------------------  Count entries and triggers
   const DVectI& dvr(dynamic_cast<const DVectI&>(*_rank.refDVect()));
   const DVectI& dvt(dynamic_cast<const DVectI&>(*ts.refDVect()));
   for (size_t i=0; i<N; i++) {
      int k = dvr[i];
      kval[k]++;
      tval[k] += dvt[i];
   }

   //----------------------------------  Check the stats
   int wi=-1;
   double wx = logCL;
   double wav; 
   for (size_t i=0; i<M; i++) {
      double average = perbin * kval[i];
      double pois    = poissonge(average, tval[i]);
      if (pois <= 0) continue;
      pois = log(pois);
      if (pois < wx) {
         wi = i;
         wx = pois;
         wav = average;
      }
   }

   //----------------------------------  Fill the statistics structure
   stat.min_index = wi;
   stat.min_likelihood = wx;
   stat.trigs_found = tval[wi];
   stat.trigs_expected = wav;

   //----------------------------------  Tell the work if this is significant
   if (wi >= 0) {
      cout << name() << "[" << wi << "] N=" << tval[wi] 
           << " expect=" << wav << " pois=" << wx << " (log " << exp(wx)
	   << ")" << endl;
      cout << "            range " << wi << ": " << _limits[wi] << " - "
	   << _limits[wi+1] << endl;
      set_reject(3);
      fail = true;
   }
   return fail;
}

//===================================== set the reject bit
void
channels::set_reject(int rej) {
   _reject = rej;
}
