/*  File WaveMon.cc      
    author: Sergey Klimenko, University of Florida
*/

#ifndef _USE_DMT
#define _USE_DMT
#endif

//#ifndef __CINT__

//  The next three lines are needed if to generate triggers.
//  The descriptive title in PIDTITLE should the monitor function.
// #define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/WaveMon/WaveMon.cc 7987 2018-01-11 08:18:07Z john.zweizig@LIGO.ORG $"
// #define PIDTITLE  "Data monitor environment template"
// #include "ProcIdent.hh"


#include <fstream>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <iostream>

#include "TSeries.hh"
#include "Dacc.hh"
#include "Interval.hh"
#include "Time.hh"
#include "DVector.hh"

#include "WaveMon.hh"
#include "wseries.hh"
#include "Biorthogonal.hh"
#include "Symlet.hh"

#include "xsil/Xwriter.hh"
#include "xsil/xsil.hh"

EXECDAT(WaveMon)        // Generate the main routine.

  //#endif               // !def(__CINT__)

using namespace std;

//--- WaveMon constructor -------------

WaveMon::WaveMon(int argc, const char *argv[]) 
   : DatEnv(argc, argv), timeGate(-1.), pfraction(20.), timeStride(8.), 
     timeInterval(30.), resolution(32.), samplingRate(2048.), liveTime(0.), 
     totalTime(0.), asymmetryCut(0.), likelihoodCut(0.), offset(2.), minClusterSize(4), 
     nDump(0), mDump(0), maxDump(0), nBuffer(720), verbose(0), firstcall(true),
     line60Hz(0), waveorder(32), mOSC(getDacc())
{ 
   FILE* out;
   char dir[256];

   char *w;
   char offlineFile[256];
   char serverName[256];
   char htmlDir[256];
   char trendDir[256];
   char outputDir[256];
   char gps[4];

   bool configure   = false;
   bool offline     = false;

   sprintf(masterChannel,"%s", "" );
   sprintf(configFile,   "%s", "" );
   sprintf(triggerFile,  "%s", "" );
   sprintf(summaryFile,  "%s", "" );
   sprintf(backupFile,   "%s", "" );
   sprintf(trendFile,    "%s", "" );
   sprintf(indexFile,    "%s", "" );
   sprintf(htmlDir,      "%s", "" );
   sprintf(trendDir,     "%s", "" );
   sprintf(outputDir,    "%s", "" );
   sprintf(lockCondition,"%s", "" );
   sprintf(suffix,       "%s", "" );
   sprintf(gps,          "%s","%s");

   wTrigger.clear();
   

/******************************************************************
 * The LM parameters can be set either from command line or from  *
 * configuration file (option -i fileName)                        *
 ******************************************************************/

// command line input of WaveMon parameters

   if(argc < 2) DumpHelp();
   else {
      for(int ac = 1; ac < argc; ++ac) {

	 if( strcmp(argv[ac],"-i") == 0 ){
	   sprintf(configFile,"%s",argv[++ac]);
	   configure = true;
	 }
	 
	 else if( strcmp(argv[ac],"-C") == 0 ){
	   sprintf(masterChannel,"%s",argv[++ac]);
	   cout<<"master channel    = "<<masterChannel<<endl;
	 }

	 else if( strcmp(argv[ac],"-c") == 0 ){
	   w = new char[128];
	   sprintf(w,"%s",argv[++ac]);
	   slaveChannel.push_back(w);
	   cout<<"slave channel     = "<<w<<endl;
	 }

	 else if( strcmp(argv[ac],"-H") == 0 ){
	   sprintf(summaryFile,"%s",argv[++ac]);
	 }

	 else if( strcmp(argv[ac],"-lock") == 0 ){
	   sprintf(lockCondition,"%s",argv[++ac]);
	 }

	 else if( strcmp(argv[ac],"-suffix") == 0 ){
	   sprintf(suffix,"%s",argv[++ac]);
	 }

	 else if( strcmp(argv[ac],"-infile") == 0 ){
	   sprintf(offlineFile,"%s",argv[++ac]);
	   offline = true;
	 }

	 else if( strcmp(argv[ac],"-inlist") == 0 ){
	   sprintf(offlineFile,"%s",argv[++ac]);
	   if( (out=fopen(offlineFile,"r")) == NULL ){ 
	     DumpHelp(); exit(0); 
	   }
	   else {
	     char s[256];
	     while(fgets(s,256,out) != NULL) maxDump++;
	     offline = true;
	     fclose(out);
	   }
	 }

	 else if ( strcmp(argv[ac],"-f") == 0 )
	   sscanf(argv[++ac],"%lf",&resolution);

	 else if ( strcmp(argv[ac],"-v") == 0 )
	   sscanf(argv[++ac],"%d",&verbose);

	 else if ( strcmp(argv[ac],"-line") == 0 )
	   sscanf(argv[++ac],"%d",&line60Hz);

	 else if ( strcmp(argv[ac],"-w") == 0 )
	   sscanf(argv[++ac],"%d",&waveorder);

	 else if ( strcmp(argv[ac],"-a") == 0 )
	   sscanf(argv[++ac],"%lf",&asymmetryCut);

	 else if ( strcmp(argv[ac],"-gate") == 0 )
	   sscanf(argv[++ac],"%lf",&timeGate);

	 else if ( strcmp(argv[ac],"-L") == 0 )
	   sscanf(argv[++ac],"%lf",&likelihoodCut);

	 else if ( strcmp(argv[ac],"-p") == 0 )
	   sscanf(argv[++ac],"%lf",&pfraction);

	 else if ( strcmp(argv[ac],"-rate") == 0 )
	   sscanf(argv[++ac],"%lf",&samplingRate);

	 else if ( strcmp(argv[ac],"-T") == 0 )
	   sscanf(argv[++ac],"%lf",&timeStride);

         else if ( strcmp(argv[ac],"-dump") == 0 )
	   sscanf(argv[++ac],"%d",&nDump);

         else if ( strcmp(argv[ac],"-t") == 0 )
	   timeInterval = strtod(argv[++ac], 0);

         else if ( strcmp(argv[ac],"-size") == 0 )
	   sscanf(argv[++ac],"%d",&minClusterSize);

         else if ( strcmp(argv[ac],"-b") == 0 )
            sscanf(argv[++ac],"%d",&nBuffer);

         else if ( strcmp(argv[ac],"-B") == 0 )
            sscanf(argv[++ac],"%lf",&offset);

         else cout<<"unknown argument: "<<argv[ac]<<endl;
      }
   }

   if(configure) readConfig(configFile);

// print out parameters
   timeStride = double(int(timeStride+0.5));
   if(timeStride<=0.) timeStride = 30.;
   resolution = double(1<<(getDeep(resolution,2.)+1));
   if(resolution<=0.) resolution = 16.;
   samplingRate = double(1<<(getDeep(samplingRate,2.)+1));
   if(samplingRate<=0.) samplingRate = 2048.;

   cout<<"time Stride, s    = "<<timeStride<<endl; 
   cout<<"time Interval, s  = "<<timeInterval<<endl; 
   cout<<"percentile, %     = "<<pfraction<<endl;       
   cout<<"resolution, Hz    = "<<resolution<<endl;        
   cout<<"sampling rate, Hz = "<<samplingRate<<endl;     
   cout<<"asymmetry cut     = "<<asymmetryCut<<endl;     
   cout<<"likelihood cut    = "<<likelihoodCut<<endl;     
   cout<<"min cluster size  = "<<minClusterSize<<endl;         
   cout<<"WaveMon buffer, s = "<<offset<<endl;       
   cout<<"dmtviewer buffer  = "<<nBuffer<<endl;       
   cout<<"dump rate         = "<<nDump<<endl;         

   if(timeGate > 0.)
     cout<<"coincidence gate  = "<<timeGate<<endl;     
   else timeGate = -1.;

   if(line60Hz)
   cout<<"remove 60Hz lines = "<<line60Hz<<endl;     
   cout<<"wavefilter length = "<<waveorder<<endl;     
   if(strlen(lockCondition))
   cout<<"lock condition    = "<<lockCondition<<endl;         
   maxDump = int(maxDump*16./timeStride);
   if(maxDump>0)
   cout<<"number of strides = "<<maxDump<<endl;

   if(!slaveChannel.size()) DumpHelp(); 

   Filter = new LineFilter(-60.,timeStride);
   Filter->setFilter(2,line60Hz);

   readLockLoss(offline);

// get OUTPUT directories

   if((w = getenv("DMTHTMLOUT")))
     if(strlen(w)){

       if(w[strlen(w)-1] == '/')
	 sprintf(htmlDir,"%s",w);
       else
	 sprintf(htmlDir,"%s/",w);

       sprintf(dir,"%s.",htmlDir);
       if( (out=fopen(dir,"r"))==NULL ) {
	 cout << "Error: can't open output html file " << dir << endl;
	 exit(0);
       }
       fclose(out);
     }

   if((w = getenv("DMTOUTPUT")))
     if(strlen(w)){

       if(w[strlen(w)-1] == '/')
	 sprintf(outputDir,"%s",w);
       else
	 sprintf(outputDir,"%s/",w);

       sprintf(dir,"%s.",outputDir);
       if( (out=fopen(dir,"r"))==NULL ) {
	 cout << "Error: can't open output trend  file " << dir << endl;
	 exit(0);
       }
       fclose(out);
     }

   if((w = getenv("DMTRENDOUT")))
     if(strlen(w)){

       if(w[strlen(w)-1] == '/')
	 sprintf(trendDir,"%s",w);
       else
	 sprintf(trendDir,"%s/",w);

       sprintf(dir,"%s.",trendDir);
       if( (out=fopen(dir,"r"))==NULL ) {
	 cout << "Error: can't open output frame  file " << dir << endl;
	 exit(0);
       }
       fclose(out);
     }

     
/**********************************************
 * check useage of initialization function    *
 **********************************************/

// no channel is specified
   if (!(strlen(masterChannel)>2 && masterChannel[2] == ':')){  
     cout << "No channel is specified: "<< endl;
     DumpHelp();
   }

   string ifo(masterChannel, 2);

// set time stride
   float tStep(timeStride);
   getDacc().setStride(tStep);
   getDacc().setIgnoreMissingChannel(true);

// set channels
   getDacc().addChannel(masterChannel);
   masterTS = NULL;
   for(size_t i=0; i<slaveChannel.size(); i++){
     getDacc().addChannel(slaveChannel[i]);
     slaveTS.push_back(NULL);
   }

// get GPS time
   if(!offline) sprintf(gpsTime,"%d",int(Now().getS()));


// set server Name
   if(strlen(suffix)<1)
     sprintf(serverName,"WaveMon_%s",masterChannel);
   else
     sprintf(serverName,"WaveMon_%s_%s",ifo.c_str(),suffix);

   while((w = strchr(serverName,':'))) w[0] = '-';
   setServerName(serverName);
   cout <<"server name: "<< serverName << endl;


// set file name for summary
   if(strlen(summaryFile)){
     sprintf(dir,"%s%s",htmlDir,summaryFile);
     sprintf(summaryFile,"%s",dir);
   }
   else{
     sprintf(summaryFile,"%s%s.txt",htmlDir,serverName);
   }
   while((w = strchr(summaryFile,':'))) w[0] = '-';
   cout <<"output summary file name: "<< summaryFile << endl;

// set file name for backup summary 
   sprintf(backupFile,"%s%s-%s.txt",htmlDir,serverName,gps);
   while((w = strchr(backupFile,':'))) w[0] = '-';
	 
// set file name for triggers 
   sprintf(triggerFile,"%s%s-%s.trg",outputDir,serverName,gps);
   while((w = strchr(triggerFile,':'))) w[0] = '-';
	 

// setup frame trend data
   while((w = strchr(serverName,'-'))) w[0] = 'x';
   while((w = strchr(serverName,'-'))) w[0] = 'x';
   while((w = strchr(serverName,'-'))) w[0] = 'x';
   mTrend.setName(serverName);
   mTrend.setType(Trend::kMinute);
   mTrend.setFrameCount(1);
   mTrend.setIFO(ifo.c_str());

// allocate memory for pixel significance
// allocate memory for pixel occupancy
// allocate memory for channelName
// allocate memory for clusterRate


   cout<<"set trend"<<endl;

   wavearray<float> a(nBuffer);
   wavearray<double> r(10);
   wavearray<float>* pf;
   wavearray<double>* pd;
   a = 0.; r = 0.;
   wavearray<float> b(60);
   b.rate(1.);
   b.start(double(long(Now().getS())));
   

         slave.clear();
    wavebuffer.clear();
   dmtviewerTS.clear();
   clusterRate.clear();
   currentRate.clear();
   clusterEffy.clear();
   currentEffy.clear();

   TSeries* pts;
   char* pc;
   char* channel;
   char  ref1[256];
   char  ref2[256];

   size_t n = slaveChannel.size();

   for(size_t i=0; i<n+3; i++){

     if(i<n){
       master = new wavearray<double>;
       slave.push_back(master);          // storage for slave data
       channel = slaveChannel[i];
     }
     else{
       channel = masterChannel;
       if(i==n+1) { sprintf(ref1,"%s_ref1",channel); channel = ref1; }
       if(i==n+2) { sprintf(ref2,"%s_ref2",channel); channel = ref2; }
     }

     string pch(channel);
     if (pch.find(":") != string::npos) pch.erase(0,pch.find(":")+1);
     for (string::size_type w=pch.find("-"); w<string::npos; w=pch.find("-")) 
       pch[w] = '_';

     pts = new TSeries;
     pts->setData(Time(0), Interval(1.), a.data, nBuffer);
     dmtviewerTS.push_back(pts);      // rate TS
     pc = new char[100];
     sprintf(pc,"%s:DMT-WAVM_%s_%s",ifo.c_str(),pch.c_str(),suffix);
     channelName.push_back(pc);       // TS name
     serveData(pc, pts);
     mTrend.addChannel(pc);
     cout<<"trend channel: "<<pc<<endl;
     
     pts = new TSeries;
     pts->setData(Time(0), Interval(1.), a.data, nBuffer);
     dmtviewerTS.push_back(pts);      // false alarm TS
     pc = new char[100];
     sprintf(pc,"%s:DMT-WAVM_%s_FA_%s",ifo.c_str(),pch.c_str(),suffix);
     channelName.push_back(pc);       // false alarm channel name
     if(i<=n){
       serveData(pc, pts);
       mTrend.addChannel(pc);
       cout<<"trend channel: "<<pc<<endl;
     }
            
     pts = new TSeries;
     pts->setData(Time(0), Interval(1.), a.data, nBuffer);
     dmtviewerTS.push_back(pts);      // acceptance TS
     pc = new char[100];
     sprintf(pc,"%s:DMT-WAVM_%s_AC_%s",ifo.c_str(),pch.c_str(),suffix);
     channelName.push_back(pc);       // acceptance channel name
//     serveData(pc, pts);
//     mTrend.addChannel(pc);
//     cout<<"trend channel: "<<pc<<endl;
     
       
     pf = new wavearray<float>;
     *pf = b;
     wavebuffer.push_back(pf);         // rate
     pf = new wavearray<float>;
     *pf = b;
     wavebuffer.push_back(pf);         // false alarm rate     
     pf = new wavearray<float>;
     *pf = b;
     wavebuffer.push_back(pf);         // acceptance     
     
     pd = new wavearray<double>;
     *pd = r;
     clusterRate.push_back(pd);        // total rate     
     pd = new wavearray<double>;
     *pd = r;
     clusterRate.push_back(pd);        // false alarm total rate     
     pd = new wavearray<double>;
     *pd = r;
     clusterRate.push_back(pd);        // dummy     
     
     pd = new wavearray<double>;
     *pd = r;
     currentRate.push_back(pd);        // current rate     
     pd = new wavearray<double>;
     *pd = r;
     currentRate.push_back(pd);        // false alarm current rate     
     pd = new wavearray<double>;
     *pd = r;
     currentRate.push_back(pd);        // dummy     
     
     pd = new wavearray<double>;
     *pd = r;
     clusterEffy.push_back(pd);         // total efficiency    
     pd = new wavearray<double>;
     *pd = r;
     clusterEffy.push_back(pd);         // false alarm total efficiency       
     pd = new wavearray<double>;
     *pd = r;
     clusterEffy.push_back(pd);         // dummy     
     
     pd = new wavearray<double>;
     *pd = r;
     currentEffy.push_back(pd);         // current efficiency      
     pd = new wavearray<double>;
     *pd = r;
     currentEffy.push_back(pd);         // false alarm current efficiency      
     pd = new wavearray<double>;
     *pd = r;
     currentEffy.push_back(pd);         // dummy     
   }

   master = new wavearray<double>;

// set name for index file
   if(offline)
     sprintf(indexFile,"%s%s.cfg",trendDir,serverName);
   else
     sprintf(indexFile,"%s%s-%s.cfg",trendDir,serverName,gpsTime);
   while((w = strchr(indexFile,':'))) w[0] = '-';
   mTrend.writeIndex(indexFile);
   sprintf(indexFile,"%schannel.cfg",trendDir);
//   mTrend.writeIndex(indexFile);

}

//--------------------------------------  Skeleton object destructor.
WaveMon::~WaveMon() 
{
  char gzip[256];
  char name[256];
  sprintf(name,triggerFile,gpsTime);
  sprintf(gzip,"gzip %s",name);
  system(gzip);                      // gzip trigger file

  sprintf(name,backupFile,gpsTime);
  sprintf(backupFile,"cp %s %s",summaryFile,name);
  system(backupFile);                // save backup summary file

  for(size_t i=0; i<channelName.size(); i++){
    if(dmtviewerTS[i])  delete    dmtviewerTS[i];
    if(channelName[i])  delete [] channelName[i];
    if( wavebuffer[i])  delete     wavebuffer[i];
    if(clusterRate[i])  delete    clusterRate[i];
    if(currentRate[i])  delete    currentRate[i];
    if(clusterEffy[i])  delete    clusterEffy[i];
    if(currentEffy[i])  delete    currentEffy[i];
  }

  for(size_t i=0; i<slave.size(); i++){
    if(slave[i])   delete[] slave[i];
    if(slaveTS[i]) delete[] slaveTS[i];
  }

  if(master)   delete [] master;
  if(masterTS) delete [] masterTS;

  mTrend.close();

  cout << "WaveMon is finished" << endl;
}

//-- This skeleton function needed for writing to the dmt viewer

void WaveMon::Attention(void) {
  MonServer::Attention();
  return;
}

/*  Frame processing function.      *
 *  do this step every timestride.  */
void WaveMon::ProcessData(void) {

  wavearray<float>    buf(int(timeStride+0.5));
  buf.rate(1.); buf=0;
  wavearray<double>    tmp;

  Biorthogonal<double> B(32);
   Symlet<double>      D32(32,1);
   Symlet<double>      D(waveorder,1);
  WSeries<double>      wavelet(B);
  WSeries<double>      wave;
  WSeries<double>      wMaster(D);
  WSeries<double>      pMaster(D);
  WSeries<double>      wSwap;
  WSeries<double>      pSwap;
  WSeries<double>      wSlave(D);
  WSeries<double>      pSlave(D);
  WSeries<double>      swRandom;
  WSeries<double>      sRandom;
  WSeries<double>      mwRandom;
  WSeries<double>      mRandom;

  wavecluster          sCluster;

  TSeries* ts = NULL;

  char name[256];
  int nLPF;
  bool inlock  = true;
  size_t n,m;

  if(verbose>8) cout<<"\nprepare read data"<<endl;

  ts = getDacc().refData(masterChannel);
  if(!ts) {cout<<"WaveMon: get no data"<<endl; exit(0);}

  if(!masterTS) {
    masterTS = new TSeries(*ts);
    tmp = *masterTS;
    masterTS->eraseStart(Interval(timeStride-offset));
  }
  else if(masterTS->Append(*ts)){     // if return code != 0 - no append
    masterTS->eraseStart(Interval(timeStride-offset));
    tmp = *ts;
    inlock = false;
  }
  else {                         // append sucsessful
    tmp= *masterTS;
    masterTS->eraseStart(Interval(timeStride));
  }

  if(verbose>7){ 
    printf("lock condition %s\n",lockCondition);
    printf("master size=%zd, rate=%f, start=%f\n",
	   tmp.size(), tmp.rate(), tmp.start());
  }

// set lock condition
  if(strlen(lockCondition) && inlock) inlock = mOSC.satisfied(lockCondition);

  nLPF = getDeep(tmp.rate(),samplingRate);
  currentTime = ts->getStartTime()-Interval(offset/2.);

// decimate master channel 
  if(nLPF>0){
    wavelet.Forward(tmp,nLPF);
    wavelet.getLayer(*master,0);
  }
  else *master = tmp;

// read slave channel
  for(size_t i=0; i<slave.size(); i++){
    ts = getDacc().refData(slaveChannel[i]);
    if(!ts) {cout<<"WaveMon: get no data"<<endl; return;}

    if(!slaveTS[i]) {
      slaveTS[i] = new TSeries(*ts);
      tmp = *slaveTS[i];
      slaveTS[i]->eraseStart(Interval(timeStride-offset));
    }
    else if(slaveTS[i]->Append(*ts)){   // if return code != 0 - no append
      slaveTS[i]->eraseStart(Interval(timeStride-offset));
      tmp = *ts;
      inlock = false;
    }
    else {                         // append sucsessful
      tmp= *slaveTS[i];
      slaveTS[i]->eraseStart(Interval(timeStride));
    }

    if(verbose>7) 
      printf("%s:  size=%zd, rate=%f, start=%f\n", slaveChannel[i],
	     tmp.size(), tmp.rate(), tmp.start());

    nLPF = getDeep(tmp.rate(),samplingRate);

    if(inlock){
      if(nLPF>0){
	wavelet.Forward(tmp,nLPF);
	wavelet.getLayer(*(slave[i]),0);
      }
      else *(slave[i]) = tmp;
    }
  }

// clean from 60 Hz
  if(inlock && line60Hz){
    if(verbose>8) cout<<"WaveMon: prepare remove lines ... "<<endl;
    Filter->apply(*master);
    Filter->resize(100);
    if(verbose>8) cout<<" ... lines removed"<<endl;
  }

  nLPF = getDeep(master->rate(),resolution*2); 
  wave.Forward(*master,D32,nLPF);
  if(verbose>8) cout<<"muster channel has been transformed."<<endl;

  wave.white(timeInterval);

  if(wave.size()/wave.rate() > timeStride+offset/2.)
    wMaster=wave[slice(int(offset/2*wave.rate()+0.1),int(timeStride*wave.rate()+0.1),1)];
  else
    wMaster=wave;

  //  wMaster.white(timeInterval);

  n = channelName.size();

  if(firstcall){
    sprintf(gpsTime,"%d",int(currentTime.getS()));
    wAlarm = wMaster; wAlarm = 0.;
    wBurst = wMaster; wBurst = 0.;
    firstcall = false;
    inlock = false;
    for(size_t i=0; i<n; i++) wavebuffer[i]->start(currentTime.getS());
  }

  if(!mDump)
    sprintf(gpsTime,"%d",int(currentTime.getS()));

  if(inlock) liveTime += timeStride;
  totalTime += timeStride;

  buf.start(currentTime.getS());
  if(verbose>0) printf("data start time: %10f,  lock: %d \n",buf.start(),inlock);

  swRandom = wMaster;                               
  mwRandom = wMaster;
  sRandom.percentile(pfraction/100., 3, &swRandom);
  mRandom.percentile(pfraction/100., 3, &mwRandom);

// second reference channel
  sRandom.Coincidence(mRandom, timeGate);

  sCluster.init(sRandom,true);
  sCluster.cluster();
  sCluster.apush(swRandom);        // add slave wavelet amplitude
  sCluster.apush(mRandom);         // add master percentile amplitude
  sCluster.apush(mwRandom);        // add master wavelet amplitude
  setCuts(sCluster);

  addTrend(getRate(sCluster),n-3);
  addTriggers(sCluster, n-3);

  if(verbose>7){ 
    //    printf("rand2  size=%d, rate=%f, start=%f\n",sRandom.size(),sRandom.rate(),sRandom.start());
    //    printf("swap   size=%d, rate=%f, start=%f\n",pSwap.size(),pSwap.rate(),pSwap.start());
  }
  if(verbose>6) cout<<"  rand2 masksize="<<sCluster.size()<<endl;


  if(inlock){

// calculate master percentile
    wSwap = wMaster;
    m = wMaster.size();

    wSwap.cpf(wMaster,m/2,m/2,0);  // swap master channel
    wSwap.cpf(wMaster,m/2,0,m/2);

    pSwap.percentile(pfraction/100., 1, &wSwap);     // master swap percentile amplitude
    pSwap.getLayer(tmp,0); tmp = 0.; pSwap.putLayer(tmp,0);

    pMaster.percentile(pfraction/100., 1, &wMaster); // master percentile amplitude
    pMaster.getLayer(tmp,0); tmp = 0.; pMaster.putLayer(tmp,0);

// first reference channel
    mRandom.Coincidence(pMaster, timeGate);    

    sCluster.init(mRandom,true);
    sCluster.cluster();
    sCluster.apush(mwRandom);        // add slave wavelet amplitude
    sCluster.apush(pMaster);         // add master percentile amplitude
    sCluster.apush(wMaster);         // add master wavelet amplitude
    setCuts(sCluster);

    addTrend(getRate(sCluster),n-6);
    addTriggers(sCluster, n-6);
  }      
  else{
    addTrend(buf,n-6);
//    addTrend(buf,n-4);
  }

  if(verbose>6 && inlock) 
    cout<<"  rand1 masksize="<<sCluster.size()<<endl;


// fill in slave occupancy and TSeries
  for(size_t i=0; i<slave.size(); i++){
    
    if(inlock){

      nLPF = getDeep(slave[i]->rate(),resolution*2); 
      wave.Forward(*(slave[i]),D,nLPF);
      if(verbose>8) cout<<"slave channel has been transformed."<<endl;

      wave.white(timeInterval);

      if(wave.size()/wave.rate() > timeStride+offset/2.)
	wSlave=wave[slice(int(offset/2*wave.rate()+0.1),int(timeStride*wave.rate()+0.1),1)];
      else
	wSlave = wave;

      pSlave.percentile(pfraction/100., 1, &wSlave);

      if(verbose>8) cout<<"start cluster analysis"<<endl;

// swap rate
      sRandom  = pSlave;
      sRandom.Coincidence(pSwap, timeGate);

      sCluster.init(sRandom,true);
      sCluster.cluster();
      sCluster.apush(wSlave);          // add slave wavelet amplitude
      sCluster.apush(pSwap);           // add master percentile amplitude
      sCluster.apush(wSwap);           // add master wavelet amplitude
      setCuts(sCluster);
      
      addTrend(getRate(sCluster),3*i+1); 
      addTriggers(sCluster, 3*i+1);

      //      sRandom.mask(1,true);
      //      wAlarm += sRandom;
    }
    else 
      addTrend(buf,3*i+1);

    if(verbose>6 && inlock)
      cout<<"  swap  masksize="<<sCluster.size()<<endl;
      
// trigger
    if(inlock){
      pSlave.Coincidence(pMaster, timeGate);

      sCluster.init(pSlave,true);
      sCluster.cluster();
      sCluster.apush(wSlave);          // add slave wavelet amplitude
      sCluster.apush(pMaster);         // add master percentile amplitude
      sCluster.apush(wMaster);         // add master wavelet amplitude
      setCuts(sCluster);
      
      addTrend(getRate(sCluster),3*i); 
      addTriggers(sCluster, 3*i);

      //      pSlave.mask(1,true);
      //      wBurst += pSlave;
    }
    else{
//      addTrend(buf,3*i+2)
      addTrend(buf,3*i);
    }

    if(verbose>6 && inlock)
      cout<<"  slave masksize="<<sCluster.size()<<endl;
  }

/*
// total trigger & swap rate
  if(inlock){
    wAlarm.setMask(maxCS,true);
    wAlarm.start(wMaster.start());
    wAlarm.cluster();
    addTrend(getRate(wAlarm),n-8);
    addTriggers(wAlarm, n-8);

    wBurst.setMask(maxCS,true);
    wBurst.start(wMaster.start());
    wBurst.cluster();
//    addTrend(getAcceptance(wBurst),n-7);
    addTrend(getRate(wBurst),n-9);
    addTriggers(wBurst, n-9);
  }
  else{
//    addTrend(buf,n-7);
*/

  addTrend(buf,n-8);
  addTrend(buf,n-9);

/*
  }

  if(verbose>6 && inlock){
    cout<<"  trigr masksize="<<wBurst.masksize()<<endl;
    cout<<"  alarm masksize="<<wAlarm.masksize()<<endl;
  }
  wAlarm = 0.;
  wBurst = 0.;
*/

// dump summary
  DumpSummary(summaryFile);

// dump triggers & update rates
  sprintf(name,triggerFile,gpsTime);
  DumpTrigger(name, 1);
  
  n = currentRate.size();
  for (size_t i=0; i<n; i++){
    *(currentRate[i]) = 0.;
    *(currentEffy[i]) = 0.;
  }

  nDump++;
  mDump++;

  if(mDump > 1440*(60./timeStride) && !inlock) {
    mDump=0;
    char gzip[256];
    sprintf(gzip,"gzip %s",name);
    system(gzip);                      // gzip trigger file
    sprintf(name,backupFile,gpsTime);
    DumpSummary(name);
  }

  if(maxDump>0 && nDump >= maxDump){
    sprintf(name,triggerFile,gpsTime);
    char gzip[256];
    sprintf(gzip,"gzip %s",name);
    system(gzip);                      // gzip trigger file
    exit(0);
  }
  if(verbose>8) cout<<"exit data processing"<<endl;
}
    

/*
wavearray<float> WaveMon::getAcceptance(WSeries<double>& ws) 
{
  wavearray<double> occ = ws.occupancy(false);

  size_t N = int(occ.size()/occ.rate()+0.5);
  size_t n = int(occ.rate()+0.5);
  wavearray<double> tmp(n);
  tmp.rate(occ.rate());

  wavearray<float> buf(N);
  buf.start(ws.start());
  buf.rate(1.);
  buf = 0.;

  for (size_t i=0; i<N; i++){
    tmp.cpf(occ, n, i*n);
    buf.data[i] = 1.-100.*tmp.mean()/pfraction;
    if(buf.data[i]<0.) buf.data[i] = 0.; 
  }
  return buf;
}
*/

wavearray<float> WaveMon::getRate(wavecluster& ws) 
{
  size_t i, n;
  wavearray<float> t = ws.get("time");
  size_t t0 = size_t(ws.start);

  wavearray<float> buf(int(timeStride+0.5));
  buf.start(ws.start);
  buf.rate(1.);
  buf = 0.;


  for(i=0; i<t.size(); i++){
     n = size_t(t.start()+t.data[i]) - t0;
     if(n>= buf.size()) 
       cout<<"WaveMon::getRate() error. "<<t.data[i]<<"  "<<t0<<endl;
     else buf.data[n] += 1.;
  }
  return buf;
}


size_t WaveMon::setCuts(wavecluster& ws) 
{
  int k;
  size_t i, id;
  size_t nCluster = 0;;
  double T = log(double(minClusterSize))*fabs(likelihoodCut);
  wavearray<float> c = ws.get("ID");
  wavearray<float> s = ws.get("size"); 
  wavearray<float> L = ws.get("significance",-1);
  //  wavearray<float> E = ws.get("energy", 2);

  for(i=0; i<c.size(); i++){

    id = size_t(c.data[i]+0.5);
     k = int(s.data[i]+0.5);
    if(k < minClusterSize)                  { ws.ignore(id); continue; }
    if(timeGate > 0.)                       {    nCluster++; continue; }
    if(L.data[i] < T)                       { ws.ignore(id); continue; }
    nCluster++;
  }
  return nCluster;
}


size_t WaveMon::addTriggers(wavecluster& ws, size_t index) 
{
  wavearray<float> s = ws.get("size");
  size_t nCluster = s.size(); 
  if(!nCluster) return 0;

  wavearray<float> T  = ws.get("start");
  wavearray<float> D  = ws.get("stop");
  wavearray<float> F  = ws.get("frequency");
  wavearray<float> L  = ws.get("low");
  wavearray<float> B  = ws.get("high");
  wavearray<float> V  = ws.get("volume");
  wavearray<float> S  = ws.get("significance");
  wavearray<float> Ls = ws.get("likelihood");
  wavearray<float> Es = ws.get("SNR",2);
  wavearray<float> Lm = ws.get("likelihood",-3);
  wavearray<float> Em = ws.get("SNR",4);

  D -= T;       // duration
  B -= L;       // bandwith

  double start, tStart;
  list<char*>::iterator ib;
  list<char*>::iterator ie;
  char* w;
  char* v;
  int m;  

  for(size_t i=0; i<nCluster; i++){

// update cluster rates
    m = ((int)s.data[i] - minClusterSize)/2;
    if(m>=(int)clusterRate[index]->size()) m = clusterRate[index]->size()-1;

    for(int j=m; j>=0; j--){
       clusterRate[index]->data[j] += 1.;
       currentRate[index]->data[j] += 1.;
       clusterEffy[index]->data[j] += V.data[i]/V.rate();
       currentEffy[index]->data[j] += V.data[i]/V.rate();
    }

  size_t n = channelName.size(); 
  if(index == n-8 || index == n-9) continue;  // skip total trigger

    w = new char[256];

// write triggers
    start = ws.start + double(T.data[i]);
    sprintf(w,"%18.6f %9.6f %5.0f %5.0f %5d %5d %8.2e %8.2e %8.2e %8.2e %8.2e %s\n",
	    start, D[i], F[i], B[i], (int)s[i], (int)V[i], S[i], Em[i],
	    Es[i], Lm[i], Ls[i], channelName[index]);

    //    cout<<"trigger: "<<w<<endl;
    if(!wTrigger.size()) { wTrigger.push_back(w); continue; }

    ib = wTrigger.begin();
    ie = wTrigger.end();
    while(true) { 
      ie--; v = *ie;
      sscanf(v,"%18lf",&tStart); 
      if(start > tStart) { wTrigger.insert(++ie, w); break; } 
      if(ie == ib)       { wTrigger.push_front(w); break; }
    }
  }
  return nCluster;
}



void WaveMon::DumpTrigger(char* fname, int app)
{
  list<char*>::iterator it_end = wTrigger.end();
  list<char*>::iterator it_beg = wTrigger.begin();
  list<char*>::iterator it;
  char mode[3] = "w";
  if (app == 1) strcpy (mode, "a");

  FILE *fp;

  if ( (fp = fopen(fname, mode)) == NULL ) {
     cout << " WaveMon::DumpTrigger() error: cannot open file " << fname <<". \n";
     return;
  };

  for (it=it_beg; it!=it_end; it++) { 
    fprintf( fp,"%s", *it); 
    delete [] (char*)(*it); 
  }
  wTrigger.clear();
  fclose(fp); 
}


void WaveMon::DumpSummary(char* fname)
{
  int n=channelName.size();

  wavearray<int> q(10);
  wavearray<double> r(10);

  char cbuf[256];
  double* p;
  int s = (mDump+1)*int(timeStride);
  double invTime = 0.;
  bool skip;
  bool VETO = true;

  FILE *fp;

  if ( (fp = fopen(fname, "w")) == NULL ) {
     cout << " WaveMon::DumpSummary() error: cannot open file " << fname <<". \n";
     return;
  };

  LocalStr(Now(), cbuf, "%M %d, %Y %02H:%02N");

  fprintf( fp, "WaveMon statistics at:      %s\n", cbuf);
  fprintf( fp, " \n");
  fprintf( fp, "master channel              %s\n", masterChannel);
  if(strlen(lockCondition))
  fprintf( fp, "lock condition              %s\n", lockCondition);
  fprintf( fp, "process GPS time            %9ld\n",currentTime.getS());
  fprintf( fp, " \n");
  fprintf( fp, "total time, sec             %6d\n",int(totalTime));
  fprintf( fp, "live time,  sec             %6d\n",int(liveTime));
  fprintf( fp, " \n");
  fprintf( fp, "Input parameters:\n\n");
  fprintf( fp, "    time stride, sec      = %6.0f\n", timeStride);
  fprintf( fp, "    time interval, sec    = %6.0f\n", timeInterval);
  fprintf( fp, "    resolution, Hz        = %6.0f\n", resolution);
  fprintf( fp, "    sampling rate, Hz     = %6.0f\n", samplingRate);
  fprintf( fp, "    wavemon buffer, s     = %6.0f\n", offset);
  fprintf( fp, "    dmtviewer buffer      = %6d\n", nBuffer);
  fprintf( fp, "    min cluster size      = %6d\n", minClusterSize);
  fprintf( fp, "    percentile fraction   = %6.2f\n", pfraction);
  fprintf( fp, "    asymmetry cut         = %6.2f\n", asymmetryCut);
  fprintf( fp, "    likelihood cut        = %6.2f\n", likelihoodCut);
  if(timeGate > 0.)
  fprintf( fp, "    coincidence gate, sec = %6.4f\n", timeGate);
  if(line60Hz)
  fprintf( fp, "    remove 60Hz lines     = %6d\n", line60Hz);
  fprintf( fp, "    wavelet filter length = %6d\n", waveorder);


  for(int i=0; i<n; i+=3){

    if( VETO && i/3 != n/3-3) continue;
    if(!VETO && i/3 == n/3-3) continue;

    if(i/3 == n/3-1)
      invTime = (totalTime>0.) ? 1./totalTime : 0.;
    else
      invTime = (liveTime>0.) ? 1./liveTime : 0.;


    fprintf( fp, " \n");
     
    for(size_t j=0; j<10; j++) q.data[j] = minClusterSize + (j*2) - 1;

    skip = false;

    if(i/3 < (int)slave.size())  
      fprintf(fp, "***** coincidence with %s\n\n", slaveChannel[i/3]);
    else if(i/3 == n/3-2) 
      {fprintf(fp, "***** coincidence with random channel\n\n"); skip=true; }
    else if(i/3 == n/3-1) 
      {fprintf(fp, "***** coincidence of two random channels\n\n"); skip=true; }
    else if(i/3 == n/3-3)   
       fprintf(fp, "***** VETO trigger for %s\n\n",masterChannel);

    fprintf( fp, 
	     "     cluster size:      >%2d     >%2d     >%2d     >%2d     >%2d",
	     q[0],q[1],q[2],q[3],q[4]);
    fprintf( fp, "     >%2d     >%2d     >%2d     >%2d     >%2d \n",
	     q[5],q[6],q[7],q[8],q[9]);

    p = clusterRate[i]->data;
    for(size_t j=0; j<10; j++) q.data[j] = (int)p[j];
    
    fprintf( fp, 
	     "   cluster counts:  %7d %7d %7d %7d %7d %7d %7d %7d %7d %7d \n", 
	     q[0],q[1],q[2],q[3],q[4],q[5],q[6],q[7],q[8],q[9]);
    
    if(!verbose) fprintf( fp, "----- AVERAGE -----\n");
    
    r = *(clusterRate[i]);
    r *= invTime;
    fprintf( fp, 
	     " cluster rate, Hz:  %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e \n", 
	     r[0],r[1],r[2],r[3],r[4],r[5],r[6],r[7],r[8],r[9]);
    
    if(!skip){
      r = *(clusterRate[i+1]);
      r *= invTime;
      fprintf( fp, 
	     "off-time rate, Hz:  %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e \n", 
	       r[0],r[1],r[2],r[3],r[4],r[5],r[6],r[7],r[8],r[9]);
    }
    
    r = *(clusterEffy[i]);
    r *= invTime*99.9/pfraction; r-=1.; r*=-1.;
    fprintf( fp,        
	     "       acceptance:  %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f \n",
	     r[0],r[1],r[2],r[3],r[4],r[5],r[6],r[7],r[8],r[9]);
    
    if(!skip && !verbose){
      r = *(clusterEffy[i+1]);
      r *= invTime*99.9/pfraction; r-=1.; r*=-1.;
      fprintf( fp, 
	       "       ACCEPTANCE:  %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f \n",
	       r[0],r[1],r[2],r[3],r[4],r[5],r[6],r[7],r[8],r[9]);
    }
    
    if(!verbose) fprintf( fp, "----- LAST %4d seconds -----\n", s);
    
    r = *(currentRate[i]);
    r *= 1./float(s);
    if(!verbose) 
      fprintf( fp, 
	       " cluster rate, Hz:  %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e \n", 
	       r[0],r[1],r[2],r[3],r[4],r[5],r[6],r[7],r[8],r[9]);
    
    if(!skip && !verbose){
      r = *(currentRate[i+1]);
      r *= 1./float(s);
      fprintf( fp, 
	       "off-time rate, Hz:  %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e %3.1e \n", 
	       r[0],r[1],r[2],r[3],r[4],r[5],r[6],r[7],r[8],r[9]);
    }
    
    r = *(currentEffy[i]);
    r *= 99.9/pfraction/float(s); r-=1.; r*=-1.;
    if(!verbose) 
      fprintf( fp, 
	       "       acceptance:  %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f \n",
	       r[0],r[1],r[2],r[3],r[4],r[5],r[6],r[7],r[8],r[9]);
    
    if(!skip && !verbose){
      r = *(currentEffy[i+1]);
      r *= 99.9/pfraction/float(s); r-=1.; r*=-1.;
      fprintf( fp, 
	      "       ACCEPTANCE:  %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f %7.3f \n\n\n", 
	       r[0],r[1],r[2],r[3],r[4],r[5],r[6],r[7],r[8],r[9]);
    }
    
    if(VETO) { VETO = false; i=-3; }

  }

  fclose(fp); 
}


void WaveMon::addTrend(wavearray<float> buff, int index)
{
  wavearray<float>* p =  wavebuffer[index];
  double start;
  double delta = buff.start() - p->start();

  size_t N = buff.size();
  size_t M = p->size();
  size_t buffStart = delta<0. ? -size_t(delta-0.5) : 0;
  size_t waveStart = delta>0. ?  size_t(delta+0.5) : 0;
  size_t i, length;

  wavearray<float> temp;
  TSeries ts;

//  printf("start time %10f  %10f \n",buff.start(), p->start());
  
  if(buffStart >= N) return;

  size_t k = waveStart/M;
  waveStart -= k*M;
  if(k){ *p = 0.; p->start(p->start()+M*k); }

//  cout<<buffStart<<" "<<waveStart<<"  "<<N<<"  p.size="<<p->size()<<endl;

  do{
    if(waveStart<M){                              // copy data
      length = N - buffStart;
      if(length > M - waveStart) length = M - waveStart;
      p->cpf(buff,length,buffStart,waveStart);
      buffStart += length;
      waveStart += length;
    }
    
    if(waveStart>=M){                             // update trends
      temp = *(dmtviewerTS[index]);               // dmtviewer
      temp.cpf(temp,temp.size()-1,k+1);
      for(i=0; i<k; i++) temp.data[temp.size()-k-1] = 0.;
      temp.data[temp.size()-1] = p->mean();
      start = p->start()-nBuffer*M; 
      dmtviewerTS[index]->setData(Time((long unsigned int)start), Interval(M), temp.data, nBuffer);
      
      ts.setData(Time((long unsigned int)p->start()), Interval(1.), p->data, M);
      mTrend.trendData(channelName[index], ts);   // trend file
      
      p->start(p->start()+M);
      waveStart = 0;
      k = 0;
    }
  } while(buffStart < N);
}


char *WaveMon::getToken(pchar &s, bool ignore)
// ignore = true - ignore punctuation except space
{
   char *w = 0;
   char *eos = s+strlen(s); 

//   cout << endl << s;

   s--;
   while(++s < eos){

      if(ispunct(*s) && !ignore && *s != '-' && *s != '.' ){
	*s = '\0'; break;
      }

      if(!isspace(*s) && !w) w = s;  // first character
      if( isspace(*s) &&  w){ *(s++) = '\0'; break; }

   }
//   if(w) cout <<strlen(w)<<"  token  "<< w << endl;
   return (w) ? w : eos;
}


// read LineMonitor parameters from configuration file
bool WaveMon::readConfig(char *){
  FILE* out;

  if( (out=fopen(configFile,"r")) == NULL ){ DumpHelp(); exit(0); } 

  char s[256];
  char *w = NULL;

  while(fgets(s,100,out) != NULL){
    pString = s;
//    cout << endl << s;
    if(!strlen(s)) break;
       
    while(strlen(w=getToken(pString))>1){

	if(getParameter(w,"-T",timeStride))        continue;
	if(getParameter(w,"-t",timeInterval))      continue;
	if(getParameter(w,"-p",pfraction))         continue;
	if(getParameter(w,"-f",resolution))        continue;
	if(getParameter(w,"-rate",samplingRate))   continue;
	if(getParameter(w,"-a",asymmetryCut))      continue;
	if(getParameter(w,"-L",likelihoodCut))     continue;
	if(getParameter(w,"-b",nBuffer))           continue;
	if(getParameter(w,"-B",offset))            continue;
	if(getParameter(w,"-v",verbose))           continue;
	if(getParameter(w,"-w",waveorder))         continue;
	if(getParameter(w,"-line",line60Hz))       continue;
	if(getParameter(w,"-size",minClusterSize)) continue;
	if(getParameter(w,"-gate",timeGate))       continue;
	
	if(strcmp(w,"-C") == 0){
	  sprintf(masterChannel,"%s",w=getToken(pString, true));
	  cout<<"master channel    = "<<masterChannel<<endl;
	}
	
	else if( strcmp(w,"-suffix") == 0 )
	  sprintf(suffix,"%s",w=getToken(pString, true));
	
	else if(strcmp(w,"-c") == 0){
	  char* pch = new char[128];
	  sprintf(pch,"%s",w=getToken(pString, true));
	  slaveChannel.push_back(pch);
	  cout<<"slave channel     = "<<pch<<endl;
	}
	
	else if(strcmp(w,"-H") == 0)
	  sprintf(summaryFile,"%s",w=getToken(pString, true));

	else if(strcmp(w,"-lock") == 0)
	  sprintf(lockCondition,"%s",w=getToken(pString, true));
	
      }
  }
  fclose(out);
  return true;
}

void WaveMon::readLockLoss(bool offline){
  FILE* out;
  string s;

// read OperStateCondition file
   if(strlen(lockCondition)) {
     mOSC.setStride(int(timeStride+0.1));
     if( (out=fopen("/export/home/ops/pars/LockLoss.conf","r"))==NULL || offline) {
       if(!offline) cout << "Error: can't open /export/home/ops/pars/LockLoss.conf "<< endl;
       fclose(out);

       if( (out=fopen("LockLoss.conf","r"))==NULL ) {
	 cout << "Error:         can't open LockLoss.conf file"<< endl;
	 cout << "copy LockLoss.conf file into local directory"<< endl;
	 exit(0);
       }
       else{
	 fclose(out);
	 s += string("LockLoss.conf");
       }
     }
     
     else{
       fclose(out);
       s += string("/export/home/ops/pars/LockLoss.conf");
     }
   }
     
   if (!s.empty()) {
     cout<<"read lock loss file: "<<s.c_str()<<endl;
     mOSC.readConfig(s.c_str());
   }

}

void WaveMon::DumpHelp(void) {
   cout << "                   Wave Burst Monitor                    " << endl
        << "                                                         " << endl  
        << "reference: http://www.phys.ufl.edu/LIGO/wavelet/         " << endl
        << "           S.Klimenko, University of Florida             " << endl
        << "                                                         " << endl  
        << "Perpose: WaveMon is used to detect glitches and bursts   " << endl
        << "         for coincidence of master (usually LSC-AS_Q)    " << endl  
        << "         and slave (PEM or control) channels.            " << endl  
        << "                                                         " << endl  
        << "Method: The WaveMon performs a time-frequency analysis in" << endl  
        << "        wavelet domain.                                  " << endl  
        << "                                                         " << endl  
        << "Output: The WaveMon has the following output:            " << endl
        << "        TREND DATA - rate, false alarm rate, efficiency, " << endl
        << "        which are served to the dmtviewer and saved in   " << endl
        << "        the trend frame files (minute trends).           " << endl
        << "        TRIGGERS - events generated by glitches. Triggers" << endl
        << "        are stored as string entries in ascii file. Each " << endl
        << "        trigger event (glitch) is characterized by para- " << endl
        << "        meters, including GPS time and channel name,     " << endl
        << "        which produced the trigger.                      " << endl
        << "        html SUMMARY - integrated summary, including     " << endl
        << "        number of triggers, average rates, false alarm   " << endl
        << "        rates, ...                                       " << endl  
        << "                                                         " << endl  
        << "command line options :                                   " << endl  
        << "                                                         " << endl  
        << "  -C <channel name>: Master channel name is mandatory.   " << endl
        << "     Only one master channel is permitted.               " << endl  
        << "                                                         " << endl  
        << "  -c <channel name>: Slave channel name is mandatory. At " << endl
        << "     least one slave channel name should be specified.   " << endl  
        << "     As many slave channels can be entered, as CPU can   " << endl  
        << "     handle it in real time. It's recommended to put     " << endl  
        << "     slave channels in the order of decreasing of the    " << endl  
        << "     sampling rate.                                      " << endl  
        << "                                                         " << endl  
        << "  -i <input config file>: The LM configuration file to   " << endl
        << "     track several lines. All options except -i & -infile" << endl
        << "     can be used to build an entry (string) in the file. " << endl            
        << "     Only one parameter should be specified in the string" << endl            
        << "     For example,                                        " << endl            
        << "        -C L1:LSC-AS_Q     // master channel             " << endl            
        << "        -c L1:LSC-AS_I     // slave channel              " << endl            
        << "        -l L1:X_arm_loced  // when x arm locked          " << endl            
        << "        -t 32              // stride 32 sec              " << endl            
        << "        -d 4               // dump every 4 strides       " << endl            
        << "        -r 64.             // resolution 32 Hz           " << endl            
        << "        -p 29.             // percentile 30%             " << endl            
        << "                                                         " << endl  
        << "  -lock <lock condition> Track lines if some lock        " << endl
        << "     condition is specified. For example:                " << endl
        << "           H2:Both_arms_locked                           " << endl
        << "           H1:Mode_cleaner_locked                        " << endl
        << "           L1:X_arm_locked                               " << endl
        << "                                                         " << endl  
        << "  -T <time stride (sec)>: Length of time series requested" << endl
        << "     by WaveMon.                                         " << endl
        << "                                                         " << endl  
        << "  -t <time interval (sec)>: minimal time interval when   " << endl
        << "     data is considered to be stationary. It is recom-   " << endl
        << "     mended do not use time interval less then 10 sec.   " << endl
        << "                                                         " << endl  
        << "  -f <frequency resolution (Hz)>: defines frequency scale" << endl
        << "     resolution                                          " << endl
        << "                                                         " << endl  
        << "  -rate <maximum sampling rate> [2048]:                  " << endl
        << "                                                         " << endl  
        << "  -p <percentile fraction, %> [10]: Fraction of pixels,  " << endl
        << "     which pass the percentile selection                 " << endl
        << "                                                         " << endl  
        << "  -a <asymmetry>: cluster asymmetry selection cut        " << endl
        << "                                                         " << endl  
        << "  -L <likelihood>: cluster likelihood selection cut.     " << endl
        << "                                                         " << endl  
        << "  -size <minimum cluster size>: All clusters with smaller" << endl
        << "     size are excluded from analysis                     " << endl
        << "                                                         " << endl  
        << "  -v <printout verbose level> [0]                        " << endl
        << "                                                         " << endl  
        << "  -w <length of wavelet filter> [32]                     " << endl
        << "                                                         " << endl  
        << "  -gate <coincidence time gate, sec> [0]                 " << endl
        << "                                                         " << endl  
        << "  -line m <remove m first 60Hz lines from master channel>" << endl
        << "  -lock <lock condition> (e.g. *:Both_arms_locked )      " << endl
        << "                                                         " << endl  
        << "  -H name <file name to dump summary file>               " << endl
        << "                                                         " << endl  
        << "  -infile <input frame file>: If  -infile is specified,  " << endl
        << "     the WM will read data from the input frame file.    " << endl
        << "                                                         " << endl  
        << "useage: WaveMon                                          " << endl  
        << "  -c <channel name> [required]                           " << endl
        << "  -H <output summary file name>                          " << endl
        << "  -i <input config file> []                              " << endl
        << "  -f <frequency resolution, Hz> [16]                     " << endl
        << "  -rate <maximum sampling rate, Hz> [2048]               " << endl
        << "  -p <percentile fraction, %> [20]                       " << endl
        << "  -T <time stride, sec> [30.]                            " << endl
        << "  -t <time interval, sec> [10.]                          " << endl
        << "  -size <minimum cluster size> [3]                       " << endl
        << "  -a <cluster asymmetry cut> [0]                         " << endl
        << "  -L <cluster likelihood cut (likelihood)> [0]           " << endl
        << "  -gate <coincidence time gate> [0]                      " << endl
        << "  -v <suppress printout>                                 " << endl
        << "  -b <length of the DMTVIEWER buffer> length=[720]       " << endl
        << "  -B <length of the WaveMon buffer> [2 sec]              " << endl
        << "  -W <lifting wavelet order>[8]                          " << endl
        << "  -lock <lock condition> (*:Both_arms_locked )           " << endl
        << "  -line m <remove m first 60Hz lines from master channel>" << endl
        << "  -suffix <suffix added to WaveMon and channel names>    " << endl;
  exit(0);
  return;
}










































