/* -*- mode: c++; c-basic-offset: 4; -*- */
//   
//   Pulsar Sensitivity Monitor 
//   version: 6.2 (August 2008)
//    author: Giovanni Santostasi (gsantostasi@mcneese.edu)
//
//   Based on SenseMonitor
//   authors: Kevin C. Schlaufman (kcs149@psu.edu)
//            Patrick J. Sutton (psutton@ligo.caltech.edu)
//
///////////////////////////////////////////////////////////////////////////

  
//---------- Defined constants:
// #define CalPSD_Dump_Period 1000  // Time (sec) in lock between each dump of a calibrated AS_Q PSD.
#define DMTViewer_Period 43200  // Time (sec) over which to display range histories on DMTViewer.

//---------- Must include this first for some reason.
#include "PulsarMon.hh"      

#ifndef __CINT__

//---------- Include standard headers
#include <stdlib.h>
#include <iostream>
#include <string>
#include <iomanip>

//---------- Include LIGO-specific headers
#include "TSeries.hh"
#include "FSeries.hh"
#include "FSpectrum.hh"
#include "Dacc.hh"
#include "html/writer.hh" 
#include "html/Attrib.hh"
#include "html/align.hh"
#include "html/color.hh"
#include "html/document.hh"
#include "html/font.hh"
#include "html/hline.hh"
#include "html/image.hh"
#include "html/label.hh"
#include "html/link.hh"
#include "html/size.hh"
#include "html/style.hh"
#include "html/table.hh"
#include "html/text.hh"

#endif               // !def(__CINT__)

//---------- Include monitor-specific headers

#include "Integrand.hh"
#include "Integrate.hh"
#include "Range.hh"
#include "FOMpulsar.hh"
#include "FOM3unknown.hh"
#include "FOM4CumSpectrum.hh"
#include "FOM4CumSpectrumWrite.hh"



#ifndef __CINT__
//======================================  Generate the main routine.
EXECDAT(PulsarMon)
#endif               // !def(__CINT__)

  using namespace std;

const char *USAGE_INFO =
"\n"
"PulsarMon: Pulsar Sensitivity Monitor (Figure of Merit Estimation)"
"\n\n"
"Usage:\n\n"
"./PulsarMon [optional arguments] <IFO>\n\n"
"where <IFO> is one of H1, H2, or L1 and the optional arguments are:\n"
"                \n"
"-fmax #         Specify maximum frequency to include in range estimate.\n"  
"                Default 1400Hz.\n"
"                \n"
"-fmin #         Specify minimum frequency to include in range estimate.\n"
"                Default  20Hz.\n"
"                \n"
"-h, --help      Print this usage information.\n"
"                \n"
"                \n"
"-local          Local mode:  All output files are dumped in local directory\n"
"                instead of the default directory specified by $DMTHTMLOUT;\n"
"                plain-text log file renamed <GPS_Start_Time>.log.\n"
"                \n"
"-logfile <name> Name plain-text version of log file <name>.  Only workSearchs if\n"
"                -local option is also selected.  Default filename is\n"
"                <GPS_start_time>.log (-local mode) or\n"
"                <IFO>_PulsarMon_CumLog.txt (otherwise).\n"
"                \n"
"-max #          Specify maximum number of strides to process before program\n"
"                exits.\n"
"                \n"
"-n #            Specify number of sections to divide data into for \n" 
"                calculating PSD.  THIS NUMBER MUST DIVIDE 'stride' EVENLY!\n"
"                Default 15.\n"
"                \n"
"-OSCfile <file> Specify the OSC configuration file defining the conditions\n"
"                for the interferometer to be in lock.  If not specified,\n"
"                PulsarMon will attempt to open\n"
"                $PULSARMON_OSCCONF/PulsarMon_LockLoss.conf,\n"
"                then ./PulsarMon_LockLoss.conf.  PulsarMon will exit\n"
"                if no OSC configuration file can be found.\n"
"                \n"
"-screen         Update and error messages are sent to the screen instead of\n"
"                to the default files <IFO>_PulsarMon_Log.txt and \n"
"                <IFO>_PulsarMon_Errors.txt.\n"
"                \n"
"-stride #       Specify length in seconds of data to use for each range \n"
"                estimate.  Default 100.\n"
"                \n"
"-trend          Write range, cumulative volume, gain, and calibration-line-\n"
"                amplitude data to trend files.\n"
"                \n"
"-window <type>  Specify data window type, where <type> is: \n"
"                  hanning for Hanning window,\n"
"                  blackman for Blackman window,\n"
"                  flattop for FlatTop window (NOT RECOMMENDED),\n"
"                  hamming for Hamming window (NOT RECOMMENDED),\n"
"                  square for no window (NOT RECOMMENDED).\n"
"                Default Hanning.\n"
"                \n"
"-xmlcal <file>  REQUIRED. Use dynamical calibration based on a reference \n"
"                calibration file <file>. \n"
"                \n";

//----- Html Text
const char* const kHtmlTitle = "Known and Unknown Pulsar:"
			       "Sensitivity Monitor";
const char* const kHtmlAuthor = "Giovanni Santostasi (gsantostasi@mcneese.edu)" ;
const char* const kHtmlDescription =
   "This monitor calculate Figures Of Merit (FOM) for known and unknown pulsars)";
const char* const kHtmlLIGO =
   "Laser Interferometer Gravitational-wave Observatory ";
const char* const kHtmlLIGOLink = "http://www.ligo.caltech.edu";


//======================================  PulsarMon constructor
PulsarMon::PulsarMon(int argc, const char *argv[]) 
  : DatEnv(argc, argv), fatal_error(false), prev_endtime(0.0), 
    running_average(0.0), running_count(0.0), MaxStride(99999), 
    missing_strides(0), NStride(0), time_since_update(0),
    mCalibrate(0), history_alpha(0), history_beta(0), history_ampl(0), 
    history_ellipticity(0), history_crab(0), UpLastTime(false)
{

    //---------- Initialize structures.
    mOutput.error_log = true;
    mOutput.local = false;
    mOutput.screen_output = false;
    mOutput.trend = false;
    mOutput.write_log = true;
    Run_Par.beta = -1.0;
    Run_Par.T = 60.0;
    Run_Par.l_freq = 20.0;
    Run_Par.h_freq = 1400.0;
    Run_Par.num_ints = 1;// used to be 15 (1 gives freq resolution of 1/60 Hz)
    Run_Par.CalPSD_Dump_Period=0;
    R_Dat.alpha = 0.0;
    R_Dat.beta = 0.0;
    R_Dat.ampl = 0.0;
    R_Dat.ellipticity = 0.0;
    R_Dat.CrabTime =0.0;
    R_Dat.CrabTimePEM1 =0.0;
    R_Dat.CrabTimePEM2 =0.0;
    R_Dat.CrabTimePEM3=0.0;
    // other Pulsars          
    R_Dat.pulsar1Time =0.0;
    R_Dat.pulsar2Time =0.0;
    R_Dat.pulsar3Time =0.0;
    R_Dat.pulsar4Time =0.0;
    R_Dat.pulsar1TimePEM1 =0.0;
    R_Dat.pulsar2TimePEM1 =0.0;
    R_Dat.pulsar3TimePEM1 =0.0;
    R_Dat.pulsar4TimePEM1 =0.0;
    R_Dat.pulsar1TimePEM2 =0.0;
    R_Dat.pulsar2TimePEM2 =0.0;
    R_Dat.pulsar3TimePEM2 =0.0;
    R_Dat.pulsar4TimePEM2 =0.0;
    R_Dat.pulsar1TimePEM3 =0.0;
    R_Dat.pulsar2TimePEM3 =0.0;
    R_Dat.pulsar3TimePEM3 =0.0;
    R_Dat.pulsar4TimePEM3 =0.0;



    R_Dat.T5_ellipticity = 0.0;
    R_Dat.T15_ellipticity = 0.0;
    for(int i=0; i<5; i++)  R_Dat.T5_array[i] = 0.0;
    for(int i=0; i<15; i++)  R_Dat.T15_array[i] = 0.0;

// This cause the read-in function to ignore the request for any channels  that 
// aren't in the frame.

   getDacc().setIgnoreMissingChannel(true);

    //---------- Look for command-line arguments.
    //           First check if configuration file is being used.
    //           In this case the command line should read 
    //           PulsarMon  -config <filename>
    //           If file <filename> does not exist or cannot be parsed 
    //           then exit with error message.
    //           If not using config file, make sure we have some arguments and that
    //           the last one does not start with a hyphen.
    if ((argc == 3) && (!strcmp("-config", argv[1]))) {
        Run_Par.config_file_name = argv[2];
        Configure(Run_Par.config_file_name);
    } else {
        Configure(argc, argv); 
    }


    //---------- Send monitor name to MonServer.
    std::string MonitorName = "PulsarMon_"; 
    MonitorName += Run_Par.IFO;
    // MonitorName += "_TEST";     
    MonServer::setServerName(MonitorName.c_str());

    //---------- Do some elementary checks on the parameters.
    VerifyParameters(Run_Par,fatal_error);
    //---------- Finish setup of monitor.
    if (!fatal_error) {

        //-------- Check to see if running locally.  If not, dump html 
        //         files to $DMTHTMLOUT instead of to local directory.
	//         Plain-text log file is named in ProcessData() method.
        if (!mOutput.local) {
	    mOutput.html_dir = getenv("DMTHTMLOUT");
            if (mOutput.html_dir) {
                mOutput.dmtviewer_file_name = string(mOutput.html_dir) + "/";
		mOutput.error_file_name = string(mOutput.html_dir) + "/";
		mOutput.log_file_name = string(mOutput.html_dir) + "/";
		mOutput.revolver_file_name = string(mOutput.html_dir) + "/";
		mOutput.summary_file_name = string(mOutput.html_dir) + "/";
	    } else { 
		std::cerr << "PulsarMon WARNING: Environment variable "
			  << "DMTHTMLOUT not set; switching to\n";
		std::cerr << "                      -local mode (all output "
			  << "files go to the local directory).\n";
		mOutput.local = true;
	    }
	}
        mOutput.dmtviewer_file_name += Run_Par.IFO + "_DMTViewer_Data.txt";
        mOutput.error_file_name += Run_Par.IFO;
        mOutput.log_file_name += Run_Par.IFO;
        mOutput.revolver_file_name += Run_Par.IFO;
        mOutput.revolver_file_name += "_PulsarMon_Summary.revolver.html";
        mOutput.error_file_name += "_PulsarMon_Errors.txt";
	mOutput.error_file_link = Run_Par.IFO + "_PulsarMon_Errors.txt";
        mOutput.log_file_name += "_PulsarMon_Log.txt";
	mOutput.log_file_link = Run_Par.IFO + "_PulsarMon_Log.txt";  
        if (!mOutput.local) mOutput.cumlog_file_link = Run_Par.IFO + "_PulsarMon_CumLog.txt"; 
	mOutput.summary_file_name += "index.html";

        //-------- Divert log and error messages to files, if desired(default).
	FILE* tmp;
        if (mOutput.error_log) {
	    tmp = freopen(mOutput.error_file_name.c_str(), "w", stderr);
	    if (tmp == NULL) perror("Unable to divert stderr");
	}
        if (mOutput.write_log) {
	    tmp = freopen(mOutput.log_file_name.c_str(), "w", stdout);	
	    if (tmp == NULL) perror("Unable to divert stdout");
	}

        //---------- Set up array that will hold data for DMTViewer history 
        //           FSeries.  Used to insure that fixed length of data is 
        //           displayed.  Initialize data to -1 to indicate PulsarMon 
        //           was not running previously.
        tserieslength = (int)( DMTViewer_Period / Run_Par.T);
        calvalues = new double[tserieslength];
        crabvalues=new double[tserieslength];
        pulsar1values=new double[tserieslength];
        pulsar2values=new double[tserieslength];
        pulsar3values=new double[tserieslength];
        pulsar4values=new double[tserieslength];
        amplvalues = new double[tserieslength];
        alphavalues = new double[tserieslength];
        betavalues = new double[tserieslength];
        //---------- Initialize TSeries using data from previous run, if possible.
      

        ifstream dmtviewer_data;
        dmtviewer_data.open(mOutput.dmtviewer_file_name.c_str());
        if (dmtviewer_data.fail()) {
            for (int i=0; i<tserieslength; i++) {
                calvalues[i] = -1.0;
                crabvalues[i]=-1.0;
                pulsar1values[i]=-1.0;
                pulsar2values[i]=-1.0;
                pulsar3values[i]=-1.0;
                pulsar4values[i]=-1.0;
                amplvalues[i] = -1.0;
                alphavalues[i] = -1.0;
                betavalues[i] = -1.0;
            }
        } else {
            //:KLUDGE:  This could go wrong if old history from previous run is longer than new one.
            //:KLUDGE:  This could go wrong if there is a long time gap between runs of program.
            //:KLUDGE:  Check on dumps of 0 vs -1 vs -2 and also cumulative vol stat.
            int i=0;
            while (dmtviewer_data >> calvalues[i]) { 
                dmtviewer_data >> crabvalues[i];
                dmtviewer_data >> pulsar1values[i];
		dmtviewer_data >> pulsar2values[i];
		dmtviewer_data >> pulsar3values[i];
		dmtviewer_data >> pulsar4values[i]; 
                dmtviewer_data >> amplvalues[i];
                dmtviewer_data >> alphavalues[i];
                dmtviewer_data >> betavalues[i];
                ++i;
            }
            //----- Reset latest values to -2 to indicate that histories have been read from a file.
            calvalues[i-1] = -2.0;
            crabvalues[i-1]=-2.0;
	    pulsar1values[i-1]=-2.0;
	    pulsar2values[i-1]=-2.0;
	    pulsar3values[i-1]=-2.0;
	    pulsar4values[i-1]=-2.0;
            amplvalues[i-1] = -2.0;
            alphavalues[i-1] = -2.0;
            betavalues[i-1] = -2.0;
            //----- Check that our new histories are of the desired length.
            if (i != tserieslength) {
                cerr << "PulsarMon WARNING:  Stored DMTViewer history from previous run not of ";
                cerr << "required length.  Re-initializing DMTViewer histories.\n";
                for (int i=0; i<tserieslength; i++) {
                    calvalues[i] = -1.0;
                    crabvalues[i]=-1.0;
		    pulsar1values[i]=-1.0;
		    pulsar2values[i]=-1.0;
		    pulsar3values[i]=-1.0;
		    pulsar4values[i]=-1.0;
                    amplvalues[i] = -1.0;
                    alphavalues[i] = -1.0;
                    betavalues[i] = -1.0;
                }
            }
        }
        dmtviewer_data.close();

        //-------- Set window type and intialize PSD object.
        if (Run_Par.window_name == "hamming") {
            ham = new Hamming;
            psd = new PSD(ham, Run_Par.num_ints);
	   
        } else if (Run_Par.window_name == "hanning") {
            han = new Hanning;
            psd = new PSD(han, Run_Par.num_ints);
 
        } else if (Run_Par.window_name == "flattop") {
            flat = new FlatTop;
            psd = new PSD(flat, Run_Par.num_ints);
 
        } else if (Run_Par.window_name == "blackman") {
            black = new Blackman;
	
        } else if (Run_Par.window_name == "square") {
            psd = new PSD("square", Run_Par.num_ints);
   
        } else {
            Run_Par.window_name = "hanning";
            han = new Hanning;
            psd = new PSD(han, Run_Par.num_ints);
           }
         
         

        //---------- Set the time stride.
        getDacc().setStride(Interval(Run_Par.T));
	setStrideAlignment(60, 0.0);

        //---------- Get AS_Q channel; decimate to maximum frequency 4096Hz.
	//           (Keep frequencies above 2kHz for the calibrated PSD dumps.)
        getDacc().addChannel((Run_Par.mChannel).c_str(),2);
        //----------------Get PEM channels: 
        // This should be tuned to IFO and PEM channel choise


            if (Run_Par.IFO == "H1") { 
		Run_Par.IFOPEM="H0";        
            } else if (Run_Par.IFO == "L1") { 
		Run_Par.IFOPEM="L0";
            } else if (Run_Par.IFO == "H2") {
		Run_Par.IFOPEM="H0" ;    
	    }
        cerr<<"here 1"<<endl;

        Run_Par.PEMChannel1  = Run_Par.IFOPEM; 
        Run_Par.PEMChannel1 += ":PEM-LVEA_SEISX";
        Run_Par.PEMChannel2  = Run_Par.IFOPEM;
        Run_Par.PEMChannel2 += ":PEM-LVEA_SEISY";
        Run_Par.PEMChannel3  = Run_Par.IFOPEM;
        Run_Par.PEMChannel3 += ":PEM-LVEA_SEISZ";
        //std::string PEMChannel4 = ":PEM-LVEA_MAGX" ;
        //std::string PEMChannel5 = ":PEM-MX_V1" ;
        getDacc().addChannel((Run_Par.PEMChannel1).c_str(),1);
        getDacc().addChannel((Run_Par.PEMChannel2).c_str(),1);
        getDacc().addChannel((Run_Par.PEMChannel3).c_str(),1);
        

        cerr<<"here 2"<<endl;
        
        //---------- Initialize FDEasyCalibrate object.  
        //           This should be in monitor constructor (ie, before any data 
        //           is requested) because FDEasyCalibrate adds new channels to
        //           the Dacc. 
        //           KLUDGE: Hard-wire for generating new alphas, betas.
        //           KLUDGE: Hard-wire requested frequencies.
        mCalibrate = new FDEasyCalibrate(
                            &getDacc(),
                            (Run_Par.xml_file_name).c_str(),
                            true,
                            0.0,
                            (Run_Par.num_ints)/(Run_Par.T),
                            ((int) (4096.0-0.0)*(Run_Par.T)/(Run_Par.num_ints)+1) 
                     );

	//---------- Initialize OSC & read config file.
        mOsclist = new OperStateCondList(getDacc());
        mOsclist->readConfig((Run_Par.osc_file_name).c_str());
        mOsclist->ignoreAllExcept(Run_Par.IFO);
        

	if (mOutput.trend)
	{
	    //---------- Intialize Trend frame.
	    mTrend.setIFO(Run_Par.IFO.c_str());
	    mTrend.setName(MonitorName.c_str());
            mTrend.setType(Trend::kMinute);

	    //---------- Add trend channels.  Force calibration-line-related 
	    //           trends even if not tracking a line (we don't want to 
	    //           change the trends that are being written during a science
	    //           run, as this causes the trend writer to crash). 
	    
         
              




            std::string trendName;
	    trendName = Run_Par.IFO + ":DMT-PULS_500HZ_ELLIPTICITY";
            mTrend.addChannel(trendName.c_str());  
            trendName = Run_Par.IFO + ":DMT-PULS_CRAB_IT_SPINDOWN";  
            mTrend.addChannel(trendName.c_str());


      
            trendName = Run_Par.IFO + ":DMT-PULS_J1952_plus_3252_IT_SPINDOWN";
            mTrend.addChannel(trendName.c_str());  
            trendName = Run_Par.IFO + ":DMT-PULS_J1913_plus_1011_IT_SPINDOWN";
            mTrend.addChannel(trendName.c_str());  
            trendName = Run_Par.IFO + ":DMT-PULS_J0737_minus_3039A_IT_SPINDOWN";
            mTrend.addChannel(trendName.c_str());
	    trendName = Run_Par.IFO + ":DMT-PULS_J0537_minus_6910_IT_SPINDOWN";
            mTrend.addChannel(trendName.c_str());

            // trend name problem here?
            cerr<<"here 3"<<endl;
           // trendName = Run_Par.IFO + ":DMT-PULS_CRAB_IT_" + Run_Par.IFOPEM + ":PEM-LVEA_SEISX";  
          //  mTrend.addChannel(trendName.c_str());
           // trendName = Run_Par.IFO + ":DMT-PULS_J1952_plus_3252_IT_" + Run_Par.IFOPEM + ":PEM-LVEA_SEISX";
         //   mTrend.addChannel(trendName.c_str());  
         //   trendName = Run_Par.IFO + ":DMT-PULS_J1913_plus_1011_IT_" + Run_Par.IFOPEM + ":PEM-LVEA_SEISX";
         //   mTrend.addChannel(trendName.c_str());  
         //   trendName = Run_Par.IFO + ":DMT-PULS_J0737_minus_3039A_IT_" + Run_Par.IFOPEM + ":PEM-LVEA_SEISX";
         //   mTrend.addChannel(trendName.c_str());
	//    trendName = Run_Par.IFO + ":DMT-PULS_J0537_minus_6910_IT_" + Run_Par.IFOPEM + ":PEM-LVEA_SEISX";
         //   mTrend.addChannel(trendName.c_str()); 
        
              
            cerr<<"here 4"<<endl;

	    trendName = Run_Par.IFO + ":DMT-PULS_CAL_LINE_AMPL_DARM_ERR";
	    mTrend.addChannel(trendName.c_str());
	    //---------- Trends to carry alpha, alpha*beta 
	    //           (names requested by Inspiral group).
	    trendName = Run_Par.IFO + ":DMT-PULS_CAL_CAV_FAC";
	    mTrend.addChannel(trendName.c_str());
	    trendName = Run_Par.IFO + ":DMT-PULS_CAL_OLOOP_FAC";
	    mTrend.addChannel(trendName.c_str());
            //---------- Tell manager to make list of trended channels.
            mTrend.writeIndex();
           
             cerr<<"here 5"<<endl;
	}
           

    }   //---------- !fatal_error

    //---------- Flush any lingering data
    fflush(stderr);
    fflush(stdout);
      

}

// --End of PulsarMon constructor
//======================================  PulsarMon destructor
PulsarMon::~PulsarMon() 
{
    //---------- No pointers are assigned if a fatal error is registered.
    if (!fatal_error) {
        //---------- Export DMTViewer data to a file to be read if monitor is restarted.
        //           Name of file set elsewhere; uses mOutput.html_dir if defined.
        ofstream dmtviewer_data;
        dmtviewer_data.open(mOutput.dmtviewer_file_name.c_str());
        for (int i=0; i<tserieslength; i++) 
	{
                dmtviewer_data << calvalues[i] << "  ";
                dmtviewer_data << crabvalues[i] << "  ";
		dmtviewer_data << pulsar1values[i] << "  ";
		dmtviewer_data << pulsar2values[i] << "  ";
		dmtviewer_data << pulsar3values[i] << "  ";
		dmtviewer_data << pulsar4values[i] << "  ";
                dmtviewer_data << amplvalues[i] << "  ";              
                dmtviewer_data << alphavalues[i] << "  ";
                dmtviewer_data << betavalues[i] << "  ";
                dmtviewer_data << endl; 
                //cerr<<i<<endl;
               
        } 
        dmtviewer_data.close();


        //----- Pointers assigned in PulsarMon::PulsarMon().
        //---------- These pointers are always assigned.
        delete [] calvalues;
        delete [] crabvalues;
	delete [] pulsar1values;
	delete [] pulsar2values;
	delete [] pulsar3values;
	delete [] pulsar4values;
        delete [] amplvalues;
        delete [] alphavalues;
        delete [] betavalues;
        //---------- Delete the window pointer (if any).
        if (Run_Par.window_name == "hamming") delete ham;
        if (Run_Par.window_name == "hanning") delete han;
        if (Run_Par.window_name == "flattop") delete flat;
        if (Run_Par.window_name == "blackman") delete black;
        delete psd;
        delete mOsclist;
	delete mCalibrate;

        //----- Pointers assigned in PulsarMon::ProcessData().
	    delete history_ellipticity;
            delete history_crab;
            delete history_pulsar1;
            delete history_pulsar2;
            delete history_pulsar3;
            delete history_pulsar4;
	    delete history_ampl;
	    delete history_alpha;
	    delete history_beta;

        //----- Close files.
	if (mOutput.trend) mTrend.close();
        mOutput.cumlog_file.close(); 
    }
    //----- Send "finished happily" message.
    cout << "PulsarMon MESSAGE: PulsarMon is finished after "
	 << NStride << " strides.\n";
}


//======================================  Frame processing function
void
PulsarMon::ProcessData(void) {
            

	//---------- Get current GPS time and check that it is an integer.
        Time time = getDacc().getFillTime();
        if (time.getS() != time.totalS()) {
            cerr << "PulsarMon WARNING: GPS start time not an integer: ";
            cerr << time.totalS() << endl;
        }

//:KLUDGE:GAPS      
	//---------- Check for gap in data.
        if ( (time.totalS() != prev_endtime) && (NStride != 0) ) { 
          char msg[150];
          sprintf(
              msg,"Missing data between GPS times (%0.3f, %0.3f).\n",
              prev_endtime,time.totalS()
          );
	  std::cerr << "PulsarMon ERROR: " << msg;
          missing_strides = int(1+(time.totalS()-prev_endtime)/Run_Par.T);
        } else { 
	  missing_strides = 0;
	}
        prev_endtime = time.totalS();
        prev_endtime += Run_Par.T; 
//:END:KLUDGE:GAPS      

	//---------- Calculate start time of histories to DMTViewer.
	//           Note:  TSeries start time = "time" - "DMTViewer_Period",
        //           where DMTViewer_Period is a defined quantity copied to 
        //           tserieslength. 
	Time history_start = time;
	history_start -= (tserieslength-1)*(Run_Par.T);
        
	//---------- Set cumlog_file_name.  Choices in order of preference are 
	//           (1) "$DMTHTMLOUT/<IFO>_PulsarMon_CumLog.txt" if 
        //               -local option NOT selected (default).
	//           (2) User's choice if -local and -logfile options selected.
	//           (3) Use <GPS_start_time>.log if only -local selected. 
        if (!mOutput.local) {  
	    mOutput.cumlog_file_name = string(mOutput.html_dir) + "/" + Run_Par.IFO +
				    "_PulsarMon_CumLog.txt";
	}
        if ( (mOutput.cumlog_file_name).empty() && !NStride ) {
            char temp_file_name[50]; 
            sprintf(temp_file_name,"%ld",time.getS());
            mOutput.cumlog_file_name = temp_file_name;
	    mOutput.cumlog_file_name += ".log";
        }
	
          //---------- Things to do once only.
         	        
                if (NStride == 0) {
            // Here also initialization of exponential averaging
                  N_unlock = 0;                    
                //cerr<<"Stride N="<<NStride<<endl;
            //---------- Need some data for initializing certain storage.
            //           Doesn't matter if IFO is locked.
                 ts_asq = getDacc().refData((Run_Par.mChannel).c_str());//time sequence ?

            if (Run_Par.IFO == "H1") { 
		Run_Par.IFOPEM="H0";         
            } else if (Run_Par.IFO == "L1") { 
		Run_Par.IFOPEM="L0";
            } else if (Run_Par.IFO == "H2") {
		Run_Par.IFOPEM="H0";    
	    }


                Run_Par.PEMChannel1  = Run_Par.IFOPEM; 
                Run_Par.PEMChannel1 += ":PEM-LVEA_SEISX";
                Run_Par.PEMChannel2  = Run_Par.IFOPEM;
                Run_Par.PEMChannel2 += ":PEM-LVEA_SEISY";
                Run_Par.PEMChannel3  = Run_Par.IFOPEM;
                Run_Par.PEMChannel3 += ":PEM-LVEA_SEISZ";
                 //std::string PEMChannel2 = "L0:PEM-LVEA_MAGX";
                 //std::string PEMChannel3 = "L0:PEM-MX_V1" ;
   
                ts_PEM1 = getDacc().refData((Run_Par.PEMChannel1).c_str());//time sequence
                ts_PEM2 = getDacc().refData((Run_Par.PEMChannel2).c_str());//time sequence 
                ts_PEM3 = getDacc().refData((Run_Par.PEMChannel3).c_str());//time sequence 
                
	        //---------- Calcuate PSD.
	        psd->generate(PSDed_data,ts_asq);
                //psd->generate(PSDed_data,ts_PEM1);
                //psd->generate(PSDed_data,ts_PEM2);
                //psd->generate(PSDed_data,ts_PEM3);
            //---------- Output run parameters.  
	    //----- Set starttime 
 	    starttime = time.getS();
            //cerr<<starttime<<endl;

	    //----- Open logfile
	    mOutput.cumlog_file.open((mOutput.cumlog_file_name).c_str(), std::ios::app);

          //----- Dump parameters to html log file and cout. 
            DumpParameters(cout);
            DumpParameters(mOutput.cumlog_file);

            DumpReportHeader(cout);
            DumpReportHeader(mOutput.cumlog_file);

            if (!mOutput.local) {  
                //----- Set up two-week summary file
	        std::string summary_name = string(mOutput.html_dir) + "/" + Run_Par.IFO +
		    		           "_summary.txt";
	        mRangeSumm.init(summary_name, time);
	    }

            //---------- Write histories to DMTViewer.
            //----- Initialize and subscribe history of
	    // 	    (ellipticity and Time to reach EC for Crab Pulsar without calibration line). 
            
                //----- Initialize and subscribe history of
	  	//      (range with calibration line).
                history_ellipticity = new TSeries(history_start,Run_Par.T,tserieslength,calvalues);
		std::string DMTViewer_name = Run_Par.IFO;
                DMTViewer_name += " Ellipticity for 500 Hz, 1kpc, 1year IT Pulsar with Cal. Line";
                serveData(DMTViewer_name.c_str(), history_ellipticity);
                
                history_crab = new TSeries(history_start,Run_Par.T,tserieslength,crabvalues);
                DMTViewer_name = Run_Par.IFO;
                DMTViewer_name += " EC lim. Integration Time for Crab Pulsar with Cal. Line (days)";
                serveData(DMTViewer_name.c_str(), history_crab);
		history_pulsar1 = new TSeries(history_start,Run_Par.T,tserieslength,pulsar1values);
                DMTViewer_name = Run_Par.IFO;
                DMTViewer_name += " EC lim. Integration Time for Pulsar1 with Cal. Line (days)";
                serveData(DMTViewer_name.c_str(), history_pulsar1);
			history_pulsar2 = new TSeries(history_start,Run_Par.T,tserieslength,pulsar2values);
                DMTViewer_name = Run_Par.IFO;
                DMTViewer_name += " EC lim. Integration Time for Pulsar2 with Cal. Line (days)";
                serveData(DMTViewer_name.c_str(), history_pulsar2);
		history_pulsar3 = new TSeries(history_start,Run_Par.T,tserieslength,pulsar3values);
                DMTViewer_name = Run_Par.IFO;
                DMTViewer_name += " EC lim. Integration Time for Pulsar3 with Cal. Line (days)";
                serveData(DMTViewer_name.c_str(), history_pulsar3); //change EC to Spin-Down
			history_pulsar4 = new TSeries(history_start,Run_Par.T,tserieslength,pulsar4values);
                DMTViewer_name = Run_Par.IFO;
                DMTViewer_name += " EC lim. Integration Time for Pulsar4 with Cal. Line (days)";
                serveData(DMTViewer_name.c_str(), history_pulsar4);
                //----- Initialize and subscribe history of calibration-line amplitude.
                history_ampl = new TSeries(history_start,Run_Par.T,tserieslength,amplvalues);
                DMTViewer_name = Run_Par.IFO;
                DMTViewer_name += " Calibration Line Amplitude (DARM_ERR counts)";
                serveData(DMTViewer_name.c_str(), history_ampl);
                //----- Initialize and subscribe history of alpha parameter.
                history_alpha = new TSeries(history_start,Run_Par.T,tserieslength,alphavalues);
                DMTViewer_name = Run_Par.IFO;
                DMTViewer_name += " Alpha From Calibration Line";
                serveData(DMTViewer_name.c_str(), history_alpha);
                //----- Initialize and subscribe history of beta parameter.
                history_beta = new TSeries(history_start,Run_Par.T,tserieslength,betavalues);
                DMTViewer_name = Run_Par.IFO;
                DMTViewer_name += " Beta From Input Matrix";  
                serveData(DMTViewer_name.c_str(), history_beta);

  //---------- Make fake strain_noise_ampl for DMTViewer initialization. 
         // this also initialize exponential averaging

	    strain_noise_ampl = PSDed_data;
            DMTViewer_name = Run_Par.IFO;
	    DMTViewer_name += " First Calibrated Noise Amplitude Spectrum (Hz^-0.5)";
	    serveData(DMTViewer_name.c_str(), &strain_noise_ampl);

            //initialization of FOM4 parameters
            CumS_Old=PSDed_data;
//	    DMTViewer_name = Run_Par.IFO;
//	    DMTViewer_name += " Pulsar FOM1: hEC (strain amplitude at EC limit)";

//:END:KLUDGE: 
        }


        	//---------- Check that IFO is locked before proceeding.
     //	std::string osc_cond1 = Run_Par.IFO + ":Both_arms_locked_strict";
     //	std::string osc_cond2 = Run_Par.IFO + ":Both_arms_locked_common_mode";
    //	std::string osc_cond1 = Run_Par.IFO + ":Both_arms_locked";
     //	std::string osc_cond2 = Run_Par.IFO + ":Both_arms_locked";
		// K.Riles - replaced "ScienceMode" with "IFO_UP" 10/14/05
		//            std::string osc_cond1 = Run_Par.IFO + ":SV_ScienceMode";
		//            std::string osc_cond2 = Run_Par.IFO + ":SV_ScienceMode";

//CHANGE HERE FOR DOL (1)

       std::string osc_cond1 = Run_Par.IFO + ":Both_arms_locked_strict_cm";
       std::string osc_cond2 = Run_Par.IFO + ":Both_arms_locked_strict_cm";

// here are conditions for lock, check this to control behavior after lockloss
///////////////////////////////////////////////////////////////////////////////
         if (mOsclist->satisfied(osc_cond1.c_str()) && 
	     mOsclist->satisfied(osc_cond2.c_str()) )  //CHANGE HERE FOR DOL (1)
	 {
	    //---------- Get AS_Q data to TSeries.
	    ts_asq = getDacc().refData((Run_Par.mChannel).c_str());
            //..........Get PEM data to TSeries


            if (Run_Par.IFO == "H1") { 
		Run_Par.IFOPEM="H0";        
            } else if (Run_Par.IFO == "L1") { 
		Run_Par.IFOPEM="L0";
            } else if (Run_Par.IFO == "H2") {
		Run_Par.IFOPEM="H0";     
	    }

                 Run_Par.PEMChannel1  = Run_Par.IFOPEM; 
                 Run_Par.PEMChannel1 += ":PEM-LVEA_SEISX";
                 Run_Par.PEMChannel2  = Run_Par.IFOPEM;
                 Run_Par.PEMChannel2 += ":PEM-LVEA_SEISY";
                 Run_Par.PEMChannel3  = Run_Par.IFOPEM;
                 Run_Par.PEMChannel3 += ":PEM-LVEA_SEISZ";
              // std::string PEMChannel2 = "L0:PEM-LVEA_MAGX"; 
              // std::string PEMChannel3 = "L0:PEM-MX_V1" ;
   
                ts_PEM1 = getDacc().refData((Run_Par.PEMChannel1).c_str());//time sequence
                ts_PEM2 = getDacc().refData((Run_Par.PEMChannel2).c_str());//time sequence 
                ts_PEM3 = getDacc().refData((Run_Par.PEMChannel3).c_str());//time sequence 
  

            //---------- Calcuate PSD.
	    psd->generate(PSDed_data,ts_asq);
            psd->generate(PSDed_PEM1,ts_PEM1);
            psd->generate(PSDed_PEM2,ts_PEM2);
            psd->generate(PSDed_PEM3,ts_PEM3);
            

               //---------- Apply calibration (extracts subsection of PSD 
	    //           for which we have calibration info).
            mCalibrate->UpdateResponseFunction();
	    R_Dat.alpha = mCalibrate->GetAlpha();
	    //std::cout << "alpha = " << R_Dat.alpha << std::endl;
	    R_Dat.beta = mCalibrate->GetBeta();
	    //std::cout << "beta  = " << R_Dat.beta << std::endl;
            PSDed_data = mCalibrate->Apply(PSDed_data);

            //---------- KLUDGE: Don't yet have "get" method for retrieving 
            //           line amplitude from FDCalibrate.
	    R_Dat.ampl = 0.0;

	    //---------- Compute strain_noise_ampl from PSDed_data, for DMTViewer.
            //           This code effectively duplicates the Calibrated PSD dump code.
            //           I should remove the latter.

	    //----- Set up array to hold values of calibrated AS_Q PSD.
            float* strainvalues = new float[PSDed_data.getNStep() + 1];
            float* ampPEM1 = new float[PSDed_PEM1.getNStep() + 1];
            float* ampPEM2 = new float[PSDed_PEM2.getNStep() + 1];
            float* ampPEM3 = new float[PSDed_PEM3.getNStep() + 1];
           
	    //----- Get values of calibrated PSD.
            PSDed_data.getData(PSDed_data.getNStep() + 1, strainvalues);
            PSDed_PEM1.getData(PSDed_PEM1.getNStep() + 1, ampPEM1);
            PSDed_PEM2.getData(PSDed_PEM2.getNStep() + 1, ampPEM2);
            PSDed_PEM3.getData(PSDed_PEM3.getNStep() + 1, ampPEM3);

            //----- Specify arm length in nm for conversion of PSD units to strain:
            float arm_length = 1.0;  
            if (Run_Par.IFO == "H2") { 
                arm_length = 2.0e12;  
            } else if ((Run_Par.IFO == "L1") || (Run_Par.IFO == "H1")) { 
                arm_length = 4.0e12;  
            }

            //----- Compute sqrt of power spectrum, converted to units of strain/Hz^1/2.  
            for (unsigned int j=0; j<PSDed_data.getNStep()+1; j++) {
                strainvalues[j] = (float) pow((double) strainvalues[j],0.5)/arm_length;
            }
 
            // Exponential averaging  (this is done because we use just one 
	    // stride to produce PSD. We average over time using previous 
	    // stride PSD information. Initialization is done above in
	    // Things to do once.

	    if (UpLastTime) // Check if IFO was locked during the previous stride
           // otherwise don't do averaging
          {
	        float lambda=1.3; 
               // the "decay constant" in units of time in terms of lambda strides 
               //variables for exponential averaging
		double eLambda = exp(-lambda);
		float* yold = new float[PSDed_data.getNStep() + 1]; 
		strain_noise_ampl.getData(PSDed_data.getNStep() + 1, yold);
		for (size_t j=0; j<PSDed_data.getNStep()+1; j++) {
		    // ynew[j] = (1-exp(-lambda))*x[j] + exp(-lambda)*yold[j];
		    strainvalues[j] = (1-eLambda)*strainvalues[j] + 
			              eLambda*yold[j];
		}
		delete [] yold;
	    }
	    UpLastTime = true; // Comfirms IFO was locked during previous stride

            //----- Copy data to strain_noise_ampl and clean up.

            strain_noise_ampl = PSDed_data;
            strain_noise_ampl.setData(PSDed_data.getNStep() + 1, strainvalues);

  //--------Compute Pulsars FOMs------------ 
 //---------Compute test Epsilon for Pulsar at 1kpc, 500 Hz, 1 year integration time
//--------------------------------------------o-----------------------------------------
// Antenna Parameters
    float AntPar=10.8;// new antenna parameter
    float CrabAntPar=1;//initialization of Parameter
    float pulsar1AntPar=1;// initialization of Parameter
    float pulsar2AntPar=1;// initialization of Parameter
    float pulsar3AntPar=1;// initialization of Parameter
    float pulsar4AntPar=1;// initialization of Parameter

     // Set the antenna pattern parameter for Crab IT for each 
     // observatory
            if (Run_Par.IFO == "H1") { 
		CrabAntPar = 0.93;  
                pulsar1AntPar = 0.93; 
                pulsar2AntPar=0.93;
                pulsar3AntPar=0.93;
                pulsar4AntPar=0.93;
                
            } else if (Run_Par.IFO == "L1") { 
		CrabAntPar = 1.01;  
                pulsar1AntPar = 0.93; 
                pulsar2AntPar=0.93;
                pulsar3AntPar=0.93;
                pulsar4AntPar=0.93;
            } else if (Run_Par.IFO == "H2") {
		CrabAntPar = 0.93;
                pulsar1AntPar = 0.93; 
                pulsar2AntPar=0.93;
                pulsar3AntPar=0.93;
                pulsar4AntPar=0.93;
                
	    }
	  

           // Global constants and Parameters for Pulsars  (HERE: maybe this should be in FOMPulsars.hh)
           //float I= 1e45;// moment of Inertia for test Pulsar
           //float r= 1;// distance in kpc for test Pulsar
           float freqP= 250;// rotation frequency of test Pulsar
           float* freqIFO = new float[PSDed_data.getNStep() + 1];

           float fIFOmin=PSDed_data.getLowFreq();//minimum frequency of calibrated strain noise
           float fIFOstep=PSDed_data.getFStep();  // frequency step of calibrated strain noise
           for (unsigned int j=0; j<PSDed_data.getNStep()+1; j++) {
            freqIFO[j]=fIFOmin+fIFOstep*j;//frequency series of IFO
            }
           int k=0;
           while (freqIFO[k] <= freqP*2) {
           k++;
           } 


           //PEMChannel1
           float* freqPEM1 = new float[PSDed_PEM1.getNStep() + 1];
           float fPEM1min=PSDed_PEM1.getLowFreq(); // f-min of calibrated PEM1
           //float fPEM1max=PSDed_PEM1.getHighFreq();// f-max of calibrated PEM1
           float fPEM1step=PSDed_PEM1.getFStep();  // f-step of calibrated PEM1
           for (unsigned int j=0; j<PSDed_PEM1.getNStep()+1; j++) {
	       freqPEM1[j]=fPEM1min+fPEM1step*j; //frequency series of IFO
	   }
           //PEMChannel2
           float* freqPEM2 = new float[PSDed_PEM2.getNStep() + 1];
           float fPEM2min=PSDed_PEM2.getLowFreq();// f-min of calibrated PEM2
           //float fPEM2max=PSDed_PEM2.getHighFreq();//f-max of calibrated PEM2
           float fPEM2step=PSDed_PEM2.getFStep();  // f-step of calibrated PEM2
           for (unsigned int j=0; j<PSDed_PEM2.getNStep()+1; j++) {
	       freqPEM2[j]=fPEM2min+fPEM2step*j;//frequency series of IFO
	   }
           //PEMChannel3
           float* freqPEM3 = new float[PSDed_PEM3.getNStep() + 1];
           float fPEM3min=PSDed_PEM3.getLowFreq();// f-min of calibrated PEM3
           //float fPEM3max=PSDed_PEM3.getHighFreq();// f-max of calibrated PEM3
           float fPEM3step=PSDed_PEM3.getFStep();  // f-step of calibrated PEM1
           for (unsigned int j=0; j<PSDed_PEM3.getNStep()+1; j++) {
	       freqPEM3[j]=fPEM3min+fPEM3step*j;//frequency series of IFO
	   }

	   //---------------------------  Calculate Ellipticity for test Pulsar
	   float h_500 = strainvalues[k];// strain noise at twice rot freq.
	   // double yearT = 365*60*60*24;//seconds in a year
	   float h_500_y =AntPar*h_500 / pow( 365*24*60*60 , 0.5);
	   // strain noise after one year of integration 
	   // (CHECK NEW ANTENNA PATTERN CORRECTION)                
	   float Ellipt_ref = 9.5e-6 * h_500_y / 1e-23 * pow(2*freqP/1000, -2);
    // Ellipticity for a pulsar at 1kpc, 500Hz, 1 year integration time
            R_Dat.ellipticity=Ellipt_ref;// reference ellipticity
     //----Calculate Integration Time to reach spin-down limit for Crab Pulsar---------------------------------
     // Crab Parameters
             float freqCrab0=30.059224; // frequency of the Crab in Hz at initial time Feb 15 1982
             float fdotCrab=-3.81008e-10;// change in frequency of the Crab at initial time Feb 15 1982
	     float timeRef=66571200;	// this value is the value of GPS time from the initial date of
					// Jan 1 1980 to the first date available on the frequency data
					// of pulsars from online catalogs which is chosen as the referrence
					// date for referrence data on freqCrab0 and fdotCrab

	     //timeRef below is the initial time when freqCrab0 and fdotCrab were measured

	     float freqCrab = freqCrab0 + fdotCrab*(time.totalS() - timeRef);
             
             //float difft=time.totalS()-840000000;
             //cerr<<freqCrab<<endl;
             float rCrab=2.00;//distance in kpc of the Crab                
             int kk=0;
             while (freqIFO[kk]<= 2*freqCrab)
             {
             kk++;
              } 

/////////////////Noise for PEM1 at Crab IT freq
               int pp1=0;
               int pp2=0;
               int pp3=0;
               
             while (freqPEM1[pp1]<= 2*freqCrab)
             {
             pp1++;
              }
             while (freqPEM2[pp2]<= 2*freqCrab)
             {
             pp2++;
              }
              while (freqPEM3[pp3]<= 2*freqCrab)
             {
             pp3++;
              }
              
      // this gives frequency bigger than Crab
     // float Snoise_IFO_CrabP=pow(strainvalues[kk],2);
    // PSD at 2 times the CrabFreq (above the real frequency)
             float hnoise_IFO_Crab=strainvalues[kk-1];
             float PdotCrab= - pow(freqCrab,-2) * fdotCrab;// Pdot for the Crab
             float h_EC_Crab=5.7e-24/(rCrab)*pow(2 * freqCrab /1000 * PdotCrab / 1e-13 , 0.5);// strain at EC limit
             float TEC_Crab = 
             pow(CrabAntPar*AntPar, 2 )*pow(hnoise_IFO_Crab/h_EC_Crab,2)/60/60/24 ;

             float PEM1noise_Crab=ampPEM1[pp1-1];
             float PEM2noise_Crab=ampPEM2[pp2-1];
             float PEM3noise_Crab=ampPEM3[pp3-1];

          ////here get some strain for spectrogram
          //ofstream spec1;
          //  spec1.open("output/spec1.txt",ios::app);
           // spec1<<strainvalues[kk-9]<<" "<<strainvalues[kk-8]<<" "<<strainvalues[kk-7]<<" "<<strainvalues[kk-6]<<" "<<strainvalues[kk-5]<<" "<<strainvalues[kk-4]<<" "<<strainvalues[kk-3]<<" "<<strainvalues[kk-2]<<" "<<strainvalues[kk-1]<<" "<<strainvalues[kk]<<" "<<strainvalues[kk+1]<<" "<<strainvalues[kk+2]<<" "<<strainvalues[kk+3]<<" "<<strainvalues[kk+5]<<" "<<strainvalues[kk+6]<<" "<<strainvalues[kk+7]<<" "<<strainvalues[kk+8]<<" "<<strainvalues[kk+9]<<endl;// here write spec1 sequence
           // spec1.close();
           //HERE CHANGES     //from ABOVE 
            R_Dat.CrabTime=TEC_Crab;// reference Integration Time for Crab
            R_Dat.CrabTimePEM1=PEM1noise_Crab;
            R_Dat.CrabTimePEM2=PEM2noise_Crab;
            R_Dat.CrabTimePEM3=PEM3noise_Crab;
        
       

 //---------------------------------------o-----------------------------------------------
//// The New Pulsars Here
// NHAT NGUYEN
// Name of pulsar and information 
//                         Freq.        Fdot             Dist
//J1952+3252(Pulsar1)    25.296    -3.7401x10e-12       2.50
//J1913+1011(Pulsar2)    27.848    -2.6120x10e-12       4.48
//J0737-3039A(Pulsar3)    44.054    -3.4150x10e-15       0.33
//J0537-6910(Pulsar4)    62.026    -1.9920x10e-10      49.40


//----Calculate Integration Time to reach spin-down limit for pulsar1---------------------------------
     // pulsar1 Parameters
             float freqpulsar10=25.296; // frequency of the pulsar1 in Hz
             float fdotpulsar1=-3.74e-12;// fdot of the pulsar1
             float freqpulsar1 = freqpulsar10 ;
             float rpulsar1=2.50;//distance in kpc of the pulsar1                
             k=0;
	     while (freqIFO[kk]<= 2*freqpulsar1)
             {
             kk++;
              } 
               pp1=0;
               pp2=0;
               pp3=0;
              
             while (freqPEM1[pp1]<= 2*freqpulsar1)
             {
             pp1++;
              }
             while (freqPEM2[pp2]<= 2*freqpulsar1)
             {
             pp2++;
              }
              while (freqPEM3[pp3]<= 2*freqpulsar1)
             {
             pp3++;
              }
      // this gives frequency bigger than pulsar1
     // float Snoise_IFO_pulsar1P=pow(strainvalues[kk],2);
    // PSD at 2 times the pulsar1Freq (above the real frequency)
             float hnoise_IFO_pulsar1=strainvalues[kk-1];
             float Pdotpulsar1= - pow(freqpulsar1,-2) * fdotpulsar1;// Pdot for the pulsar1
             float h_EC_pulsar1=5.7e-24/(rpulsar1)*pow(2 * freqpulsar1 /1000 * Pdotpulsar1 / 1e-13 , 0.5);// strain at EC limit
             float TEC_pulsar1 = 
             pow(pulsar1AntPar*AntPar, 2 )*pow(hnoise_IFO_pulsar1/h_EC_pulsar1,2)/60/60/24 ;
          ////here get some strain for spectrogram
          //ofstream spec1;
          //  spec1.open("output/spec1.txt",ios::app);
           // spec1<<strainvalues[kk-9]<<" "<<strainvalues[kk-8]<<" "<<strainvalues[kk-7]<<" "<<strainvalues[kk-6]<<" "<<strainvalues[kk-5]<<" "<<strainvalues[kk-4]<<" "<<strainvalues[kk-3]<<" "<<strainvalues[kk-2]<<" "<<strainvalues[kk-1]<<" "<<strainvalues[kk]<<" "<<strainvalues[kk+1]<<" "<<strainvalues[kk+2]<<" "<<strainvalues[kk+3]<<" "<<strainvalues[kk+5]<<" "<<strainvalues[kk+6]<<" "<<strainvalues[kk+7]<<" "<<strainvalues[kk+8]<<" "<<strainvalues[kk+9]<<endl;// here write spec1 sequence
           // spec1.close();
           //HERE CHANGES     //from ABOVE 
            float PEM1noise_pulsar1=ampPEM1[pp1-1];
            float PEM2noise_pulsar1=ampPEM2[pp2-1];
            float PEM3noise_pulsar1=ampPEM3[pp3-1];
           

            R_Dat.pulsar1Time=TEC_pulsar1;// reference Integration Time for pulsar1
            R_Dat.pulsar1TimePEM1=PEM1noise_pulsar1;
            R_Dat.pulsar1TimePEM2=PEM2noise_pulsar1;
            R_Dat.pulsar1TimePEM3=PEM3noise_pulsar1;
            

          // FILTERING FOR PULSAR1

          // Correlation between DARM_ERR channel and PEM1 for Pulsar 1
          // use first t_asq and ts_PEM1 (remember they have different sampling frequency)
         


//----Calculate Integration Time to reach spin-down limit for pulsar2---------------------------------
     // pulsar2 Parameters
             float freqpulsar20=27.848; // frequency of the pulsar1 in Hz
             float fdotpulsar2=-2.612e-12;// fdot of the pulsar2
             float freqpulsar2 = freqpulsar20 ;
             float rpulsar2=4.48;//distance in kpc of the pulsar2                
             kk=0;
             while (freqIFO[kk]<= 2*freqpulsar2)
             {
             kk++;
              } 
               pp1=0;
               pp2=0;
               pp3=0;
               
             while (freqPEM1[pp1]<= 2*freqpulsar2)
             {
             pp1++;
              }
             while (freqPEM2[pp2]<= 2*freqpulsar2)
             {
             pp2++;
              }
              while (freqPEM3[pp3]<= 2*freqpulsar2)
             {
             pp3++;
              }
      // this gives frequency bigger than pulsar1
     // float Snoise_IFO_pulsar1P=pow(strainvalues[kk],2);
    // PSD at 2 times the pulsar1Freq (above the real frequency)
             float hnoise_IFO_pulsar2=strainvalues[kk-1];
             float Pdotpulsar2= - pow(freqpulsar2,-2) * fdotpulsar2;// Pdot for the pulsar2
             float h_EC_pulsar2=5.7e-24/(rpulsar2)*pow(2 * freqpulsar2 /1000 * Pdotpulsar2 / 1e-13 , 0.5);// strain at EC limit
             float TEC_pulsar2 = 
             pow(pulsar2AntPar*AntPar, 2 )*pow(hnoise_IFO_pulsar2/h_EC_pulsar2,2)/60/60/24 ;
          ////here get some strain for spectrogram
          //ofstream spec1;
          //  spec1.open("output/spec1.txt",ios::app);
           // spec1<<strainvalues[kk-9]<<" "<<strainvalues[kk-8]<<" "<<strainvalues[kk-7]<<" "<<strainvalues[kk-6]<<" "<<strainvalues[kk-5]<<" "<<strainvalues[kk-4]<<" "<<strainvalues[kk-3]<<" "<<strainvalues[kk-2]<<" "<<strainvalues[kk-1]<<" "<<strainvalues[kk]<<" "<<strainvalues[kk+1]<<" "<<strainvalues[kk+2]<<" "<<strainvalues[kk+3]<<" "<<strainvalues[kk+5]<<" "<<strainvalues[kk+6]<<" "<<strainvalues[kk+7]<<" "<<strainvalues[kk+8]<<" "<<strainvalues[kk+9]<<endl;// here write spec1 sequence
           // spec1.close();
           //HERE CHANGES     //from ABOVE 
            float PEM1noise_pulsar2=ampPEM1[pp1-1];
            float PEM2noise_pulsar2=ampPEM2[pp2-1];
            float PEM3noise_pulsar2=ampPEM3[pp3-1];
           

            R_Dat.pulsar2Time=TEC_pulsar2;// reference Integration Time for pulsar1
            R_Dat.pulsar2TimePEM1=PEM1noise_pulsar2;
            R_Dat.pulsar2TimePEM2=PEM2noise_pulsar2;
            R_Dat.pulsar2TimePEM3=PEM3noise_pulsar2;

//----Calculate Integration Time to reach spin-down limit for pulsar3---------------------------------
     // pulsar3 Parameters
             float freqpulsar30=44.054; // frequency of the pulsar3 in Hz
             float fdotpulsar3=-3.415e-12;// fdot of the pulsar3
             float freqpulsar3 = freqpulsar30 ;
             float rpulsar3=0.33;//distance in kpc of the pulsar3                
             kk=0;
             while (freqIFO[kk]<= 2*freqpulsar3)
             {
             kk++;
              } 
               pp1=0;
               pp2=0;
               pp3=0;
              
             while (freqPEM1[pp1]<= 2*freqpulsar3)
             {
             pp1++;
              }
             while (freqPEM2[pp2]<= 2*freqpulsar3)
             {
             pp2++;
              }
              while (freqPEM3[pp3]<= 2*freqpulsar3)
             {
             pp3++;
              }
      // this gives frequency bigger than pulsar3
     // float Snoise_IFO_pulsar3P=pow(strainvalues[kk],2);
    // PSD at 2 times the pulsar3Freq (above the real frequency)
             float hnoise_IFO_pulsar3=strainvalues[kk-1];
             float Pdotpulsar3= - pow(freqpulsar3,-2) * fdotpulsar3;// Pdot for the pulsar3
             float h_EC_pulsar3=5.7e-24/(rpulsar3)*pow(2 * freqpulsar3 /1000 * Pdotpulsar3 / 1e-13 , 0.5);// strain at EC limit
             float TEC_pulsar3 = 
             pow(pulsar3AntPar*AntPar, 2 )*pow(hnoise_IFO_pulsar3/h_EC_pulsar3,2)/60/60/24 ;
          ////here get some strain for spectrogram
          //ofstream spec1;
          //  spec1.open("output/spec1.txt",ios::app);
           // spec1<<strainvalues[kk-9]<<" "<<strainvalues[kk-8]<<" "<<strainvalues[kk-7]<<" "<<strainvalues[kk-6]<<" "<<strainvalues[kk-5]<<" "<<strainvalues[kk-4]<<" "<<strainvalues[kk-3]<<" "<<strainvalues[kk-2]<<" "<<strainvalues[kk-1]<<" "<<strainvalues[kk]<<" "<<strainvalues[kk+1]<<" "<<strainvalues[kk+2]<<" "<<strainvalues[kk+3]<<" "<<strainvalues[kk+5]<<" "<<strainvalues[kk+6]<<" "<<strainvalues[kk+7]<<" "<<strainvalues[kk+8]<<" "<<strainvalues[kk+9]<<endl;// here write spec1 sequence
           // spec1.close();
            float PEM1noise_pulsar3=ampPEM1[pp1-1];
            float PEM2noise_pulsar3=ampPEM2[pp2-1];
            float PEM3noise_pulsar3=ampPEM3[pp3-1];
           

            R_Dat.pulsar3Time=TEC_pulsar3;// reference Integration Time for pulsar1
            R_Dat.pulsar3TimePEM1=PEM1noise_pulsar3;
            R_Dat.pulsar3TimePEM2=PEM2noise_pulsar3;
            R_Dat.pulsar3TimePEM3=PEM3noise_pulsar3;


//----Calculate Integration Time to reach spin-down limit for pulsar4---------------------------------
     // pulsar4 Parameters
             float freqpulsar40=62.0262; // frequency of the pulsar4 in Hz
             float fdotpulsar4=-1.992e-12;// fdot of the pulsar4
             float freqpulsar4 = freqpulsar40;
             float rpulsar4=49.4;//distance in kpc of the pulsar4                
             kk=0;
             while (freqIFO[kk]<= 2*freqpulsar4)
             {
             kk++;
              } 
               pp1=0;
               pp2=0;
               pp3=0;
               
             while (freqPEM1[pp1]<= 2*freqpulsar4)
             {
             pp1++;
              }
             while (freqPEM2[pp2]<= 2*freqpulsar4)
             {
             pp2++;
              }
              while (freqPEM3[pp3]<= 2*freqpulsar4)
             {
             pp3++;
              }
      // this gives frequency bigger than pulsar4
     // float Snoise_IFO_pulsar4P=pow(strainvalues[kk],2);
    // PSD at 2 times the pulsar4Freq (above the real frequency)
             float hnoise_IFO_pulsar4=strainvalues[kk-1];
             float Pdotpulsar4= - pow(freqpulsar4,-2) * fdotpulsar4;// Pdot for the pulsar4
             float h_EC_pulsar4=5.7e-24/(rpulsar4)*pow(2 * freqpulsar4 /1000 * Pdotpulsar4 / 1e-13 , 0.5);// strain at EC limit
             float TEC_pulsar4 = 
             pow(pulsar4AntPar*AntPar, 2 )*pow(hnoise_IFO_pulsar4/h_EC_pulsar4,2)/60/60/24 ;
          ////here get some strain for spectrogram
          //ofstream spec1;
          //  spec1.open("output/spec1.txt",ios::app);
           // spec1<<strainvalues[kk-9]<<" "<<strainvalues[kk-8]<<" "<<strainvalues[kk-7]<<" "<<strainvalues[kk-6]<<" "<<strainvalues[kk-5]<<" "<<strainvalues[kk-4]<<" "<<strainvalues[kk-3]<<" "<<strainvalues[kk-2]<<" "<<strainvalues[kk-1]<<" "<<strainvalues[kk]<<" "<<strainvalues[kk+1]<<" "<<strainvalues[kk+2]<<" "<<strainvalues[kk+3]<<" "<<strainvalues[kk+5]<<" "<<strainvalues[kk+6]<<" "<<strainvalues[kk+7]<<" "<<strainvalues[kk+8]<<" "<<strainvalues[kk+9]<<endl;// here write spec1 sequence
           // spec1.close();
           //HERE CHANGES     //from ABOVE 
            float PEM1noise_pulsar4=ampPEM1[pp1-1];
            float PEM2noise_pulsar4=ampPEM2[pp2-1];
            float PEM3noise_pulsar4=ampPEM3[pp3-1];
           

            R_Dat.pulsar4Time=TEC_pulsar4;// reference Integration Time for pulsar1
            R_Dat.pulsar4TimePEM1=PEM1noise_pulsar4;
            R_Dat.pulsar4TimePEM2=PEM2noise_pulsar4;
            R_Dat.pulsar4TimePEM3=PEM3noise_pulsar4;
            
////////////////-----------CALL PulsarFom.hh-----------------------------------------------
// This function calculates the FOMs for Known Pulsars 
//
//         int lengthPSD=PSDed_data.getNStep()+1;
 //        float StrainVal[lengthPSD];    
  //       float FREQIFO[lengthPSD];
  //       float HPULEC[lengthPSD];
//         float THEC[lengthPSD];
//         float EPSI[lengthPSD];
   //      int LENGHTPSD;
      //   float* hPulsarEC= new float[PSDed_data.getNStep() + 1];
     //    float* ThEC_f = new float[PSDed_data.getNStep() + 1];
     //    float* Epsilon_Run = new float[PSDed_data.getNStep() + 1];
           float* hIFOY= new float[PSDed_data.getNStep() + 1];
           for (unsigned int j=0; j<PSDed_data.getNStep()+1; j++) 
           {
           hIFOY[j] = AntPar * strainvalues[j]/pow(60*60*24*365,0.5);
           }
 //   FOMPulsar( strainvalues, freqIFO, lengthPSD, hPulsarEC, ThEC_f, Epsilon_Run);  
       //  hP_EC= PSDed_data;
       //  hP_EC.setData(PSDed_data.getNStep() + 1, hPulsarEC);
           hIFO_year=PSDed_data;
           hIFO_year.setData(PSDed_data.getNStep() + 1, hIFOY );
      //   TimeInt_knownpul= PSDed_data;
      // TimeInt_knownpul.setData(PSDed_data.getNStep() + 1, ThEC_f);
     //    EpsiRun=PSDed_data; 
     //    EpsiRun.setData(PSDed_data.getNStep() + 1, Epsilon_Run); 
              std::string DMTViewer_name = Run_Par.IFO;
              DMTViewer_name += " Pulsar FOM1: IFO's Noise Strain Amplitude for 1 year IT";
              serveData(DMTViewer_name.c_str(), &hIFO_year);                        
         //   DMTViewer_name = Run_Par.IFO;
      //      DMTViewer_name += " Pulsar FOM1: Strain for known pulsars at EC limit, freq. signal> 5 Hz";
           // serveData(DMTViewer_name.c_str(), &hP_EC);
    //        DMTViewer_name = Run_Par.IFO;
  //          DMTViewer_name += " Pulsar FOM2: IT to reach EC limit for known pulsars (days), freq. signal> 50 Hz";
     //       serveData(DMTViewer_name.c_str(), &TimeInt_knownpul);
           
       //     DMTViewer_name = Run_Par.IFO;
        //    DMTViewer_name += " Pulsar FOM3: Ellipticity for known pulsars, IT = run, freq. signal> 50 Hz";
        //    serveData(DMTViewer_name.c_str(), &EpsiRun);
 
//------CALL FOM3unknown---calculate FOM3 for unknown pulsars at 1kpc, 1 year IT
            FOM3unknown(Ellipticity_1kpc,strain_noise_ampl);            
	    DMTViewer_name = Run_Par.IFO;
            DMTViewer_name += " Pulsar FOM3: Ellipticity for Pulsar at 1kpc, 1 year IT";
            serveData(DMTViewer_name.c_str(), &Ellipticity_1kpc);      

//----------------------CALL FOM4CumSpectrum---------calculate FOM4: Cumulative PSD over run time
             if(!UpLastTime)
	      //if ( (R_Dat.alpha > 1.4)|| (R_Dat.beta >6) || 
              // (R_Dat.alpha <0.35) || (R_Dat.beta <0.65)   ) 
             // this is to check if the IFO is unlocked
	      {
		N_unlock = N_unlock + 1;
		//cerr<<"Not in lock N times = "<<N_unlock<<endl;
	      }
	    else 
            {
                  double nstride=NStride*1.0;
                  double N_half_h=fmod(nstride,30.0);
                  if (N_half_h==0.0) //Write to file every 30 minutes Here fmod instead
                  {
	          FOM4CumSpectrumWrite(NStride,FOM4_PSD,strain_noise_ampl,CumS_New,CumS_Old);
                  CumS_Old=CumS_New;
	          DMTViewer_name += " Pulsar FOM4: Cum. Strain Spec. for current Run Time";
	          serveData(DMTViewer_name.c_str(), &FOM4_PSD); 
     
                  }
                  else 
                  {
                  FOM4CumSpectrum(NStride,FOM4_PSD,strain_noise_ampl,CumS_New,CumS_Old);
                  CumS_Old=CumS_New;
	          DMTViewer_name += " Pulsar FOM4: Cum. Strain Spec. for current Run Time";
	          serveData(DMTViewer_name.c_str(), &FOM4_PSD); 
                  }                       

	    }

              delete [] freqIFO;
              delete [] freqPEM1;
              delete [] freqPEM2;
              delete [] freqPEM3;
         //   delete hPulsarEC;
        //    delete ThEC_f; 
       //     delete Epsilon_Run;
              delete [] hIFOY;  
              delete [] strainvalues;
              delete [] ampPEM1;
              delete [] ampPEM2;
              delete [] ampPEM3;
            //  delete [] tS;
           //--------------------------------------------o-----------------------------------------------
     //:END:KLUDGE:STRAIN:

	  } else 
            { 
	    //---------- IFO not Locked; set range to zero, don't increment
	    //		 4-volume.
	    UpLastTime = false;
	    R_Dat.alpha = 0.0;
	    R_Dat.beta = 0.0;
      	    R_Dat.ellipticity = 0.0;
            R_Dat.CrabTime=0.0;
	    R_Dat.pulsar1Time=0.0;
	    R_Dat.pulsar2Time=0.0;
	    R_Dat.pulsar3Time=0.0;
	    R_Dat.pulsar4Time=0.0;
	    R_Dat.ampl = 0.0;	
            
	    //----- Set to 1 PSDs to be send to DMTViewer.
	    int nsteps = PSDed_data.getNStep();
	    float *data = new float[nsteps + 1];
	     for (int i=0; i<(nsteps + 1); i++) 
             {
		data[i] = 1;
      	     }
             strain_noise_ampl.setData(nsteps + 1, data);
	     delete [] data;
           }

       //---------- Shift DMTViewer TSeries data one time step.
        int nshift = 1+missing_strides;
        for (int ii=0; ii<(tserieslength-nshift); ii++) {
            calvalues[ii] = calvalues[ii+nshift];
            crabvalues[ii]= crabvalues[ii+nshift];
	    pulsar1values[ii]= pulsar1values[ii+nshift];
	    pulsar2values[ii]= pulsar2values[ii+nshift];
	    pulsar3values[ii]= pulsar3values[ii+nshift];
	    pulsar4values[ii]= pulsar4values[ii+nshift];
            amplvalues[ii] = amplvalues[ii+nshift];
            alphavalues[ii] = alphavalues[ii+nshift];
            betavalues[ii] = betavalues[ii+nshift];
        } 

	//---------- Fill any missing strides with -1.0.
	if (missing_strides) {
          for (int ii=(tserieslength-nshift); ii<(tserieslength-1); ii++) {
            calvalues[ii]   = -1.0;
            crabvalues[ii]  = -1.0;
	    pulsar1values[ii]  = -1.0;
	    pulsar2values[ii]  = -1.0;
	    pulsar3values[ii]  = -1.0;
	    pulsar4values[ii]  = -1.0;
            amplvalues[ii]  = -1.0;
            alphavalues[ii] = -1.0;
            betavalues[ii]  = -1.0;
          } 
	}

	//---------- Store most recent values.
        calvalues[tserieslength-1] = R_Dat.ellipticity;
        crabvalues[tserieslength-1]=R_Dat.CrabTime;
	pulsar1values[tserieslength-1]=R_Dat.pulsar1Time;
	pulsar2values[tserieslength-1]=R_Dat.pulsar2Time;
	pulsar3values[tserieslength-1]=R_Dat.pulsar3Time;
	pulsar4values[tserieslength-1]=R_Dat.pulsar4Time;
        amplvalues[tserieslength-1] = R_Dat.ampl;
	alphavalues[tserieslength-1] = R_Dat.alpha;
	betavalues[tserieslength-1] = R_Dat.beta;

	//---------- Export data to DMTViewer. 

	    history_ellipticity->setData(history_start, Run_Par.T, calvalues, tserieslength);
	    history_crab->setData(history_start, Run_Par.T, crabvalues, tserieslength);
	    history_pulsar1->setData(history_start, Run_Par.T, pulsar1values, tserieslength);
	    history_pulsar2->setData(history_start, Run_Par.T, pulsar2values, tserieslength);
	    history_pulsar3->setData(history_start, Run_Par.T, pulsar3values, tserieslength);
	    history_pulsar4->setData(history_start, Run_Par.T, pulsar4values, tserieslength);
 	    history_ampl->setData(history_start, Run_Par.T, amplvalues, tserieslength);
 	    history_alpha->setData(history_start, Run_Par.T, alphavalues, tserieslength);
 	    history_beta->setData(history_start, Run_Par.T, betavalues, tserieslength);

	//---------- Write the range and 4-volume scanned to the log file
	//	     and cout.
	ReportResults(mOutput.cumlog_file, time, Run_Par);
        if (mOutput.screen_output || mOutput.write_log) {
	    ReportResults(cout, time, Run_Par);
        }

	//---------- Update Trends. HERE MORE STUFF ABOUT TRENDS !!!!!
	if (mOutput.trend) {
	    std::string trendName;
	        trendName = Run_Par.IFO + ":DMT-PULS_500HZ_ELLIPTICITY";
                mTrend.trendData(trendName.c_str(), time, R_Dat.ellipticity);
                trendName = Run_Par.IFO + ":DMT-PULS_CRAB_IT_SPINDOWN";
                mTrend.trendData(trendName.c_str(), time, R_Dat.CrabTime);
            

            trendName = Run_Par.IFO + ":DMT-PULS_J1952_plus_3252_IT_SPINDOWN";
            mTrend.trendData(trendName.c_str(),time, R_Dat.pulsar1Time );  
            trendName = Run_Par.IFO + ":DMT-PULS_J1913_plus_1011_IT_SPINDOWN";
            mTrend.trendData(trendName.c_str(), time, R_Dat.pulsar2Time);  
            trendName = Run_Par.IFO + ":DMT-PULS_J0737_minus_3039A_IT_SPINDOWN";
            mTrend.trendData(trendName.c_str() ,time, R_Dat.pulsar3Time);
	    trendName = Run_Par.IFO + ":DMT-PULS_J0537_minus_6910_IT_SPINDOWN";
            mTrend.trendData(trendName.c_str() ,time, R_Dat.pulsar4Time);  

         //   trendName = Run_Par.IFO + ":DMT-PULS_CRAB_IT_SPINDOWN_" + Run_Par.IFOPEM + ":PEM-LVEA_SEISX";  
         //   mTrend.trendData(trendName.c_str(), time, R_Dat.CrabTimePEM1);
         //   trendName = Run_Par.IFO + ":DMT-PULS_J1952_plus_3252_IT_" + Run_Par.IFOPEM + ":PEM-LVEA_SEISX";
         //   mTrend.trendData(trendName.c_str(), time, R_Dat.pulsar1TimePEM1);  
         //   trendName = Run_Par.IFO + ":DMT-PULS_J1913_plus_1011_IT_" + Run_Par.IFOPEM + ":PEM-LVEA_SEISX";
         //   mTrend.trendData(trendName.c_str(),time, R_Dat.pulsar2TimePEM1);  
         //   trendName = Run_Par.IFO + ":DMT-PULS_J0737_minus_3039A_IT_" + Run_Par.IFOPEM + ":PEM-LVEA_SEISX";
         //   mTrend.trendData(trendName.c_str(),time, R_Dat.pulsar3TimePEM1);
	 //   trendName = Run_Par.IFO + ":DMT-PULS_J0537_minus_6910_IT_" + Run_Par.IFOPEM +  ":PEM-LVEA_SEISX";
         //   mTrend.trendData(trendName.c_str(),time, R_Dat.pulsar4TimePEM1);


            trendName = Run_Par.IFO + ":DMT-PULS_CAL_LINE_AMPL_DARM_ERR";
            mTrend.trendData(trendName.c_str(), time, R_Dat.ampl);
	    trendName = Run_Par.IFO + ":DMT-PULS_CAL_CAV_FAC";
            mTrend.trendData(trendName.c_str(), time, R_Dat.alpha);
	    trendName = Run_Par.IFO + ":DMT-PULS_CAL_OLOOP_FAC";
            mTrend.trendData(trendName.c_str(), time, (R_Dat.alpha)*(R_Dat.beta));
	 }

        //---------- Do periodic range and calibrated PSD dumps if running in 
        //           background (!local) mode:    
	//           -> two-week summary file  
	//           -> calibrated psd dumped to ascii file
	//           NOTE: Since stride "Run_Par.T" may not exactly divide
        //           the specified update period "CalPSD_Dump_Period",
	//	     not all strides will contribute with equal rate
	//	     to a given average.  Eg: If CalPSD_Dump_Period is 1000       //// 
        //           and Run_Par.T = 300, the first through 
	//	     four2th strides contribute to first 1000-sec average with
	//	     weights 1, 1, 1, 1/3.  Fourth stride also contributes to
	//	     second 1000-sec average with weight 2/3.
	//           Note also that report time (GPS truncated to last multiple
        //           of CalPSD_Dump_Period) is 
	//           guaranteed to lie in the 1000-sec interval reported on.
        //           Times when the IFO is not locked contribute zero to the average.
	if (!mOutput.local) {
	    time_since_update += (int) Run_Par.T;
	    if (Run_Par.CalPSD_Dump_Period && time_since_update >= Run_Par.CalPSD_Dump_Period) {

	        //---------- 2-week summary file update:
	        //----- Add current range to running average with
	        //      weight = fraction of stride coming before (CalPSD_Dump_Period)-sec mark.
	        running_average += (Run_Par.T - time_since_update + Run_Par.CalPSD_Dump_Period) /
		   		    Run_Par.T * R_Dat.ellipticity;
                running_count += (Run_Par.T - time_since_update + Run_Par.CalPSD_Dump_Period)/Run_Par.T;
	        running_average /= running_count;

	        //----- Dump running average to two-week summary file.
                //      Report time to last multiple of CalPSD_Dump_Period 
                //      (eg, to last 000 sec for CalPSD_Dump_Period = 1000).
                //cout <<  (time.getS() % CalPSD_Dump_Period) << endl; //// HERE PulsarDataSummary
                RangeDataSummary tmpstatus = {time.getS() - (time.getS() % Run_Par.CalPSD_Dump_Period), running_average};
                mRangeSumm.append(tmpstatus);
                mRangeSumm.dumpList();

	        //----- Initialize next running average with whatever is left
	        //      of current stride.
	        time_since_update -= Run_Par.CalPSD_Dump_Period;
		running_average = (time_since_update) / Run_Par.T * R_Dat.ellipticity;
	        running_count = (time_since_update)/Run_Par.T;

		//----- calibrated psd file dump:  
               // if (mOsclist->satisfied(osc_cond1.c_str())  && 
                 //   mOsclist->satisfied(osc_cond2.c_str())  ) // CHANGE for DOL (3)

                   if (true) {

	            //---------- Set up array to hold values of calibrated AS_Q PSD.
                    float* CalPSDValues = new float[PSDed_data.getNStep() + 1];		   
		    char* time_holder = new char[128];

		    //---------- Get values of calibrated PSD.
            	    PSDed_data.getData(PSDed_data.getNStep() + 1, CalPSDValues);

	    	    //---------- Print values to text file
                    //----- Specify arm length in nm for conversion of PSD units to strain:
                    float arm_length = 1.0;  
                    if (Run_Par.IFO == "H2") { 
                      arm_length = 2.0e12;  
                    } else if ((Run_Par.IFO == "L1") || (Run_Par.IFO == "H1")) { 
                      arm_length = 4.0e12;  
                    } else {
                      cerr << "PulsarMon ERROR: IFO " << Run_Par.IFO << " not recognized.  Calibrated PSD files will \n";
                      cerr << "                     not have units of strain.\n";
                    }

                    //TimeStr(ts_asq->getStartTime(), time_holder, "%Y%M%d.%H:%N:%S.txt");
            	    TimeStr(ts_asq->getStartTime(), time_holder, "%s.txt");
       //TimeStr( operator-(ts_asq->getStartTime(),Interval(time_since_update,0)), time_holder, "%s.txt");
       		    mOutput.calpsd_file_name = time_holder;
            	    mOutput.calpsd_file_name = string(mOutput.html_dir) + "/" + Run_Par.IFO + "_CalPSD_GPS:" + mOutput.calpsd_file_name;
            	    mOutput.calpsd_file.open(mOutput.calpsd_file_name.c_str());
            	    mOutput.calpsd_file << "#Calibrated PSD of AS_Q Channel, produced by SenseMonitor version 5.0." << endl;
            	    mOutput.calpsd_file << "#Units: (strain)/Hz^{-0.5}" << endl;
                    mOutput.calpsd_file << "#Number of Averages for PSD: " << Run_Par.num_ints << endl;
                    mOutput.calpsd_file << "#Window: " << Run_Par.window_name << endl;
            	    mOutput.calpsd_file << "#GPS start time of this data: " << time.getS() << endl;
                    int npoints = PSDed_data.getNStep() + 1; 
                    mOutput.calpsd_file << "#Number of points: " << npoints << endl;
            	    mOutput.calpsd_file << "#Frequency step: " << PSDed_data.getFStep() << " Hz" << endl;
            	    mOutput.calpsd_file << "#Min Frequency: " << PSDed_data.getLowFreq() << " Hz" << endl;
            	    mOutput.calpsd_file << "#Max Frequency: " << PSDed_data.getLowFreq() + (npoints-1)*(PSDed_data.getFStep())<< " Hz" << endl;
            	    //----- Dump sqrt of power spectrum, converted to units of strain/Hz^1/2.  
            	    for (unsigned int j=0; j<PSDed_data.getNStep()+1; j++) 
                   {
                        mOutput.calpsd_file << pow((double) CalPSDValues[j],0.5)/arm_length << endl;
                    }
            	    mOutput.calpsd_file.close();

                    //---------- Clean up.
                    delete[] CalPSDValues;
		    delete[] time_holder;
	        }
            } else 
              { 
	        //----- Add current range to running average with weight 1
    	        running_average += R_Dat.ellipticity;
    	        running_count   += 1.0;
	    }
	}

	//----------  Calculate average of last five time segments.

	R_Dat.T5_array[NStride % 5] = R_Dat.ellipticity;
	for(int i = 0; i < 5; i++)
	    R_Dat.T5_ellipticity += R_Dat.T5_array[i];
	R_Dat.T5_ellipticity /= 5.0;

	//----------  Calculate average of last fifteen time segments.
	R_Dat.T15_array[NStride % 15] = R_Dat.ellipticity;
	for(int i = 0; i < 15; i++)
            R_Dat.T15_ellipticity += R_Dat.T15_array[i];
        R_Dat.T15_ellipticity /= 15.0;

	//---------- Print summary webpages.
	RevolverSummary(time, Run_Par, R_Dat);
	Summary(time, Run_Par, R_Dat);

	//---------- Clear old data.
	R_Dat.T15_ellipticity = 0.0;
	R_Dat.T5_ellipticity = 0.0;  

	//---------- Check if should finish.
	if (++NStride >= MaxStride) finish();
 
	//----------  Flush any lingering data.
        fflush(stderr);
	fflush(stdout);
}

//====================================== Write revolver.html page 
void
PulsarMon::RevolverSummary(const Time& t, Parameter Run_Par,
			      Range_Data R_Dat)
{
    //---------- Html document
    html::document doc (kHtmlTitle);
    char buf[128];

    //---------- Title
    html::block titleblk ("center");
    titleblk.lineBreak();
    html::text title (kHtmlTitle);
    html::font fonttitle ("Helvetica");
    title.setFont (fonttitle);
    title.setColor (html::color (0, 0, 128));
    title.setSize (html::size (+4));
    titleblk.addObject (title);
    titleblk.lineBreak();
    titleblk.addObject (html::text ("&nbsp;"));
    doc.addObject (titleblk);
    doc.addObject (html::hline());

    //---------- Current status section.
    html::block status_blk ("center");
    status_blk.lineBreak();
    //:KLUDGE: Should specify range is with/without cal lines 
    //(if dyncal||refcal||Cal_Line -> yes; else -> no).   
    html::text status_title ("Current Ellipticity");
    status_title.setFont (fonttitle);
    status_title.setSize (html::size (+2));
    status_blk.addObject (status_title);
    status_blk.lineBreak();

    //---------- Result table.
    html::table results;
    results.addColumn ("GPS Time");
    results.addColumn ("15-Stride Average Ellipticity");
    results.addColumn ("Next Update Time (UTC)");
    results.addRow();
    results.insertData(0, 0, html::text (TimeStr(t, buf, "%s")));
    results.insertData(0, 1, html::text (R_Dat.T15_ellipticity));
    results.insertData(0, 2, html::text (t + Run_Par.T));
    for (int j=0; j<3; ++j) {
        results.refCell(0, j).setAlign ("right");
    }
    results.setBorder(true);
    status_blk.addObject(results);
    doc.addObject(status_blk);

    //---------- Last update time.
    doc.addObject (html::hline());
    html::block updateblk ("center");
    html::text lasttime ("This page was last updated: ");
    lasttime << TimeStr (t, buf, "%M %d, %Y at %H:%N:%S");
    updateblk.addObject (lasttime);
    updateblk.lineBreak();
    doc.addObject (updateblk);
    doc.addObject (html::hline());

     //---------- Write web page to file.
    ofstream out (mOutput.revolver_file_name.c_str());
    if (out) {
        html::writer htmlout (out);
        doc.write (htmlout);
    }
}


//====================================== Write summary html page.
void
PulsarMon::Summary(const Time& t, Parameter Run_Par, Range_Data R_Dat)
{   
    //---------- Html document
    html::document doc (kHtmlTitle);
    char buf[128];

    //---------- Title
    html::block titleblk ("center");
    titleblk.lineBreak();
    html::text title (kHtmlTitle);
    html::font fonttitle ("Helvetica");
    title.setFont (fonttitle);
    title.setColor (html::color (0, 0, 128));
    title.setSize (html::size (+4));
    titleblk.addObject (title);
    titleblk.lineBreak();
    titleblk.addObject (html::text ("&nbsp;"));
    doc.addObject (titleblk);
    doc.addObject (html::hline());
   
    //---------- Short description
    doc.addObject (html::text (kHtmlDescription));
    doc.addObject (html::hline());

//:KLUDGE:UPDATE: 
    //---------- Last update time.
    html::block firstupdateblk ("center");
    html::text lasttime1 ("This page was last updated: "); 
    lasttime1 << TimeStr (t, buf, "%M %d, %Y at %H:%N:%S %Z");
    firstupdateblk.addObject (lasttime1);
    firstupdateblk.lineBreak();
    html::text lasttime2 ("Next scheduled update time: ");
    lasttime2 << TimeStr (t+Run_Par.T, buf, "%M %d, %Y at %H:%N:%S %Z");
    firstupdateblk.addObject (lasttime2);
    firstupdateblk.lineBreak();
    doc.addObject (firstupdateblk);
    doc.addObject (html::hline());
//:END:KLUDGE:UPDATE: 

    //---------- Run parameters
    html::block run_param_blk ("center");
    run_param_blk.lineBreak();
    html::text param_title ("Run Parameters");
    param_title.setFont (fonttitle);
    param_title.setSize (html::size (+2));
    run_param_blk.addObject (param_title);
    run_param_blk.lineBreak();
    //---------- Parameters table.
    html::table params; 
    //----- WARNING:  Must make columns first, then rows.
    params.addColumn("");
    params.addColumn("");
    for (int j=0; j<11; ++j) {
	params.addRow();
    }
    int i=-1;
    params.insertData(++i, 0, html::text ("Channel: "));
    params.insertData(  i, 1, html::text (Run_Par.mChannel));
// add here name of PEM channel
    params.insertData(++i, 0, html::text ("Stride: "));
    params.insertData(  i, 1, html::text (Run_Par.T));
    params.insertData(  i, 1, html::text (" sec"));
    params.insertData(++i, 0, html::text ("Number of Strides Already "
					  "Processed: "));
    params.insertData(  i, 1, html::text (NStride+1));
    params.insertData(++i, 0, html::text ("Maximum Number of Strides to "
					  "Process: "));
    params.insertData(  i, 1, html::text (MaxStride));
    params.insertData(++i, 0, html::text ("Number of Averages for PSD: "));
    params.insertData(  i, 1, html::text (Run_Par.num_ints));
    params.insertData(++i, 0, html::text ("Window: "));
    params.insertData(  i, 1, html::text (Run_Par.window_name));
    params.insertData(++i, 0, html::text ("Low Frequency : "));
    params.insertData(  i, 1, html::text (Run_Par.l_freq));
    params.insertData(  i, 1, html::text (" Hz"));
    params.insertData(++i, 0, html::text ("High Frequency : "));
    params.insertData(  i, 1, html::text (Run_Par.h_freq));
    params.insertData(  i, 1, html::text (" Hz"));
    params.insertData(++i, 0, html::text ("Monitor Start Time (GPS): "));
    params.insertData(  i, 1, html::text (TimeStr(Time(starttime), buf,
					  "%s")));
    params.insertData(++i, 0, html::text ("(UTC): "));
    params.insertData(  i, 1, html::text (TimeStr(Time(starttime), buf,
					  "%M %d, %Y at %H:%N:%S")));

    //----- WARNING:  It appears that alignments can only be set AFTER the cell
    //	    is filled.
    //      WARNING:  The (html::table).getNRow() method returns 1+(# of rows).
    //	    Beware!
    for (int j=0; j<params.getNRow()-1; ++j) {
        params.refCell(j, 0).setAlign("right");
        params.refCell(j, 1).setAlign ("left");
    }
    run_param_blk.addObject(params);
    doc.addObject (run_param_blk);
   
    //---------- Current status section.
    html::block status_blk ("center");
    status_blk.lineBreak();
    //:KLUDGE: Should specify ellipticity is with/without cal lines 
    //(yes provided using cal_line)
    html::text status_title ("Current Ellipticity");
    status_title.setFont (fonttitle);
    status_title.setSize (html::size (+2));
    status_blk.addObject (status_title);
    status_blk.lineBreak();
   
    //---------- Result table.
    html::table results;
    results.addColumn ("GPS Time");
    results.addColumn ("Crab IT DARM_ERR");
    results.addColumn ("Crab IT PEM-LVEA_SEISX");
    results.addColumn ("Crab IT PEM-LVEA_SEISY");
    results.addColumn ("Crab IT PEM-LVEA_SEISZ");
    results.addColumn ("J1952+3252 IT DARM_ERR");
    results.addColumn ("J1952+3252 IT PEM-LVEA_SEISX");
    results.addColumn ("J1952+3252 IT PEM-LVEA_SEISY");
    results.addColumn ("J1952+3252 IT PEM-LVEA_SEISZ");
    results.addColumn ("J1913+1011 IT DARM_ERR ");
    results.addColumn ("J1913+1011 IT PEM-LVEA_SEISX");
    results.addColumn ("J1913+1011 IT PEM-LVEA_MAGY");
    results.addColumn ("J1912+1011 IT PEM-LVEA_MAGZ");
    results.addColumn ("J0737-3039A IT DARM_ERR ");
    results.addColumn ("J0737-3039A IT PEM-LVEA_SEISX");
    results.addColumn ("J0737-3039A IT PEM-LVEA_SEISY");
    results.addColumn ("J0737-3039A IT PEM-LVEA_SEISZ");
    results.addColumn ("J05537_6910 IT DARM_ERR ");
    results.addColumn ("J05537_6910 IT PEM-LVEA_SEISX");
    results.addColumn ("J05537_6910 IT PEM-LVEA_SEISY");
    results.addColumn ("J05537_6910 IT PEM-LVEA_SEISZ");
    results.addColumn ("Next Update Time (UTC)");
    results.addRow();
    results.insertData(0, 0, html::text (TimeStr(t, buf, "%s")));
    results.insertData(0, 1, html::text (R_Dat.CrabTime));
    results.insertData(0, 2, html::text (R_Dat.CrabTimePEM1));
    results.insertData(0, 3, html::text (R_Dat.CrabTimePEM2));
    results.insertData(0, 4, html::text (R_Dat.CrabTimePEM3));
    results.insertData(0, 5, html::text (R_Dat.pulsar1Time));
    results.insertData(0, 6, html::text (R_Dat.pulsar1TimePEM1 ));
    results.insertData(0, 7, html::text (R_Dat.pulsar1TimePEM2 ));
    results.insertData(0, 8, html::text (R_Dat.pulsar1TimePEM3) );
    results.insertData(0, 9, html::text (R_Dat.pulsar2Time));
    results.insertData(0, 10, html::text (R_Dat.pulsar2TimePEM1));
    results.insertData(0, 11, html::text (R_Dat.pulsar2TimePEM2));
    results.insertData(0, 12,html::text (R_Dat.pulsar2TimePEM3) );
    results.insertData(0, 13, html::text (R_Dat.pulsar3Time));
    results.insertData(0, 14,html::text (R_Dat.pulsar3TimePEM1 ));
    results.insertData(0, 15,html::text (R_Dat.pulsar3TimePEM2 ));
    results.insertData(0, 16,html::text (R_Dat.pulsar3TimePEM3) );
    results.insertData(0, 17, html::text (R_Dat.pulsar4Time ));
    results.insertData(0, 18, html::text (R_Dat.pulsar4TimePEM1 ));
    results.insertData(0, 19, html::text (R_Dat.pulsar4TimePEM2 ));
    results.insertData(0, 20, html::text (R_Dat.pulsar4TimePEM3) );
    results.insertData(0, 21, html::text (t + Run_Par.T));
    for (int j=0; j<21; ++j) {
        results.refCell(0, j).setAlign ("right");
    }
    results.setBorder(true);
    status_blk.addObject(results);
    doc.addObject(status_blk);

    if (!mOutput.screen_output) {  
        //---------- Insert error message link
        html::block errorinfo ("center");

        //---------- Link to error-message log.
	html::link errormessage ("Error Messages", mOutput.error_file_link.c_str() );
	errorinfo.addObject (errormessage);
        errorinfo.lineBreak();
        doc.addObject (errorinfo);

        //---------- Insert links to current (this run of monitor) and 
        //           cumulative (all runs) log files.
        html::block output ("center");
	html::link outputlink1 ("Monitor Output: more details", (mOutput.log_file_link).c_str() );
        output.addObject (outputlink1);
        output.lineBreak();
        if (!mOutput.local) {
	    //html::link outputlink2 ("Cumulative Monitor Output from all runs", (cumlog_file_link).c_str() );
	    html::link outputlink2 ("Cumulative Monitor Output from all runs", (mOutput.cumlog_file_link).c_str() );
            output.addObject (outputlink2);
            output.lineBreak();
        }
        output.lineBreak();
        doc.addObject (output);
    }

    //---------- Last update time.
    doc.addObject (html::hline());
    html::block updateblk ("center");
    html::text lasttime ("This page was last updated: ");
//:KLUDGE:
    lasttime << TimeStr (t, buf, "%M %d, %Y at %H:%N:%S %Z");
//  lasttime << TimeStr (t, buf, "%M %d, %Y at %H:%N:%S");
    updateblk.addObject (lasttime);
    updateblk.lineBreak();

    //---------- Author info.
    updateblk.addObject( html::text ("This monitor was written by ") );
    updateblk.addObject( html::link ("Giovanni Santostasi",
				     "mailto:gsantostasi@mcneese.edu") );

    updateblk.addObject( html::link ("based on SenseMonitor by Patrick Sutton",
                                     "mailto:psutton.ligo.caltech.edu") );
    updateblk.addObject( html::text (" and ") );
    updateblk.addObject( html::link ("Kevin C. Schlaufman",
                                     "mailto:kcs149@psu.edu") );

    doc.addObject (updateblk);
    doc.addObject (html::hline());

    //---------- LIGO logo.
    html::block logoblk ("div");
    logoblk.addAttr ("align", html::align ("right"));
    logoblk.addObject (html::link (kHtmlLIGO, kHtmlLIGOLink));
    html::image logo;
    logo.setSource ("ligo_logo.gif");
    logo.setWidth ("80");
    logoblk.addObject (logo);
    doc.addObject (logoblk);

    //---------- Write web page to file.
    ofstream out (mOutput.summary_file_name.c_str());
    if (out) {
        html::writer htmlout (out);
        doc.write (htmlout);
    }
}


//====================================== Dump run parameters to output stream.
void
PulsarMon::DumpParameters(std::ostream& out)
{   
        
    //----- Dump run parameters.
    out << "# Command Line: " << command_line << endl;
    out << "# Launch Directory: " << pwd_dir <<  endl;
    out << "# Log File: " << mOutput.cumlog_file_name << endl;
    out << "# OSC Configuration File: " << Run_Par.osc_file_name << endl; 
    out << "# Reference Calibration File: " << Run_Par.xml_file_name << endl; 
    out << "# Channel: " << Run_Par.mChannel << endl;
    out << "# Stride: " << Run_Par.T << " sec" << endl;
    out << "# Max Number of Strides: " << MaxStride << " sec" << endl;
    out << "# Number of Averages for PSD: " << Run_Par.num_ints << endl;
    out << "# Low Freq.  : " << Run_Par.l_freq << " Hz" << endl;
    out << "# High Freq. : " << Run_Par.h_freq << " Hz" << endl;
    out << "# Window: " << Run_Par.window_name << endl;
    out << "# Monitor Start Time: " << starttime << endl;
    out << "# " <<endl;

}


//====================================== Dump column headers to output.
void
PulsarMon::DumpReportHeader(std::ostream& out) 
{
   //----- Dump headers for range data that will follow.
  //  out << "# 1) Data value of 0 means IFO not locked." << endl;
 //   out << "# 2) Data values of -1, -2 mean monitor not functioning." << endl;
 //   out << "# " << endl;
  //  out << "#   Start                                               " << endl;
  //  out << "# Time of   Ellipt.   Crab EC IT  Crab PEM1 1952+3252IT  J1952+3252ITPEM1  J1913+1011IT J1913+1011ITPEM1 J0737-3039AIT J0737-3039APEM1 J0537_6910IT J0537_6910ITPEM1" << endl;
   // out << "# Segment  1Kpc 500Hz   Days       Arbitrary    Days       Arbitrary            days        Arbitrary       days          Arbitrary        days         Arbitrary    " << endl;
//  out << "#       -    


  
	
}


//====================================== Dump column headers to output.
void
PulsarMon::DumpReportHeader(std::ostream& out, double freq) 
{
 ////----- Dump headers for range data that will follow.
 ////   out << "# Notes:" << endl;
 ////   out << "# 1) Range of 0 means IFO not locked." << endl;
 ////   out << "# 2) Ranges quoted are adjusted by the" << endl;
 ////   out << "#    calibration-line amplitude only if the -dyncal option is used." << endl;
 ////   out << "# " << endl;
}

//====================================== Read configuration file and pass contents 
//                                       to command-line parser.
void
PulsarMon::Configure(string& file_name) 
{
    int NEWargc = 1;
    char **NEWargv;
 
    //---------- Verify that configuration file can be opened.
    ifstream input;
    input.open(file_name.c_str());
    if (input.fail()) {
        std::cerr << "PulsarMon ERROR: Cannot open configuration file ";
        std::cerr << file_name  << ".  Exiting.\n";
        fatal_error = true;
        finish();
    } else {
        cout << "PulsarMon MESSAGE: Opened configuration file ";
        cout << file_name << endl;
        //---------- Get number of words in configuration file.
        std::string word;
        while (input >> word) {
            ++NEWargc;
          
        }

    }
    input.close();
    NEWargv = new char*[NEWargc];
    NEWargv[0] = new char[32];
    strcpy(NEWargv[0], "Configuration File: ");
    //cout << NEWargv[0] << endl;
    //---------- Open file and copy contents to char arrays.
    ifstream inp2;
    inp2.open(file_name.c_str());
    if (inp2.fail()) {
        cout << "Yikes!!!!! Can't open " << file_name << endl;
	return;
    }
    std::string word;
    for (int i=1; i<NEWargc; i++) {
        inp2 >> word;
        NEWargv[i] = new char[80];
        strcpy(NEWargv[i],word.c_str());
        //cout << NEWargv[i] << endl;
    }
    inp2.close();
    Configure(NEWargc,(const char **) NEWargv);
    for (int i=0; i<NEWargc; ++i) delete[] NEWargv[i];
    delete [] NEWargv;
}


//====================================== Parse contents of configuration file or command line.

void
PulsarMon::Configure(int argc, const char *argv[]) 
{
    //---------- Get command line, for the record.
    for (int i=0; i < argc; i++) {
        std::string arg_i = argv[i];
        command_line += arg_i;        
        command_line += "  ";         
    }
    
    //---------- Get local directory, for the record.
    char *tempdir = getenv("PWD");
    if (tempdir) {
	pwd_dir = string(tempdir) + "/";
    } 
    
    //---------- Get Default channel name and IFO.
    Run_Par.IFO = argv[argc - 1];
    Run_Par.mChannel  = Run_Par.IFO; 
    Run_Par.mChannel += ":LSC-DARM_ERR";
    cerr<<"here 6"<<endl;
    //----- Set default parameters.
    Run_Par.osc_cond = Run_Par.IFO + ":SV_IFO_UP";

    //----- Parse command-line arguments, and compare to list of known 
    //      values.  Ignore unrecognized options.


    if ( (argc <= 1) || (argv[argc-1][0] == '-') ) {
        cout << USAGE_INFO << endl;
        fatal_error = true;
        finish();
    } else {
        for (int i=1 ; i<argc-1 ; i++) {
            std::string argi = argv[i];
            if (argi == "-max") {
	        MaxStride = strtol(argv[++i], 0, 0);
 	    } else if (argi == "-stride") {
	        Run_Par.T = strtod(argv[++i], 0);
	    } else if (argi == "-n") {
	        Run_Par.num_ints = strtol(argv[++i], 0, 0);
	    } else if (argi == "-PSD_DumpPeriod") {
	        Run_Par.CalPSD_Dump_Period = strtol(argv[++i], 0, 0);
	    } else if (argi == "-fmin") {
	        Run_Par.l_freq = strtod(argv[++i], 0);
	    } else if (argi == "-fmax") {
	        Run_Par.h_freq = strtod(argv[++i], 0);
 	    } else if (argi == "-xmlfile") { 
                Run_Par.xml_file_name = argv[++i];
	    } else if (argi == "-logfile") {
	        //Run_Par.cumlog_file_name = argv[++i];
	        mOutput.cumlog_file_name = argv[++i];
 	    } else if (argi == "-OSCcond") { 
                Run_Par.osc_cond = Run_Par.IFO + ":" + argv[++i];
 	    } else if (argi == "-OSCfile") { 
                Run_Par.osc_file_name = argv[++i];
	    } else if (argi == "-procname") { 
                Run_Par.MonitorName = argv[++i];
 	    } else if (argi == "-channel") { 
                Run_Par.mChannel = argv[++i];
                cerr<<"here 8 "<<endl;
	    } else if(argi == "-readalphabeta") {
                Run_Par.genCalParam = false;
            } else if(argi == "-screen") {
                mOutput.screen_output = true;
     	        mOutput.write_log = false;
 	        mOutput.error_log = false;
	    } else if(argi == "-trend") {
	        mOutput.trend = true;
  	    } else if(argi == "-window") {
                Run_Par.window_name = argv[++i];
  	        for(unsigned int ii=0; ii<Run_Par.window_name.length(); ii++) {
                    Run_Par.window_name[ii] = tolower(Run_Par.window_name[ii]);
	        }
	    } else if ( (argi == "-h") || (argi == "--help") ) {
	        cout << USAGE_INFO << endl;
                fatal_error = true;
	        finish();
	    } else if(argi == "-local") {
                mOutput.local = true;
	    } else if (isDatEnvArg(argi.c_str())) {
	        i++;
            } else {
	        std::cerr << "PulsarMon WARNING:  Argument: " << argi;
	        std::cerr << " not recognized." << std::endl;
                cout << USAGE_INFO << endl;
                //fatal_error = true;
	        //finish();
	    }
        }
    } 
}

//====================================== Run list of checks on parameters.
//:KLUDGE: Since this is a method of the PulsarMon class, shouldn't need to 
//hand it data members of the same class (Run_Par, fatal_error).
void
PulsarMon::VerifyParameters(Parameter& Run_Par, bool& fatal_error)
{
     

    if ( ((int)(Run_Par.T) % Run_Par.num_ints) != 0 ) {
	cout << "PulsarMon ERROR: The number of averages 'n' must divide the length 'stride'.\n"; 
	cout << "                    Exiting.\n";
	cout << USAGE_INFO << endl;
	fatal_error = true;
	finish();
    } 
    if ( Run_Par.l_freq >= Run_Par.h_freq ) {
        cout << "PulsarMon ERROR: Low frequency must be less than high ";
	cout << "frequency.  Exiting.\n";
        cout << USAGE_INFO << endl;
        fatal_error = true;
        finish();
    }
    
    
    if (Run_Par.window_name == "square") {
	cout << "PulsarMon WARNING: Square window (ie, no windowing) selected for\n"; 
	cout << "                      power-spectrum estimation.  This is a VERY BAD\n";
	cout << "                      IDEA; you should select anything else.\n";
    }
    if (!fatal_error) {
        //if ( !mOutput.local && !((Run_Par.cumlog_file_name).empty()) ) {
        if ( !mOutput.local && !((mOutput.cumlog_file_name).empty()) ) {
	    cout << "PulsarMon WARNING: -local option has not been selected; specification\n";
            cout << "                      of log-file name will be ignored.\n";
        }

           //---------- Get name of OSC configuration files.
        if ( (Run_Par.osc_file_name).empty() ) {
  	    cout << "PulsarMon WARNING: No OSC configuration file specified; will look for\n";
            cout << "                      default OSC configuration file.\n";
            //---------- Look for OSC config file in directory specified
            //           in $PULSARMON_OSCCONF instead of in local directory.
            const char* osc_dir = getenv("PULSARMON_OSCCONF");
            if (osc_dir) {
                Run_Par.osc_file_name = string(osc_dir) + "/";
            } else { 
	        cout << "PulsarMon WARNING: Environment variable $PULSARMON_OSCCONF not set;\n"; 
                cout << "                      looking in local directory for default OSC file.\n";
            }
            //Run_Par.osc_file_name += "LockLoss_";
            //Run_Par.osc_file_name += Run_Par.IFO;
            Run_Par.osc_file_name += "PulsaMon_LockLoss.conf";
        }
        //---------- Verify that calibration file can be opened.
        ifstream input;  //----- previous declaration inside if () statement therefore gone.
        input.open((Run_Par.osc_file_name).c_str());
        if (input.fail()) {
            std::cerr << "PulsarMon ERROR: Cannot open OSC configuration file ";
            std::cerr << Run_Par.osc_file_name  << ".  Exiting.\n";
            fatal_error = true;
            finish();
        } else {
	    cout << "PulsarMon MESSAGE: Found OSC configuration file ";
	    cout << Run_Par.osc_file_name << endl;
	}
        input.close();
    }
}

//====================================== Write current range, etc to specified
//                                       output stream.
void
PulsarMon::ReportResults(std::ostream& out, const Time& time, Parameter& Run_Par)
{
	//---------- Write the ellipticity, crabtime, alpha, etc. to the specified 
	//	     output stream.
	out << time.getS();
	out << ' ' << setw(15) << R_Dat.ellipticity;
	out << ' ' << setw(15) << R_Dat.CrabTime;
        out << ' ' << setw(15) << R_Dat.CrabTimePEM1;
        out << ' ' << setw(15) << R_Dat.CrabTimePEM2;
        out << ' ' << setw(15) << R_Dat.CrabTimePEM3;
        out << ' ' << setw(15) << R_Dat.pulsar1Time;
        out << ' ' << setw(15) << R_Dat.pulsar1TimePEM1;
        out << ' ' << setw(15) << R_Dat.pulsar1TimePEM2;
        out << ' ' << setw(15) << R_Dat.pulsar1TimePEM3;
	out << ' ' << setw(15) << R_Dat.pulsar2Time;
        out << ' ' << setw(15) << R_Dat.pulsar2TimePEM1;
        out << ' ' << setw(15) << R_Dat.pulsar2TimePEM2;
        out << ' ' << setw(15) << R_Dat.pulsar2TimePEM3;
	out << ' ' << setw(15) << R_Dat.pulsar3Time;
        out << ' ' << setw(15) << R_Dat.pulsar3TimePEM1;
        out << ' ' << setw(15) << R_Dat.pulsar3TimePEM2;
        out << ' ' << setw(15) << R_Dat.pulsar3TimePEM3;
	out << ' ' << setw(15) << R_Dat.pulsar4Time;
        out << ' ' << setw(15) << R_Dat.pulsar4TimePEM1;
        out << ' ' << setw(15) << R_Dat.pulsar4TimePEM2;
        out << ' ' << setw(15) << R_Dat.pulsar4TimePEM3;
	
        out << endl;
}
