/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "PSLSpec.hh"
#include "PSLChan.hh"
#include "DVector.hh"
#include "xsil/XSpectrum.hh"
#include "xsil/Xwriter.hh"
#include <fstream>
#include <cstdlib>

using namespace std;

//======================================  Spectrum constructor
PSLSpec::PSLSpec(const string& name, const PSLChan* chan)
  : mName(name), mChan(chan), mAvStart(0), mAvgTime(16.0),
    mSave(false), mSvStart(0), mSavTime(3600.0)
{}

PSLSpec::PSLSpec(const PSLSpec& x)
  : mName(x.mName), mChan(x.mChan), mAvStart(0), mAvgTime(x.mAvgTime),
    mSave(false), mSvStart(0), mSavTime(x.mSavTime), mSvFile(x.mSvFile)
{}

//======================================  Spectrum destructor
PSLSpec::~PSLSpec(void) {
}

//======================================  reset
void 
PSLSpec::reset(void) {
    mAvStart = Time(0);
    mAccum.clear();
}

//======================================  Collect a spectrum
bool 
PSLSpec::crunch(void) {
    const TSeries& ts(mChan->refTSeries());
    Time tEnd = ts.getEndTime();

    //----------------------------------  Accumulate the average.
    if (!mAvStart) {
        mAvStart = ts.getStartTime();
	mAccum   = FSpectrum(mChan->refFSeries());
    } else {
	mAccum  += FSpectrum(mChan->refFSeries());
    }

    bool test(false);
    if (tEnd >= mAvStart+mAvgTime || Almost(tEnd, mAvStart+mAvgTime)) {
        double nAv = mAccum.getCount();
	mAverage  = mAccum;
	mAverage *= 1.0/double(mAccum.getCount());
	mAverage.setName(mName.c_str());
	reset();
	if (!mCompare.isEmpty()) {
	    double chisq(0);
	    int nComp = mCompare.getCount();
	    int nStep = mCompare.getNStep();
	    const DVector* DVec = const_cast<const FSpectrum&>(mCompare).refDVect();
	    for (int i=0 ; i<nStep ; i++) {
	        double a = DVec->getFloat(i);
		double b = DVec->getFloat(i);
		chisq += (a-b) * (a-b) / (a*a/nComp + b*b/nAv);
	    }
	    cout << "Comparison of spectrum " << mName 
		 << " is complete. Chisq = " << chisq  << " / " 
		 << nStep << " DOF." << endl;
	}
    }

    //----------------------------------  Save a spectrum periodically.
    if (!mSvStart) mSvStart = ts.getStartTime();
    if (ts.getEndTime() >= mSvStart + mSavTime) {
        mSvStart = ts.getEndTime();
        char fileName[128];
	TimeStr(mSvStart, fileName, mSvFile.c_str());
	ofstream ostr(fileName);
	if (ostr.good()) {
	    xsil::Xwriter xw(ostr);
	    xsil::ligolw doc(mName.c_str(), "container");
	    doc.addObject(xsil::XSpectrum(mName.c_str(), mAverage));
	    doc.Spew(xw);
	} else {
	    cout << "Error opening file: " << fileName << endl;
	}
    }
    return test;
}


//======================================  PSLSpec accessors
const char*
PSLSpec::getName(void) const {
    return mName.c_str();
}

const char*
PSLSpec::getChannel(void) const {
    return mChan->getChannel();
}

const FSpectrum&
PSLSpec::refAverage(void) const {
    return mAverage;
}

//======================================  Set averaging time.
void 
PSLSpec::setAvg(Interval avgTime){
    mAvgTime = avgTime;
}

//======================================  Specify standard spectrum
void
PSLSpec::setCompare(const FSpectrum& sp) {
    mCompare = sp;
}

//======================================  Specify Enable condition name
void
PSLSpec::setEnableCond(const string& str) {
    mEnableCond = str;
}

//======================================  Specify save interval and location
void 
PSLSpec::setSave(Interval savTime, const string& savFile) {
    mSave    = true;
    mSavTime = savTime;
    if (savFile.empty() || (savFile[0] != '/' && savFile[0] != '.')) {
        const char* dmtout = getenv("DMTOUTPUT");
	if (dmtout) mSvFile = string(dmtout) + "/";
	else        mSvFile = "";
	if (savFile.empty()) mSvFile += mName + "%s.xml";
	else                 mSvFile += savFile;
    } else {
        mSvFile = savFile;
    }
}
