#include "wreaddata.hh"
#include "wframecache.hh"
#include "matlab_fcs.hh"
#include "Dacc.hh"
#include "GaussNoise.hh"
#include "NDS2Socket.hh"

using namespace sends;

using namespace std;

namespace wpipe {
  typedef unsigned long gps_type;

  void 
  wreaddata(const wframecache& frameCache, const std::string& channelNames, 
	    const std::string& frameTypes, const Time& startTime, 
	    const Time& stopTime, double timeShifts, int debugLevel, 
	    tser_vect& data) {
    str_vect chName(1), frType(1);
    chName[0] = channelNames;
    frType[0] = frameTypes;
    dble_vect tShift(1 ,0);
    tShift[0] = timeShifts;
    wreaddata(frameCache, chName, frType, startTime, 
	      stopTime, tShift, debugLevel, data); 
  }

  //====================================  Multi-channel read.
  void 
  wreaddata(const wframecache& frameCache, const str_vect& channelNames, 
	    const str_vect& frameTypes, const Time& startTime, 
	    const Time& stopTime, const dble_vect& timeShifts, 
	    int debugLevel, tser_vect& data) {

    //----------------------------------  Data pointer manupulation.
    union {
	const char*   c;
	const double* d;
	const float*  f;
	const short*  s;
	const int*    i;
    } p;

    ///////////////////////////////////////////////////////////////////////////
    //                        process command line arguments                 //
    ///////////////////////////////////////////////////////////////////////////

    // determine number of channels
    size_t numberOfChannels = channelNames.size();

    ///////////////////////////////////////////////////////////////////////////
    //                     validate command line arguments                   //
    ///////////////////////////////////////////////////////////////////////////

    // validate frame types
    if (frameTypes.size() != numberOfChannels) {
      error("number of frame types is inconsistent with number of channels");
    }

    // validate time shifts
    if (timeShifts.size() != numberOfChannels) {
      error("number of time shifts is inconsistent with number of channels");
    }

    //----------------------------------  Initialize result structures
    data.resize(numberOfChannels);
    for (size_t i=0; i<numberOfChannels; i++) {
      data[i].Clear();
    }

    //----------------------------------  Calculate the duration
    Interval dT(stopTime - startTime);

    ///////////////////////////////////////////////////////////////////////////
    //                                  read data                            //
    ///////////////////////////////////////////////////////////////////////////

    //----------------------------------  Begin loop over channels
    for (size_t channelNo=0; channelNo<numberOfChannels; channelNo++){
      if (!data[channelNo].empty()) continue;

      //--------------------------------  Switch based on frame type
      size_t cinx   = frameTypes[channelNo].find(':');
      string upType = toupper(frameTypes[channelNo].substr(0,cinx));
      string chan_i = channelNames[channelNo];	
      string prefix = channelNames[channelNo].substr(0,1) + "-";
      prefix += upType;

      //--------------------------------  Skip if no data is requested,
      if (upType == "NONE") {
	continue;
      }

      //--------------------------------  whitenoise is requested,
      else if (upType == "WHITE" || upType == "WHITENOISE") {

	// generate white noise data
	double sampleRate = 16384;
	TSeries noise(startTime, Interval(1./sampleRate), long(sampleRate*dT),
		      GaussNoise(1.0));
      }

      //--------------------------------  NDS2 data is requested,
      else if (upType == "NDS2") {

	//----------------------------------  Get gps second limits.
	Time startShifted  = startTime + Interval(timeShifts[channelNo]);
	gps_type gps_start = startShifted.getS();
	Time stopShifted   = startShifted + dT;
	gps_type gps_stop  = stopShifted.getS();
	if (Time(gps_stop) < stopShifted) gps_stop++;

	str_vect fr_list;
	if (!frameCache.get_list(prefix, startShifted.getS(), 
				 stopShifted.getS(), fr_list) ) {
	  error("NDS server not listed in frame cache");
	}

	string nds_server = fr_list[0];
	int    nds_port   = 31200;
	string::size_type inx = nds_server.find(":");
	if (inx != string::npos) {
	  nds_port = strtol(nds_server.c_str() + inx + 1, 0, 0);
	  nds_server.erase(inx);
	}

	NDS2Socket nds(nds_server, nds_port);
	if (!nds.isOpen()) {
	  cout << "Unable to open server: " <<  nds_server << " port: " 
	       << nds_port << endl;
	  error("readdata: Unable to open nds2 server");
	}

	nds.AddChannel(chan_i, cUnknown, 0.0);
	gps_type stride = gps_stop - gps_start;

	//------------------------------  Loop over strides
	TSeries ts_data;
	for (gps_type t0=gps_start; t0<gps_stop; t0 += stride) {
	  Time t;

	  //----------------------------  Request a bunch of data.
	  int rc = nds.RequestData(t0, stride);
	  if (rc) error("readdata: nds2 data request failed.");

	  //----------------------------  Get the data
	  gps_type delta(stride);
	  for (gps_type tOff=0; tOff<stride; tOff += delta) {
	    rc = nds.GetData();
	    if (rc <= 0) error("readdata: nds2 GetData failed.");

	    //----------------------------  Set frame size parameters
	    gps_type t(nds.mRecvBuf.ref_header().GPS);
	    delta = nds.mRecvBuf.ref_header().Secs;

	    //----------------------------  Loop over channels goes here
	    {
	      DAQC_api::const_channel_iter cp = nds.FindChannel(chan_i);
	      Interval dt(1./cp->mRate);
	      unsigned long nw = cp->nwords(delta);
	      p.c = nds.mRecvBuf.ref_data() + cp->mBOffset;
	      switch (cp->mDatatype) {
	      case _16bit_integer:
		if (tOff) ts_data.Append (Time(t), dt, p.s, nw);
		else      ts_data.setData(Time(t), dt, p.s, nw);
		break;
	      case _32bit_integer:
		if (tOff) ts_data.Append (Time(t), dt, p.i, nw);
		else      ts_data.setData(Time(t), dt, p.i, nw);
		break;
	      case _32bit_float:
		if (tOff) ts_data.Append (Time(t), dt, p.f, nw);
		else      ts_data.setData(Time(t), dt, p.f, nw);
		break;
	      case _64bit_double:
		if (tOff) ts_data.Append (Time(t), dt, p.d, nw);
		else      ts_data.setData(Time(t), dt, p.d, nw);
		break;
	      default:
		error("readdata: Unsupported nds data type");
	      }
	    }
	  }
	}
	if (timeShifts[channelNo] != 0) {
	  TSeries ts_substr = ts_data.extract(startShifted, dT);
	  data[channelNo] = TSeries(startTime, ts_data.getTStep(),
				    *ts_substr.refDVect());
	} else {
	  data[channelNo] = ts_data.extract(startTime, dT);
	}
	data[channelNo].setName(channelNames[channelNo].c_str());
      } // end of nds2 read.

      //--------------------------------  Otherwise, read a frame
      else {

	double minShift = timeShifts[channelNo];
	double maxShift = minShift;
	
	//------------------------------  Build a list of channels with the
	//                                same frame type
	vector<size_t> inx_list;
	inx_list.push_back(channelNo);
	for (size_t i=channelNo+1; i<numberOfChannels; i++) {
	  if (channelNames[i].substr(0,1) != chan_i.substr(0,1)) continue;
	  size_t post_inx = frameTypes[i].find(":");
	  if (upType != toupper(frameTypes[i].substr(0,post_inx))) continue;
	  inx_list.push_back(i);
	  if (timeShifts[i] < minShift) {
	    minShift = timeShifts[i];
	  } else if (timeShifts[i] > maxShift) {
	    maxShift = timeShifts[i];
	  }
	}

	//----------------------------------  Get gps second limits.
	Time startShifted  = startTime + Interval(minShift);
	gps_type gps_start = startShifted.getS();
	Time stopShifted   = stopTime  + Interval(maxShift);
	gps_type gps_stop  = stopShifted.getS();
	if (Time(gps_stop) < stopShifted) gps_stop++;

	//------------------------------  Get frame file paths
	str_vect fr_list;
	frameCache.get_list(prefix, gps_start, gps_stop, fr_list);
	Dacc In;
	In.setIgnoreMissingChannel(true);
	size_t N = fr_list.size();
	for (size_t i=0; i<N; i++) {
	  In.addFile(fr_list[i].c_str());
	}

	//------------------------------  Request all channels to be read in.
	size_t nList = inx_list.size();
	for (size_t i=0; i<nList; i++) {
	  size_t iChan = inx_list[i];
          size_t post_inx = frameTypes[iChan].find(":");
          if (post_inx == string::npos) {
	      In.addChannel(channelNames[iChan]);
          } else if (frameTypes[iChan].substr(post_inx+1) == "simdata") {
	      In.addSimulated(channelNames[iChan]);
          } else {
	      error("Invalid frame-type postfix");
	  }
	}

	//------------------------------  Read the channel data.
	In.seek(startShifted);
	int rc = In.fillData(stopShifted - startShifted);
	if (rc) {
	  error(string("Error reading frame type: ") + prefix);
	}

	//------------------------------  Copy TSeries data to result vector
	for (size_t i=0; i<nList; i++) {
	  size_t iChan = inx_list[i];
	  TSeries* ts = In.refData(channelNames[iChan]);
	  if (ts) {
	    if (timeShifts[iChan] == 0) {
	      data[iChan] = ts->extract(startTime, dT);
	    } else {
	      TSeries ts_data(ts->extract(startTime + timeShifts[iChan], dT));
	      data[iChan] = TSeries(startTime, ts_data.getTStep(),
				    *ts_data.refDVect());
	    }
	    data[iChan].setName(channelNames[iChan].c_str());
	  }
	}

      }	 // end switch on frametype

    }  // end of loop over channels

  }  // End of wreaddata()

}  // namespace wpipe
