#include "wfigure.hh"
#include "weventlist.hh"
#include "wtile.hh"
#include "matlab_fcs.hh"

#include "lcl_array.hh"
#include "constant.hh"

#include <iostream>
#include <iomanip>
#include <sstream>
#include <string>

using namespace wpipe;
using namespace std;

const double dInf(1.0/0.0);


void
wfigure::weventgram(const weventlist& events, const wtile& tiling,
		    const Time& referenceTime, const dble_vect& tRange, 
		    const dble_vect& fRange, 
		    double durationInflation, double bandwidthInflation, 
		    const dble_vect& normalizedEnergyRange) {

  /////////////////////////////////////////////////////////////////////////////
  //                            hard coded parameters                        //
  /////////////////////////////////////////////////////////////////////////////

  // eventgram boundary
  //const double eventgramLeft = 0.14;
  //const double eventgramWidth = 0.80;
  //const double eventgramBottom = 0.28;
  //const double eventgramHeight = 0.62;

  // colorbar position
  //const double colorbarLeft = eventgramLeft;
  //const double colorbarWidth = eventgramWidth;
  //const double colorbarBottom = 0.12;
  //const double colorbarHeight = 0.02;

  /////////////////////////////////////////////////////////////////////////////
  //                       validate command line arguments                   //
  /////////////////////////////////////////////////////////////////////////////
  dble_vect timeRange = tRange;
  
  dble_vect frequencyRange = fRange;
  if (fRange.empty()) frequencyRange.resize(2, 0.0);
  if (frequencyRange[1] <= 0) frequencyRange[1] = dInf;

  // Check for two component range vectors
  if (timeRange.size() != 2) {
    error("Time range must be two component vector [tmin tmax].");
  }

  if (frequencyRange.size() != 2) {
    error("Frequency range must be two component vector [fmin fmax].");
  }

  if (!normalizedEnergyRange.empty() && normalizedEnergyRange.size() != 2) {
    error("Normalized energy range must be two component vector [Zmin Zmax].");
  }

  ////////////////////////////////////////////////////////////////////////////
  //                        identify events to display                      //
  ////////////////////////////////////////////////////////////////////////////

  /////////////////////////////////////////////////////////////////////////////
  //                         extract event properties                        //
  /////////////////////////////////////////////////////////////////////////////
  size_t numberOfEvents = events.size();
  lcl_array<double> x0(numberOfEvents);
  lcl_array<double> x1(numberOfEvents);
  lcl_array<double> y0(numberOfEvents);
  lcl_array<double> y1(numberOfEvents);
  lcl_array<double>  z(numberOfEvents);

  double eMax=0, fMin=0, fMax=0, tMin=0, tMax=0;
  size_t plotEvents = 0;
  double translateDt = double(referenceTime - events.refTime());
  double bwfact = 2 * sqrt(pi);
  for (size_t evtIndex=0; evtIndex < numberOfEvents; evtIndex++) {

    // find events overlapping specified time-frequency ranges
    double bandwidth = bwfact*events[evtIndex].frequency / events[evtIndex].q;
    double duration = 1. / bandwidth;
    bandwidth *= bandwidthInflation;
    duration  *= durationInflation;
    double tStart = events[evtIndex].t_offset + translateDt - 0.5*duration;
    if (tStart > timeRange[1]) continue;
    double tStop  = tStart + duration;
    if (tStop  < timeRange[0]) continue;

    double fLow  = events[evtIndex].frequency - 0.5 * bandwidth;
    if (fLow  >= frequencyRange[1]) continue;
    double fHigh = fLow + bandwidth;
    if (fHigh <= frequencyRange[0]) continue;

    double normE =  convertUnits(events[evtIndex].normalizedEnergy);
    if (tStart < timeRange[0]) tStart = timeRange[0];
    if (tStop  > timeRange[1]) tStop  = timeRange[1];

    x0[plotEvents] = tStart;
    x1[plotEvents] = tStop;
    y0[plotEvents] = fLow;
    y1[plotEvents] = fHigh;
    z [plotEvents] = normE;

    if (!plotEvents) {
      eMax = normE;
      fMin = fLow;
      fMax = fHigh;
      tMin = tStart;
      tMax = tStop;
    }
    else {
      if (normE  > eMax) eMax = normE;
      if (fLow   < fMin) fMin = fLow;
      if (fHigh  > fMax) fMax = fHigh;
      if (tStart < tMin) tMin = tStart;
      if (tStop  > tMax) tMax = tStop;
    }

    // number of events
    plotEvents++;
  }

  // default start time for
  if (timeRange[0] == -dInf) {
    if (tiling.empty()) {
      timeRange[0] = floor(tMin);
    } else {
      timeRange[0] = - tiling.duration() / 2;
    }
  }

  // default stop time
  if (timeRange[1] == dInf) {
    if (tiling.empty()) timeRange[1] = ceil(tMax);
    else                timeRange[1] = timeRange[0] + tiling.duration();
  }

  // default minimum frequency
  if (frequencyRange[0] <= 0) {
    if (tiling.empty()) {
      frequencyRange[0] = pow(2.0, floor(log2(fMin)));
    } else {
      frequencyRange[0] = tiling.planes(0).minimumFrequency;
    }
  }

  // default maximum frequency
  if (frequencyRange[1] == dInf) {
    if (tiling.empty()) {
      frequencyRange[1] = pow(2.0, ceil(log2(fMax)));
    }
    else {
      size_t lastPl = tiling.numberOfPlanes() - 1;
      frequencyRange[1] = tiling.planes(lastPl).maximumFrequency;
    }
  }

  //---------------------------------  Get time scale
  double dT = timeRange[1] - timeRange[0];
  double tScale = _plot.xTimeScale(dT, "Time");

  //------------------------------------  Select the events
  size_t j=0;
  for (size_t i=0; i<plotEvents; i++) {
    if (x0[i] >= timeRange[1]      || x1[i] <= timeRange[0]     ) continue;
    if (y0[i] >= frequencyRange[1] || y1[i] <= frequencyRange[0]) continue;
    if (j != i) {
      x0[j] = x0[i];
      x1[j] = x1[i];
      y0[j] = y0[i];
      y1[j] = y1[i];
      z[j]  = z[i];
    }
    x0[j] /= tScale;
    x1[j] /= tScale;
    j++;
  }
  plotEvents = j;

  /////////////////////////////////////////////////////////////////////////////
  //                          set colormap scaling                           //
  /////////////////////////////////////////////////////////////////////////////

  // if normalized energy range is not specified
  double zMin = 0;
  double zMax = 10;
  if (!normalizedEnergyRange.empty()) {
    zMin = normalizedEnergyRange[0];
    zMax = normalizedEnergyRange[1];
  } else if (plotEvents) {
    zMax = eMax;
  }

  /////////////////////////////////////////////////////////////////////////////
  //                                 plot events                             //
  /////////////////////////////////////////////////////////////////////////////

  // set y axis properties
  _plot.ylabel("Frequency [Hz]");
  _plot.ylog();
  _plot.set_zrange(zMin, zMax);

  // set title properties
  ostringstream titleString;
  titleString << events.channelName() << " at " << fixed 
 	      << setprecision(3) << referenceTime.totalS();
  _plot.title(titleString.str());
  //title(strrep(titleString, "_", "\_"));
  
  // set axis range
  _plot.set_range(timeRange[0]/tScale, timeRange[1]/tScale,
		  frequencyRange[0], frequencyRange[1]);
  if (_zUnits.empty() || _zUnits == "NormE") {
    _plot.zlabel(string("Normalized ") + events.eventClass() + " energy");
  } else {
    _plot.zlabel(events.eventClass() + " " + _zUnits);
  }
  _plot.boxes(plotEvents, x0, x1, y0, y1, z);

} // return to calling function
