/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "wchunk.hh"
#include "wcondition.hh"
#include "woutput.hh"
#include "wparameters.hh"
#include "wtile.hh"
#include "wtransform.hh"
#include "matlab_fcs.hh"
#include "MultiDacc.hh"
#include "lcl_array.hh"
#include "usertime.hh"
#include <iostream>
#include <iomanip>

using namespace std;

// bugs fixed: Injection time shift and data factors incorrect if data 
// bounced by incorrect state.

namespace wpipe {
   //=====================================  Construct a chunker
   wchunk::wchunk(void) 
      : currentTime(0), blockNo(0)
   {
      termFlag.add(SIGTERM);
      termFlag.add(SIGINT);
   }

   //=====================================  Test the terminate signal flag
   bool
   wchunk::term(void) const {
      return termFlag;
   }
   
   //=====================================  Process a block of data.
   int
   wchunk::process(double blockDuration, double blockOverlap, strchan_vect& cv, 
		   MultiDacc& frameSource, woutput& outputFiles, 
		   int debugLevel, str_vect& channelNames) 
   {
      user_timer cpuTime;
      int rc = 0;

      //////////////////////////////////////////////////////////////////////
      //                     read a chunk of data                         //
      //////////////////////////////////////////////////////////////////////
      double fillDuration = !currentTime ? blockDuration 
	                  : blockDuration-blockOverlap;

      size_t nGroups = cv.size();
      Time startTime(0);
      while (!term()) {
	 frameSource.setStride(fillDuration);

	 bool first = true;
	 rc = -8;
	 while (rc == -8 && !term()) { 
	    rc = frameSource.fillData(fillDuration, first);
	    first = false;
	    // cout << "Reading data... start: " << first << " fill time: "
	    // 	    << frameSource.getFillTime() << " current: " 
	    //	    << frameSource.getCurrentTime() << " rc: " << rc << endl; 
	 }
	 if (rc) return rc;

	 Time fillStartTime = frameSource.getFillTime();

	 //----------------------------  Append new data.
	 if (!startTime) {
	    if (fillStartTime == currentTime) {
	       startTime = currentTime - blockOverlap;
	    } else {
	       startTime = fillStartTime;
	    }
	    for (size_t i=0; i<nGroups; i++) {
	       cv[i].start_chunk();
	    }
	 }

	 //-----------------------------  Read in data from the frameSource.
	 for (size_t ig=0; ig<nGroups; ig++) {
	    cv[ig].read_data(frameSource, startTime, blockDuration);
	 }

	 currentTime = frameSource.getCurrentTime();
	 if (currentTime >= startTime + blockDuration) break;
      } // end fill loop

      //--------------------------------  stop time of block
      //                                  assume start time is aligned for now.
      Time blockStartTime = startTime;
      Time blockStopTime  = currentTime;

      //--------------------------------  report status
      if (debugLevel >= 1) {
	 cout << fixed << setprecision(2) 
	      << "  chunk start time:        " << blockStartTime.totalS() 
	      << endl
	      << "  chunk stop time:         " << blockStopTime.totalS()
	      << endl;
      }

      //--------------------------------  update all the file names.
      //ostringstream timestamp;
      //
      //timestamp << (blockStartTime + blockOverlap*0.5).getS() 
      //		<< "-" << blockDuration - blockOverlap;
      outputFiles.update(wparameters::validFileTypes,
	                 blockStartTime + blockOverlap*0.5,
	                 blockDuration - blockOverlap);

      //--------------------------------  Count total channels
      size_t nChanTot = 0;
      for (size_t ig=0; ig<nGroups; ig++) nChanTot += cv[ig].numberOfChannels();

      //--------------------------------  loop over groups, process data.
      weventstack triggersDownselected(nChanTot), clusters(nChanTot);
      for (size_t ig=0; ig<nGroups; ig++) {
	 //-------------------------  Inject signals if requested.
	 cv[ig].inject(frameSource);

	 //-------------------------  Process the channel group
	 cv[ig].process(outputFiles, blockStartTime, 
			blockDuration, blockOverlap);
	 cv[ig].validChans(channelNames);
         cout << "Number of triggered/total channels in group: " << ig 
	      << " is: " << cv[ig].trigDownselect().numberOfChannels() 
	      << "/" << cv[ig].numberOfChannels() << endl;
	 triggersDownselected.moveLists(cv[ig].trigDownselect());
	 clusters.moveLists(cv[ig].trigClusters());
      } // loop over groups.

      cout << "Block: " << blockNo << " total channels with triggers: " 
	   << triggersDownselected.numberOfChannels()
           << " total clusters/triggers: " << clusters.totalEvents() 
	   << "/" << triggersDownselected.totalEvents() << endl;
      blockNo++;

      //-----------------------------------  write single detector triggers
      str_vect triggerFields = cv[0].refParameters().triggerFields;
      string   triggerFormat = cv[0].refParameters().triggerFormat;
      if (outputFiles.enabled("DOWNSELECT")) {
	 wlog(debugLevel, 1, "  writing selected triggers");
	 triggersDownselected.writeEvents(outputFiles["DOWNSELECT"], 
					  triggerFields, triggerFormat);
      }

      //--------------------------------  Write cluster triggers
      if (outputFiles.enabled("CLUSTER")) {
	 wlog(debugLevel, 1, "  writing cluster triggers");
	 clusters.tsort();
	 if (cv[0].refParameters().writeClusters) {
	    clusters.writeEvents(outputFiles["CLUSTER"], 
				 triggerFields, triggerFormat);
	 } 
	 else {
	    clusters.writeEvents(outputFiles["CLUSTER"], 
				 triggerFields, triggerFormat);
	 }
      }

      /////////////////////////////////////////////////////////////////////////
      //                            write livetime                           //
      /////////////////////////////////////////////////////////////////////////
      wlog(debugLevel, 1, "  writing livetime");

      // analyzed channels string
      string analyzedChannels;
      for (size_t channelNumber=0; channelNumber<channelNames.size(); 
	   channelNumber++) {
	 if (channelNumber != 0) analyzedChannels += ",";
	 analyzedChannels += channelNames[channelNumber];
      }

      // open livetime file for appending
      ofstream liveStream(outputFiles["livetime"].path(0).c_str(), ios::app);

      // test for error
      if (!liveStream.is_open()) {
	 error("cannot open livetime file for writing");
      }
      
      // write block livetime to livetime file
      liveStream 
	 << setw(5) << blockNo << " " << fixed << setprecision(4)
	 << (blockStartTime + 0.5*blockOverlap).getS() << "-"
	 << (blockStopTime  - 0.5*blockOverlap).getS() << " "
	 << analyzedChannels << " " << cpuTime.get_time() << endl;

      // close livetime file
      liveStream.close();
      return rc;
   } // end fill loop


} // namespace wpipe
