/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "channelCache.hh"
#include "matlab_fcs.hh"
#include <sstream>

using namespace std;
using namespace wpipe;

//======================================  Construct a channel cache
channelCache::channelCache(const wframecache& f, int d)
   : _frcache(f), _debugLevel(d)
{}

//======================================  Destroy a channel cache
channelCache::~channelCache(void) {}

//======================================  Fetch data of a given profile
void 
channelCache::fetch(const string& type, const Time& start, const Time& stop) {
   str_vect names, types, keys;
   dble_vect shifts;
   for (map_iterator i=_cache.begin(); i != _cache.end(); i++) {
      footprint& fp(i->second);
      if (fp.dataOK()) continue;
      if (fp.type() != type || fp.start() != start || fp.stop() != stop) continue;
      names.push_back(fp.name());
      types.push_back(type);
      shifts.push_back(fp.shift());
      keys.push_back(i->first);
   }
   tser_vect tv;
   wreaddata(_frcache, names, types, start, stop, shifts, _debugLevel, tv);
   size_t N = names.size();
   for (size_t i=0; i<N; i++) {
      map_iterator p = _cache.find(keys[i]);
      if (p == _cache.end()) continue;
      p->second.set_series(tv[i]);
   }
}

//======================================  Add one or more channels
void
channelCache::request(const str_vect& name, const str_vect& type,
		      const Time& start, const Time& stop,
		      const dble_vect& shift) {
   dble_vect lcl_shift = shift;
   size_t N = name.size();
   if (type.size() != N) error("channelCache: type size don't match");
   lcl_shift.resize(N, 0.0);
   for (size_t i=0; i<N; i++) {
      footprint f(name[i], type[i], start, stop, lcl_shift[i]);
      std::string key = f.key();
      _cache.insert(map_type::value_type(key, f)).first->second.inc();
   }
}

//======================================  Add a channel
void
channelCache::read(const str_vect& name, const str_vect& type,
		   const Time& start, const Time& stop,
		   const dble_vect& shift, tser_vect& data) {
   size_t N = name.size();
   if (type.size() != N) error("channelCache: type size don't match");
   dble_vect lcl_shift = shift;
   lcl_shift.resize(N, 0.0);
   data.clear();
   data.resize(N);
   for (size_t i=0; i<N; i++) {
      footprint f(name[i], type[i], start, stop, lcl_shift[i]);
      std::string key = f.key();
      map_iterator p = _cache.find(key);
      if (p == _cache.end()) error(string("Channel not requested: ") + name[i]);
      if (!p->second.dataOK()) {
	 fetch(type[i], start, stop);
	 if (!p->second.dataOK()) error("requested data not found");
      }
      data[i] = p->second.ref_series();
      p->second.dec();
   }
}

//======================================  Construct data descriptor
channelCache::footprint::footprint(const string& name, const string& type,
				   const Time& start, const Time& stop,
				   double shift)
   : _name(name), _type(type), _start(start), _stop(stop), _shift(shift),
     _req_count(0)
{}

//======================================  Construct data descriptor
channelCache::footprint::footprint(const footprint& x)
   : _name(x._name), _type(x._type), _start(x._start), _stop(x._stop),
     _shift(x._shift), _req_count(x._req_count)
{
   if (x._ts.get() != 0) set_series(*x._ts);
}

//======================================  Destroy data descriptor
channelCache::footprint::~footprint(void) {
}

//======================================  Destroy data descriptor
bool 
channelCache::footprint::dec(void) {
   if (--_req_count) _ts.reset();
   return _req_count;
}

//======================================  Generate key string
std::string
channelCache::footprint::key(void) const {
   std::ostringstream str;
   str << _name << ";" << _type << ";" << _start << ";" << _stop << ";"
       << _shift << ";";
   return str.str();
}

//======================================  Generate key string
void
channelCache::footprint::set_series(const TSeries& ts) {
   _ts.reset(new TSeries(ts));
}
