/***************************************************************************
    File        : Window.h
    Description : Creates a window filter for signal processing
 ---------------------------------------------------------------------------
    Begin       : Thu Dec 11 2003
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#ifndef __WINDOW_H
#define __WINDOW_H

// Libs
#include <vector>

// Project files
#include "Types.h"
#include "Utilities.h"

namespace gwd {

  enum WindowType {WND_HANNING,WND_BARTLETT,WND_WELCH,WND_HAMMING};

  /** The Window class.
   **/
  class Window {
  public:
    typedef Vector::size_type SizeType;
  public:
    Window() {}
    ~Window() {}

    // Methods
    inline void window(const WindowType wt,const SizeType nn,Vector& ww);
    inline void Hanning(const SizeType nn,Vector& ww);
    inline void Bartlett(const SizeType nn,Vector& ww);
    inline void Welch(const SizeType nn,Vector& ww);
    inline void Hamming(const SizeType nn,Vector& ww);
  private:
  };


  // Inline methods

  // Method: window
  //   Generates a window of type WindowType, e.g. Hanning
  inline void
  Window::window(const WindowType wt,const SizeType nn,Vector& ww)
  {
    switch(wt)
    {
      case WND_HANNING:
        Hanning(nn,ww);
        break;
      case WND_BARTLETT:
        Bartlett(nn,ww);
        break;
      case WND_WELCH:
        Welch(nn,ww);
        break;
      case WND_HAMMING:
        Hamming(nn,ww);
        break;
    }
  }

  //!  \fun Hanning
  //!   Generates a Hanning window.
  //!   Considers only the periodic case
  inline void
  Window::Hanning(const SizeType nn,Vector& ww)
  {
    // Periodic case
    double alpha = 2.*gwd::PI / static_cast<double>(nn);
    if (ww.size() != nn)
      ww.resize(nn);
    for (SizeType i = 0; i < nn; i++)
      ww[i] = 0.5 * (1. - std::cos(static_cast<double>(i)*alpha));
  }

  //! \fun Bartlett
  //!   Generates a Bartlett window
  //!   Considers only the periodic case
  inline void
  Window::Bartlett(SizeType nn,Vector& ww)
  {
    // Periodic case
    double xmax = static_cast<double>(nn)/2.;
    if (ww.size() != nn)
      ww.resize(nn);
    for (SizeType i = 0; i < nn; i++)
      ww[i] = 1. - fabs(((double)i - xmax)/xmax);
  }


  //! \fun Welch
  //!   Generates a Welch window
  //!   Considers only the periodic case
  inline void
  Window::Welch(const SizeType nn,Vector& ww)
  {
    // Periodic case
    double xmax = static_cast<double>(nn)/2.;
    if (ww.size() != nn)
      ww.resize(nn);
    for (SizeType i = 0; i < nn; i++)
    {
      double alpha = (static_cast<double>(i) - xmax) / xmax;
      ww[i] = 1. - square(alpha);
    }
  }

  //! \fun Hamming
  //!   Generates a Hamming window
  inline void
  Window::Hamming(const SizeType nn,Vector& ww)
  {
    if (ww.size() != nn)
      ww.resize(nn);
    for (SizeType i = 0; i < nn; i++)
    {
      ww[i] = 0.54 - 0.46*cos(2.*gwd::PI*i/(nn-1));
    }
  }


} // namespace csp
#endif // WINDOW_H

