/***************************************************************************
    File        : DetectorChannel.h
    Description : Median-based noise floor tracker MNFT
 ---------------------------------------------------------------------------
    Begin       : Fri April 14 2004
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#ifndef __DETECTORCHANNEL_H
#define __DETECTORCHANNEL_H


/** \name DetectorChannel
 ** \memo Median-based noise floor tracker (MNFT): robust estimation of
 **       noise floor drifts in interferometric data.
 **
 ** \author Roberto Grosso; Brownsville, Texas
 ** \version 1.0; Last modified April 14, 2006
 **/

// Libs
#include <iostream>
#include <sstream>
#include <fstream>
#include <list>
#include <string>
#include <vector>
#include <cstdlib>
#include <limits>
#include <algorithm>


// Project files
#include "DatEnv.hh"
#include "MonServer.hh"
#include "Interval.hh"
#include "TSeries.hh"
#include "Time.hh"
#include "OperStateCondList.hh"

// HTML output
#include "html/color.hh"
#include "html/document.hh"
#include "html/table.hh"
#include "html/size.hh"
#include "html/text.hh"
#include "html/writer.hh"
#include "html/hline.hh"
#include "html/link.hh"

// Wave library
#include "Clock.h"
#include "Types.h"
#include "Utilities.h"
#include "Exception.h"
#include "Numeric.h"
#include "LowPassFilter.h"
#include "BandPassFilter.h"
#include "HighPassFilter.h"
#include "NotchFilter.h"
#include "WhiteningFilter.h"
#include "RunningMedian.h"
#include "FIRFilter.h"
#include "Multirate.h"
#include "Convolution.h"
#include "NormalDistribution.h"

// Noise Floor Monitor
#include "Parameter.h"
#include "FrequencyBand.h"

namespace gwd {


  /** \class Detectorchannel
   ** This class computes the Noise Floor of the data delivere by an specified channel
   ** of the interferometric detector. The algorithm implemented is based on the technique
   ** described in the paper Median based noise floor tracker: robust estimation of
   ** noise floor drifts in interferometric data. S. Mukherjee, Classical and Quantum
   ** Gravity, 20, S925-S936,2003.
   **/
  class DetectorChannel {
  public:
    //! Constructor
    DetectorChannel( );
    //! Destructor
    virtual ~DetectorChannel(void);

    // Methods
    //! Initialize channel.
    bool Init(const std::string& channelname,const double chsamplerate,Parameter& param,const Vector& freqs,const Vector& freqbnds);
    //! Computes the Noise Floor for this channel.
    bool ProcessData(const TSeries* ts,const Time& currentTime,const bool lockConditionFlag,const unsigned long counter);
    
    //! Get channel name.
    std::string GetName() { return mChannelName; }
    
    //! Get the time series for a frequency band.
    TSeries* GetFrequencyBand(int no)     { return &(mNoiseFloor[no].mTS);     }
    TSeries* GetFrequencyBand12hs(int no) { return &(mNoiseFloor[no].mTS12hs); }

    //! Get the number of frequency bands.
    int NoFreqBands() { return mNoFreqBands; }
    //! Get a pointer to the frequency band
    FrequencyBand* GetNoiseFloor(const int bnd) { return &(mNoiseFloor[bnd]); }

    //! Write data in binary format for debugging.
    void WriteData(const unsigned int& starttime);
    
    //! Check if the channel has delivered data.
    bool empty() { return mEmpty; }
    
  private:
    //! Channel name.
    std::string mChannelName;
    //! Length on the time stride.
    Interval mTStride;
    //! Current time
    Time mCurrentTime;
    //! Time stride in sec.
    double mTStrideSize;
    
    //! The channel has no data.
    bool mEmpty;

    //! No. of lines of channel data.
    int mNoOfLines;
    //! Order of the Notch Filter.
    unsigned int mOrderNotchFilter;
    //! Update of Whitening Filter
    unsigned int mUpdateWhiteFlt;
    //! Order of the Whitening Filter.
    unsigned int mOrderWhiteFilter;
    //! Number of frequencies in the Power Spectrum Density.
    unsigned int mNoOfFreqPsd;
    //! Size in sec. of ts. used for the computation of the PSD.
    double mSizePsd;
    //! Size is Hz. of RM filter for Whitening filter.
    double mRMSizeForWhiteningFlt;
    //! Size in sec. of the Running median filter.
    double mWindowSizeRunningMedian;
    //! Signal's sampling rate in Hz.
    double mSignalSampleRate;
    //! Processing sampling rate.
    double mSampleRate;
    //! Decimate factor for cutoff filtering 16kHz -> 4kHz.
    int mDecimateFactor;
    //! Oversampling factor.
    double mOverSamplingFactor;
    //! Filter order for low-pass and band-pass filters
    unsigned int mFltOrder;
    //! Frequency for upper cut off filtering
    double mUpperCutoffFreq;
    //! Filter order for signal bands processing.
    unsigned int mBandFltOrder;
    
    //! Number of frequency bands.
    int mNoFreqBands;
    //! The noise floor for each frequency band.
    std::vector<FrequencyBand> mNoiseFloor;
    
    //! Cutoff filter to 4kHz
    Vector mCutoffFlt;
    //! Notch Filter.
    Vector mNotchFilter;
    //! Whitening Filter.
    Vector mWhiteningFilter;
    //! Frequencies pics (lines) for the Notch filter
    Vector mFreqNotchFilter;
    //! Frequencies bands for the Notch filter
    Vector mFreqBndWidthNotFilter;

    //! Time delay due to filter operations.
    double mFilterDelay;
    //! Buffer time shift in sec.
    double mBufferTimeShift;
    //! Buffer for continuous signal processing.
    Vector buffer;

    //! List of running status messages.
    std::list<std::string*> mStatusMessage;
    //! List with warning messages.
    std::list<std::string*> mWarningMessage;
    //! List with error messages
    std::list<std::string*> mErrorMessage;

    //! Prepare data for dmtview plot: down sample.
    void PrepareDataForPlot(Vector& in,std::vector<float>& ou);
    //! Send empty plot to the DMT-viewer for non locked detector.
    void EmptyPlot();
    //! Send times series to DMT-viewer.
    void TimeSeriesPlot();
    
    //===================================== Auxiliary functions
    //! Estimates the mean value and the standard deviation of a time sereies.
    void Estimator(const Vector& ts,double& mean,double& stdDev);
    //! Mix a signal with an oscilator for base-banded frequency transformation.
    void MixWithOscilator(const double carrierFreq,const double sampleRate,Vector& signal);
  };
} // namespace gwd
#endif     //  __DETECTORCHANNEL_H
