/***************************************************************************
    File        : Crosscorrelaton.h
    Description : Estimates the sample-crosscorrelation of two time series.
 ---------------------------------------------------------------------------
    Begin       : Sun Feb 26 2006
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#ifndef __CROSSCORRELATION_H
#define __CROSSCORRELATION_H

/** @file Autocorrelation.h
 *  Estimates the normalized sample cross correlation of two time series. 
 ** A definition for the normalized sample cross correlaton is:
 ** @f[
 **   r_j = \frac{ \sum_{i=0}^{N - j} x_i \times y_{i+j} } {N - j}
 ** @f]
 **/

// Libs
#include <vector>
#include <numeric>


namespace gwd {

  /** @fn template < typename vector > bool Crosscorrelation(const vector& smpl,vector& ac)
   ** @brief
   ** Estimates the sample-crosscorrelation of two time series.
   **/
  template < typename vector >
  bool Crosscorrelation(const int lags,const vector& smpl1,const vector& smpl2,vector& cc)
  {
    if (smpl1.size() != smpl2.size())
      return false;
    typedef typename vector::size_type SizeType;
    
    // compute the ACF
    const SizeType N = smpl1.size();
    cc.resize(lags);
    for (SizeType jj = 0; jj < (SizeType)lags; jj++)
    {
      cc[jj] = double(0);
      const SizeType lim = N - jj;
      for (SizeType tt = 0; tt < lim; tt++)
      {
        cc[jj] += smpl1[tt]*smpl2[tt+jj];
      }
      cc[jj] = cc[jj] / static_cast<double>(lim);

    }

    return true;
  }
} // namespace gwd

#endif // __CROSSCORRELATION_H
