#ifndef VOLTWRITER_HH
#define VOLTWRITER_HH

#include <iostream>
#include <iomanip>
#include <math.h>
#include "Trend.hh"
#include "VoltBase.hh"
#include "VoltWatcher.hh"

   class VoltWatcher;

// Means that a slot is is use with no VoltWatcher errors
   const bool GOOD = 1;

// Means that a slot is either empty or has an error in its VoltWatcher
   const bool BAD = 0;

// maximum number of history points
   const int maxHistory = 60 * 24;

   class MonServer;


/** VoltWriter handles the results of several VoltWatchers - its major functions
are to save the results from each channel to a file and to periodically post all of the results to a webpage, summary.html.

    Source code:
    \URL[VoltWriter.hh]{source/VoltWriter.hh} and
    \URL[VoltWriter.cc]{source/VoltWriter.cc}

@memo	Handles the output for calculations on power line monitor channels
@author Charlie Shapiro
*/
   class VoltWriter : public VoltBase {
   
   public:
   
   /** Constructor.  Entire database is set to zero.  All timers are set to zero.  All slots are set to BAD.
   @param	filename Base filename for all files
   @param	HTMLfile Path and filename of the summary web page
   @param	avg Number of VoltWatcher buffers over which to average results
   */
      VoltWriter(const char* filename, const char* HTMLfile, 
                int average = 1, int ival = 1, bool harmonics = true,
                int trend = notrend, MonServer* monserver = 0);
   
   /** Constructor.  Entire database is set to zero.  All timers are set to zero. 
   All slots are set to BAD.  Default web page is used.
   @param	filename Base filename for all files
   @param	avg Number of VoltWatcher buffers over which to average results
   */
      VoltWriter(const char* filename, int average=1, int ival = 1, 
                bool harmonics = true, int trend = notrend, 
                MonServer* monserver = 0);
   
   /// Destructor
      virtual ~VoltWriter();
   
   /// Writes the trend file index
      void writeIndex();
   
   /** Make a channel name for trend and online viewing. A conforming channel
       name uses MLTV as the system name.
   @memo	Make a conforming channel name
   @param	s String to copy to (must be long enough)
   @param	name Original channel name
   @param	ext Channel name extension 
   @return 	pointer to return string
   */
      static char* makeName (char* s, const char* name, const char* ext);
      
   /** If the slot is free and the VoltWatcher passed has no errors, the VoltWatcher will write its results to the slot in the database matching its ID number.
   @memo	Add a VoltWatcher to a slot
   @param	VWatch VoltWatcher to add
   */
      void addSlot(VoltWatcher& VWatch);
   
   /** Sets the time in the specified slot to a new value, but only when the internal timer is maxed or zero
   @memo	Change the time in one slot
   @param	id Slot to change
   @param	GPSsecs New time (in GPS seconds)
   */
      void setTimeArray(const int id, const unsigned long GPSsecs)
      {
         if(timer[id]==avgFactor) timer[id] = 0;
         if(timer[id]==0) timeArray[id] = GPSsecs;
         timer[id]++;
      }
   
   /** A value is divided by <b>avgFactor</b> and ADDED to the database item.  The item is set to zero first if the internal timer is at the beginning.
   @memo	NOT THE SAME AS THE ORIGINAL METHOD IN VOLTBASE
   @param	id Slot number to add to
   @param	column Column number to add to
   @param	data Value to add
   */
      void setData(const int id, const int column, const float data)
      {
         if(timer[id]==1) database[id][column] = 0;
         database[id][column] += data/avgFactor;
      }
   
   /** Change a slot's status to GOOD or BAD.
   @param	id Slot number
   @param	status GOOD or BAD
   */
      void setSlotStatus(const int id, bool status)
      { slotStatus[id] = status; }
   
   /** Create a data file for each GOOD VoltWatcher and put fancy headers at the top of each one.
   @memo	Make data files with headers
   */
      void makeHeaders();
   
   /** Print fancy column headers on the screen to go with <b>print2Screen ()</b>
   @memo	Print headers on the screen
   */
      void printHeader();
   
   /** Add all the data from one slot in the database to the last line of that slot's data file, but only if the slot's timer is maxed
   @memo	Update a data file
   @param	id Slot number
   */
      void print2File(const int id);
   
   /** Print a slot's first few values in the database to the screen, but only if the slot's timer is maxed
   @memo	Print data to screen
   @param	id Slot number
   */
      void print2Screen(const int id) const;
   
   /** Print the entire database to a webpage (summary.html), but only if the timers for ALL slots are maxed
   @memo	Update the webpage
   */
      void print2Web();
   
   /// Set all slots' timers to zero
      void reset() { 
         for(int id=0; id<numChannels; id++) timer[id]=0; }
   
   protected:
   
   /** Makes a filename for each slot.  Filename's base is the string passed to the function, and its extension is the slot number.
   @param	filename Base for filenames
   */
      void makeFilenames (const char* filename);
   
   /** The GOOD/BAD status of each slot.  Status is GOOD if the slot is in use by a
   VoltWatcher with no errors.  Status is BAD if the slot is not in use or if its
   VoltWatcher has an error.
   */
      bool slotStatus[maxChannels];
   
   /// Name of each slot's channel
      char channelNames[maxChannels][80];
   
   /// Name of each slot's data file
      char fileNames[maxChannels][80];
   
   /// Location of web page
      char webname[200];
   
   /// Include harmonics in output?
      bool useharmonics;
   
   /// Use trend instead of ASCII files
      int usetrend;
   
   /// Trender
      Trend trend;
   
   /// Output stream for data files
      std::ofstream opd;
   
   /// Output stream for the webpage
      std::fstream web;
   
   /// monitor server
      MonServer* mserv;
   
   /// History arrays (rms, frequency, crest value, THD)
      float hist[maxChannels][4][maxHistory];
   /// History array length
      int histlen[maxChannels];
   /// History array start time
      Time histStart[maxChannels];
   /// Number of seconds per point
      const float interval;
   
   /// add a history point
      void addHistPoint (int id);
   
   /// History for monitor server (rms, frequency, crest value, THD)
      TSeries histTS[maxChannels][4];
   
   /// Number of VoltWatcher buffers over which to average calculations
      const int avgFactor;
   
   /** Number of partial calculations added so far to each slot in the database. 
   Using <b>setTimeArray ()</b> increments the timer.  The timer resets when it
   equals <b>avgFactor</b>.
   */
      int timer[maxChannels];
   
   };

#endif
