#include <time.h>
#include <iostream>
#include <fstream>
#include <iomanip>
#include <cmath>
#include <cstdio>
#include <cstring>
#include "VoltWriter.hh"
#include "VoltWatcher.hh"
#include "MonServer.hh"

using namespace std;

   VoltWriter::VoltWriter (const char* filename, const char* HTMLfile, 
                     int average, int ival, bool harmonics, int trnd, 
                     MonServer* monserver)
   : useharmonics (harmonics), usetrend (trnd), mserv (monserver),
   interval (ival), avgFactor(average)
   {
      for (int i = 0; i < maxChannels; i++) {
         histlen[i] = 0;
      }
      numChannels = maxChannels;
      numColumns=3 + maxHarmonics*2;
   
      for(int id=0; id<numChannels; id++)
      {
         slotStatus[id]=BAD;
         strcpy (fileNames[id], "");
         timer[id] = 0;
         for(int n=0; n<numColumns; n++) database[id][n]=0;
      }
      if (usetrend == usrtrend) {
         cout << "vwriter 1A" << endl;
         trend.setFile (filename);
         trend.setType (Trend::kMinute);
      }
      else if (usetrend == deftrend) {
         cout << "vwriter 1B" << endl;
         trend.setName (filename);
         trend.setType (Trend::kMinute);
      }
      else if (strlen (filename) > 0) {
         opd.setf(ios::left);
         makeFilenames(filename);
      }
      cout << "vwriter 2" << endl;
      web.setf(ios::left);
      if (HTMLfile) {
         strcpy(webname, HTMLfile);
      }
      else {
         strcpy(webname, "");
      }
      cout << "vwriter 3" << endl;
   }

   VoltWriter::VoltWriter (const char* filename, int average, 
                     int ival, bool harmonics, int trnd, 
                     MonServer* monserver)
   : useharmonics (harmonics), usetrend (trnd), mserv (monserver),
   interval (ival),  avgFactor(average)
   {
      for (int i = 0; i < maxChannels; i++) {
         histlen[i] = 0;
      }
      numChannels = maxChannels;
      numColumns=3 + maxHarmonics*2;
   
      for(int id=0; id<numChannels; id++)
      {
         slotStatus[id]=BAD;
         timer[id] = 0;
         for(int n=0; n<numColumns; n++) database[id][n]=0;
      }
   
      if (usetrend == usrtrend) {
         cout << "vwriter 1A" << endl;
         trend.setFile (filename);
         trend.setType (Trend::kMinute);
      }
      else if (usetrend == deftrend) {
         cout << "vwriter 1B" << endl;
         trend.setName (filename);
         trend.setType (Trend::kMinute);
      }
      else if (strlen (filename) > 0) {
         opd.setf(ios::left);
         makeFilenames(filename);
      }
      web.setf(ios::left);
      strcpy(webname, "");
   }


   VoltWriter::~VoltWriter () 
   {
      if (mserv) {
         delete mserv;
      }
   }


   void VoltWriter::writeIndex () 
   {
      if (usetrend != notrend) {
         trend.writeIndex();
      }
   }

   char* VoltWriter::makeName (char* s, const char* name, const char* ext)
   {
      char* ss = s;
      for (const char* p = name; *p; ++p) {
         if (*p == ':') {
	    *ss++ = ':';
	    *ss++ = 'M';
	    *ss++ = 'L';
	    *ss++ = 'T';
	    *ss++ = 'V';
	    *ss++ = '-';
	 }
	 else if (*p == '-') {
	    *ss++ = '_';
	 }
	 else {
	    *ss++ = *p;
	 }
      }
      sprintf (ss, "_%s", ext);
      return s;
   }


   void VoltWriter::addSlot(VoltWatcher& VWatch)
   {
      int id = VWatch.GetID();
   
      if(id >= numChannels)
         cout << "Could not add " << VWatch.GetChannelName() << " to slot "
            << "(ID# too high)\n";
      else if(slotStatus[id] == GOOD)
         cout << "Could not add " << VWatch.GetChannelName() << " to slot "
            << "(ID# already in use)\n";
      else if(VWatch.GetErrorStatus() == true)
         cout << "Could not add " << VWatch.GetChannelName() << " to slot "
            << "(it has an error)\n";
      else {
         // setup slot
         strcpy(channelNames[id],VWatch.GetChannelName());
         slotStatus[id] = GOOD;
      	 // intialize trend
         if (usetrend != notrend) {
            cout << "ADD TREND CHANNELS" << endl;
            char name[1024];
            makeName (name, channelNames[id], "RMS");
            cout << "ADD TREND RMS " << name << endl;
            trend.addChannel (name);
            makeName (name, channelNames[id], "FREQ");
            trend.addChannel (name);
            makeName (name, channelNames[id], "CF");
            trend.addChannel (name);
            makeName (name, channelNames[id], "THD");
            trend.addChannel (name);
            makeName (name, channelNames[id], "A1");
            trend.addChannel (name);
            if (useharmonics) {
               for (int i = 2; i <= maxHarmonics; i++) {
                  sprintf (name, "%s_%s%i", channelNames[id], "A", i);
                  trend.addChannel (name);
                  sprintf (name, "%s_%s%i", channelNames[id], "P", i);
                  trend.addChannel (name);
               }
            }
         }
      	 // initialize monitor server
         if (mserv) {
            char name[1024];
            makeName (name, channelNames[id], "RMS");
            histTS[id][0].setName (name);
            mserv->serveData (name, histTS[id] + 0, 0);
            makeName (name, channelNames[id], "FREQ");
            histTS[id][1].setName (name);
            mserv->serveData (name, histTS[id] + 1, 0);
            makeName (name, channelNames[id], "CF");
            histTS[id][2].setName (name);
            mserv->serveData (name, histTS[id] + 2, 0);
            makeName (name, channelNames[id], "THD");
            histTS[id][3].setName (name);
            mserv->serveData (name, histTS[id] + 3, 0);
         }
      	 // set writer in volt watcher
         VWatch.SetWriter(this);
      }
   }

   void VoltWriter::makeFilenames(const char* filename)
   {
      for(int id=0; id < numChannels; id++)
      {
         sprintf (fileNames[id], "%s.%02i", filename, id);
      }
   }

   void VoltWriter::makeHeaders()
   {
      if (usetrend != notrend) 
         return;
      for(int id=0; id<numChannels; id++)
      {
         if ((slotStatus[id] == BAD) || (strlen (fileNames[id]) == 0)) 
            continue;
      
         opd.open(fileNames[id]);
         opd << "Data for channel " << channelNames[id] << "\n\n"
            << "Buffer Size=" << 60 << " seconds"
            << "                            "
            << "Harmonics: % of A1, Phase\n\n"
            << "Start Time  RMS      CF       Freq.    THD      A1         ";
         if (useharmonics) {
            for(int h=2; h<=maxHarmonics; h++) { 
               opd.width(18); 
               opd << h; 
            }
         }
      
         opd << endl;
         opd << "~~~~~~~~~~#" << endl;
         opd.close();
      }
   }

   void VoltWriter::printHeader()
   {
      cout<<"\nChannel Name\tFill Time\tRMS\tCF\tFreq.\tTHD\tA1\n"
         <<"~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n";
      cout.flush();
   }

   void VoltWriter::print2File(const int id)
   {
      if(timer[id] != avgFactor) 
         return;
   
      // trend file
      if (usetrend != notrend) {
         char name[1024];
         Time time (timeArray[id]);
         //cout << "WRITE TREND @ " << time << endl;
         makeName (name, channelNames[id], "RMS");
         trend.trendData (name, time, database[id][v_RMS]);
         //cout << "WRITE TREND " << name << ": " << database[id][v_RMS] << endl;
         makeName (name, channelNames[id], "FREQ");
         trend.trendData (name, time, database[id][v_MAXF]);
         makeName (name, channelNames[id], "CF");
         trend.trendData (name, time, database[id][v_FACTOR]);
         makeName (name, channelNames[id], "THD");
         trend.trendData (name, time, database[id][v_THD]);
         makeName (name, channelNames[id], "A1");
         trend.trendData (name, time, database[id][v_A1]);
         if (useharmonics) {
            for (int i = 2; i <= maxHarmonics; i++) {
               sprintf (name, "%s_%s%i", channelNames[id], "A", i);
               trend.trendData (name, time, database[id][v_HARM+2*(i-2)]);
               sprintf (name, "%s_%s%i", channelNames[id], "P", i);
               trend.trendData (name, time, database[id][v_HARM+2*(i-2)+1]);
            }
         }
      }
      // ASCII file
      else if (strlen (fileNames[id]) > 0) {
         opd.open(fileNames[id], ios::app);
         opd.width(12);
         opd << timeArray[id];
      
         opd.precision(6);
         for (int n=v_RMS; n<=v_THD; n++)
         {
            opd.width(10); opd << database[id][n];
         }
      
         opd.precision(4);
         opd.width(12); opd << database[id][v_A1];
         if (useharmonics) {
            for (int n=v_HARM; n<v_HARM+2*maxHarmonics; n+=2)
            {
               opd.width(12); opd << database[id][n];
               opd.width(10); opd << database[id][n+1];
            }
         }
         opd << endl;
         opd.close();
      }
   
      // monitor server
      if (mserv) {
         addHistPoint (id);
         histTS[id][0].setData (histStart[id], interval, hist[id][0], histlen[id]);
         histTS[id][1].setData (histStart[id], interval, hist[id][1], histlen[id]);
         histTS[id][2].setData (histStart[id], interval, hist[id][2], histlen[id]);
         histTS[id][3].setData (histStart[id], interval, hist[id][3], histlen[id]);
      }
   }

   void VoltWriter::print2Screen(const int id) const
   {	
      if(timer[id] != avgFactor) 
         return;
   
      cout << channelNames[id] << "\t";
      cout << timeArray[id];
      for(int n=v_RMS; n <= v_A1; n++) cout << "\t" << database[id][n];
      cout << endl;
   }

   void VoltWriter::print2Web()
   {
      for(int id=0; id<numChannels; id++)
         if(slotStatus[id] == GOOD && timer[id]!=avgFactor)
            return;
   
      char timeStr[40];
      int someID=0;
      while(slotStatus[someID] == BAD) someID++;
   
      LocalStr(Time(timeArray[someID]), timeStr, "%H:%N:%S on %M %d, %Y\n");
   
      if (strlen (webname) > 0) {
         web.open(webname, ios::out);
         web
            << "<HTML>\n<HEAD>\n<TITLE>Power Line Data Summaries</TITLE>\n</HEAD>\n\n"
            << "<BODY BGCOLOR=\"#FFFFDD\">\n\n"
            << "<H2>Power Line Data Summaries</H2>\n"
            << "List of terms:\n<BR><BR>\n"
            << "<UL>\n"
            << "<LI>RMS - The root mean square of the voltage</LI>\n"
            << "<LI>CF - Crest factor (maximum value / RMS)</LI>\n"
            << "<LI>Freq. - The fundamental frequency</LI>\n"
            << "<LI>THD - Total harmonic distortion expressed as a percentage</LI>\n"
            << "<LI>A1 - Amplitude of the fundamental frequency</LI>\n"
            << "<LI>Harmonics - For each harmonic number...\n"
            << "\t<OL>\n"
            << "\t<LI>The amplitude of that harmonic expressed as a percentage of A1</LI>\n"
            << "\t<LI>The phase relative to A1 (in degrees)</LI>\n"
            << "\t</OL></LI>\n"
            << "</UL>\n"
            << "<BR><BR>\n\n"
            
            << "Averages for the last minute... (last updated: " << timeStr << " )\n"
            << "<HR>\n\n"
            << "<TABLE cellpadding=10 border=0>\n"
            << "<TR><TD colspan=7>Buffer Size=10 seconds</TD>"
            << "<TD colspan = 5>Harmonics:</TD></TR>\n"
            
            << "<TR align=left><TH>ID#</TH><TH>Channel</TH><TH>RMS</TH><TH>CF</TH><TH>Freq.</TH>"
            << "<TH>THD</TH><TH>A1</TH>";
         for(int n=2; n<=maxHarmonics; n++) web<<"<TH>"<<n<<"</TH><TH></TH>";
         web << "</TR>\n";
      
         for(int id=0; id<numChannels; id++)
         {	
            web<<"<TR><TD>"<<id<<"</TD>";
            if(slotStatus[id]==GOOD)
            {
               web << "<TD>" << channelNames[id] << "</TD>";
               for(int n=0; n<numColumns; n++)
               {
                  web << "<TD>";
                  web.width(12);
                  web << database[id][n];
                  web << "</TD>";
               }
            }
            
            else 
               for(int n=-1; n<numColumns; n++) web << "<TD>     --     </TD>";
         
            web << "</TR>\n";
         }
      
         web
            << "</TABLE>\n\n"
            << "<HR>\n\n"
            << "</BODY>\n"
            << "</HTML>";
      
         web.close();
      }
   }	// end of print2Web


   void VoltWriter::addHistPoint (int id)
   {
      Time time (timeArray[id]);
   
      // first data point?
      if (histlen[id] == 0) {
         hist[id][0][0] = database[id][v_RMS];
         hist[id][1][0] = database[id][v_MAXF];
         hist[id][2][0] = database[id][v_FACTOR];
         hist[id][3][0] = database[id][v_THD];
         histlen[id] = 1;
         histStart[id] = time;
         return;
      }
   
      // first check how many points to add
      Interval dt = time - histStart[id];
      int diff = (int) ((double) dt / interval + 0.5) - histlen[id];
      if (diff <= 0) {
         return;
      }
      // check if we have to drop all points: if yes, start from scratch
      if (diff >= maxHistory) {
         hist[id][0][0] = database[id][v_RMS];
         hist[id][1][0] = database[id][v_MAXF];
         hist[id][2][0] = database[id][v_FACTOR];
         hist[id][3][0] = database[id][v_THD];
         histlen[id] = 1;
         histStart[id] = time;
         return;
      }
   
      // check if we have to move points
      if (diff + histlen[id] > maxHistory) {
         int ofs = diff + histlen[id] - maxHistory;
         memmove (hist[id][0], hist[id][0] + ofs, 
                 (maxHistory - ofs) * sizeof (float));
         histStart[id] = time - Interval ((maxHistory - 1) * interval);		
         histlen[id] -= ofs;
      }
      // append
      for (int i = histlen[id]; i < histlen[id] + diff; i++) {
         hist[id][0][i] = 
            (i == histlen[id] + diff - 1) ? database[id][v_RMS] : 0;
         hist[id][1][i] = 
            (i == histlen[id] + diff - 1) ? database[id][v_MAXF] : 0;
         hist[id][2][i] = 
            (i == histlen[id] + diff - 1) ? database[id][v_FACTOR] : 0;
         hist[id][3][i] = 
            (i == histlen[id] + diff - 1) ? database[id][v_THD] : 0;
      }
      histlen[id] += diff;
   }


