#ifndef VOLTWATCHER_HH
#define VOLTWATCHER_HH

#ifndef __CINT__
#include <iostream>
#include <fstream>
#include "DaccAPI.hh"
#include "TSeries.hh"
#include "Time.hh"
#include "Hanning.hh"
#include "VoltBase.hh"
#include "VoltWriter.hh"
#endif

//unsigned long integer
   typedef unsigned long u_long;
//unsigned short integer
   typedef unsigned short u_short;

   const bool EVEN = 0;
   const bool ODD = 1;
   const bool PRINT = 1;
   const bool NOPRINT = 0;

   class VoltWriter;

/**  A VoltWatcher object is designed to read real-time data from an LHO 
  *  online power-monitoring channel and periodically produce calculations.  
  *  It obtains data via its static DaccAPI (Data Access) member and processes
  *  the data using a two-buffer system.  A VoltWatcher contains two TSeries 
  *  buffers - an "even" buffer and an "odd" one.  A buffer is either 
  *  "passive" or "active".  "Active" means it is in the process of being 
  *  filled with data, and "passive" means it is not.  While the active buffer
  *  is being filled, calculations can be performed on the passive buffer.  
  *  Once the active buffer is full, it becomes passive, and the other buffer 
  *  is cleared and becomes active.
  *  <p>
  *  A VoltWatcher sends its calculations to a VoltBase object (if one has 
  *  been assigned) and does not output its results any other way.  A 
  *  VoltWriter (or another object derived from VoltBase) should always 
  *  accompany a VoltWatcher so that the results can be printed to the screen,
  *  saved to a file, etc.
  *  <p>
  *  CAUTION: Although the internal timing of the VoltWatcher object is 
  *  automatic, its member functions must be carefully implemented externally 
  *  in order for it to work properly. A simple example program for monitoring
  *  one channel is vwatch.cc; a more complicated program for monitoring 
  *  several channels at once is multivw.cc.
  *  <p>
  *  NOTE: VoltWatchers get their configurations from config.vw - if the 
  *  sampling rate specified is less than 2048 Hz or not a power of 2, it will
  *  be rejected.
  *
  *  Source code:
  *  \URL[VoltWriter.hh]{source/VoltWriter.hh}, 
  *  \URL[VoltWriter.cc]{source/VoltWriter.cc} and
  *  \URL[VWMath.cc]{source/VWMath.cc}
  *
  *  @memo	A class for watching volts
  *  @author	Charlie Shapiro
  *  @version	1.0; Last modified July 26, 2000
  *  @see	VoltBase
  *  @see	VoltWriter
  *  @see	TSeries
  *  @see	DaccAPI
  */
   class VoltWatcher {
   
   public:
   
     /**@name Constructors / Destructors */
     //@{
     /** Construct a VoltWatcher based on an entry in the config.vw file
      *	@memo	Automatic constructor
      *	@param	ID The channel ID# in config.vw
      *	@param 	bufSize Number of seconds in a filled buffer (default=10)
      */
     VoltWatcher (DaccAPI& In, int ID, const char* configfile, 
		  u_short bufSize=10);
   
     /** Construct a VoltWatcher explicitly
      *	@memo	Explicit constructor
      *	@param	channel The channel name (e.g. "H0:PEM-LVEA_V1")
      *	@param	offset The DC offset in ADC units
      *	@param	scale The ratio of ADC units to volts
      *	@param	ID The channel ID# in config.vw
      *	@param 	bufSize Number of seconds in a filled buffer
      */
      VoltWatcher(char* channel, float offset, float scale, int ID, 
		  u_short bufSize=10);
   
     /// Destructor
   
     ~VoltWatcher() {
       delete[] S; delete[] _chanName;}
     //@}
   
   
     /**@name Processing functions */
     //@{
     /**	Fills the DaccAPI for one second (for all channels added!)
      *	@memo	Fill the DaccAPI for one second
      *	@return	The GPS time (in seconds) at which the method was called
      */
     //static size_t Fill();
   
     /**  Appends the time series data in the DaccAPI to the active buffer.  
       *  If the buffer was full, it is cleared first.  If the buffer becomes 
       *  filled, it becomes passive, and the other buffer becomes active.
       *  @memo	Update the active buffer
       */
      void AppendBuffers();
   
   /** 	Any frames waiting in the DaccAPI's buffer are skipped.  <b>Fill ()</b>
   *	is then called until the current GPS time has a value of <b>delay</b>
   *	in the seconds field.  For example, if <b>Delay (30)</b> is called at
   *	12:16:45, the current time will be 12:16:30 when it is finished.  If
   *	<b>delay</b> is 60 or greater, <b>Fill ()</b> will not be called.
   *	@memo	Wait for up to one minute
   *	@param	delay GPS seconds field to wait for
   */
      //static void Delay(const u_short delay=61);
   
   /** 	The passive buffer is "fixed" - i.e. the DC offset is removed, the
   *	amplitude is scaled down from ADC units to volts, and a Hanning
   *	window is applied.  All calculations are then performed on this data.
   *	The results and the start time of the data are sent to a VoltBase
   *	object if the VoltWatcher has been assigned to one.
   *	@memo	Adjust the passive buffer and perform all calculations
   */
      int Analyze();
   
   /**	The channel is monitored for the specified duration by continuously
   *	filling the buffers and making calculations.  If the duration is set
   *	to zero, the method will run indefinitely. <b>Delay ()</b> is used
   *	before beginning.
   *	@memo	Monitor one channel
   *	@param	duration Number of buffers to fill before completion
   *	@param	delay GPS seconds field to wait for before beginning
   *	@return	The VoltWatcher's ID#
   */
      void Watch(const u_short duration=0, const u_short delay=61);
   //@}
   
   
   /**@name Calculation functions */
   //@{
   /**	Calculate the root mean square of a TSeries object
   *	@memo	Root mean square
   *	@param	tVolt A time series object
   *	@return	The root mean square of the time series
   */
      float RMS();
   
   /**	The data in the passive buffer is used to calculate the complex
   *	Fourier coefficient for the given frequency.
   *	@memo	Find a complex Fourier coefficient
   *	@param	freq Frequency
   *	@return	The Fourier coefficient as a complex float object
   */
      fComplex GetCoef(const float freq) const;
   
   /**	The frequency with the highest amplitude in a small neighborhood is
   *	found for the passive buffer.  The center of that neighborhood
   *	(i.e. an initial guess) is required.
   *	@memo	Find a dominant frequency
   *	@param	freq Best guess of what the frequency should be
   *	@param	YNPrint If PRINT is specified, the amplitude of the
   *			frequency found will be written to the screen
   *	@return	The dominant frequency in the neighborhood of the specified
   *			frequency
   */
      float ApproxMaxF(const float freq, const bool YNPrint=NOPRINT) const;
   
   /**	The amplitudes and phases of all harmonics of the specified
   *	fundamental frequency (up to the Nyquist frequency) are calculated.
   *	The amplitudes are expressed as a percentage of the fundamental
   *	frequency's.  The phases are relative to the fundamental frequency
   *	and are expressed in degrees.  The total harmonic distortion is also
   *	calculated and returned.  All results are sent to a VoltBase object
   *	if one has been assigned.
   *	@memo	Harmonic properties
   *	@param	freq The fundamental frequency
   *	@param	YNPrint If PRINT is specified, all amplitudes and phases
   *			will be written to the screen
   *	@return	The total harmonic distortion
   */
      float Harmonics(const float freq);
   //@}
   
   /**@name Accessor functions */
   //@{
   /**	Get the error status
   *	@return	<b>true</b> if an error has occurred
   */
      bool GetErrorStatus() const { 
         return _errStat;}
   
   /**	A VoltWatcher's ID number is also the slot number in its VoltBase
   *	@memo	Get the ID#
   *	@return	The ID#
   */
      int GetID() const { 
         return _ID; }
   
   /**	Get the name of the channel being monitored
   *	@memo	Get the channel name
   *	@return	Channel name
   */
      char* GetChannelName() const { 
         return _chanName; }
   //@}
   
   /**	The VoltWriter will now send its calculations to the VoltBase
   *	specified by the pointer.  One should make sure that two channels with
   *	the same ID number do not use the same VoltBase.  VoltWriter's
   *	<b>addSlot ()</b> function is safer to use.
   *	@memo	Assign a VoltBase object
   *	@param	VWrite A VoltBase pointer
   */
      void SetWriter(VoltBase* VWrite) { 
         _VWrite = VWrite; }
   
   /**	Clear the data out of both buffers, set the buffer status to EVEN, and
   *	set the count to zero.  These are the conditions immediately after
   *	construction.
   *	@memo	Clear buffers and start over
   */
      void Reset()
      {
         _bufStat=EVEN;
         _bufCount=0;
         tVolt_even.Clear();
         tVolt_odd.Clear();
      }
   
   protected:
   /**	Called during construction.  
   *	Uses the ID number to get configuration information from config.vw. 
   *	Information includes channel name, DC offset, scaling factor, and
   *	sampling rate.  Errors are tested for and flagged if found.
   *	<b>TestChannel ()</b> is also called - a decimation factor is passed
   *	to it if the sampling rate is over 2048 Hz.
   *	@memo	Get the VoltWatcher's configuration from file
   *    @param configfile configuration file
   */
      void GetConfiguration(DaccAPI& In, const char* configfile);
   
   /**	Tests whether the channel name is OK.  An error is flagged if the
   *	channel is already requested on the DaccAPI or if the channel is not
   *	found.  If the DaccAPI is having trouble reading frames, <b>exit ()</b>
   *	is called.  The channel name is added to the DaccAPI (with a decimation
   *	factor) only if it checks out.
   *	@memo	Test whether the channel name is OK.
   *	@param	decim8 Factor by which to pre-decimate data from the channel
   */
      void TestChannel(DaccAPI& In, const int decim8);
   
   /**	Flags an error (sets <b>_errStat</b> to <b>true</b>) and displays an
   *	error message.
   *	@param	message Error message to display
   */
      void ErrMsg(const char* message)
      {
         _errStat = true;
         std::cout << "ERROR: ID# " << _ID << " " << message << std::endl;
      }
   
   /// Channel ID# (and slot# in a VoltBase)
      const int _ID;
   
   /// Name of the monitored channel
      char* _chanName;
   
   /// Pointer to the VoltBase (where data is sent)
      VoltBase* _VWrite;
   
   /// Number of seconds in a filled buffer
      const u_short _bufSize;
   
   /// Number of data points in a filled buffer
      u_long n;
   
   /// Sampling interval
      float dt;
   
   /// Cleans up the frequency spectrum
      Hanning window;
   
   /// DC offset
      float _offset;
   
   /// How much to scale down amplitude
      float _scale;
   
   /// Pointer to the channel's TSeries in the DaccAPI
      TSeries* TS;
   
   /// Even buffer
      TSeries tVolt_even;
   
   /// Odd buffer
      TSeries tVolt_odd;
   
   /// Pointer to a complex array copied from passive buffer
      fComplex* S;
   
   
   /// Denotes the active buffer (EVEN or ODD)
      bool _bufStat;
   
   /// Number of seconds currently filled in the active buffer
      u_short _bufCount;
   
   /// Denotes whether an error occured during configuration
      bool _errStat;
   
   };

#endif
