#ifndef VOLTBASE_HH
#define VOLTBASE_HH

// The maximum number of channels a VoltBase can store
   const int maxChannels = 20;

// The maximum harmonic number to store data for
   const int maxHarmonics = 17;

   enum {v_RMS, v_FACTOR, v_MAXF, v_THD, v_A1, v_HARM};
   enum {notrend, usrtrend, deftrend};

   class VoltWatcher;

/** A VoltBase object can temporarily store calculations from several VoltWatcher objects.  Child classes (such as VoltWriter) can be derived from VoltBase and designed to output the calculations various ways.

    Source code:
    \URL[VoltBase.hh]{source/VoltBase.hh}

@memo Base class for power monitor summary output
@author Charlie Shapiro
*/
   class VoltBase {
   
   protected:
   
   /** Two dimensional array for storing calculations. The first dimension is the slot number (a VoltWatcher's ID number).  It tells which VoltWatcher is writing data to that slot.  The second dimension denotes the different calculated quantities.
   */
      float database[maxChannels][4 + maxHarmonics*2];
   
   // The last GPS time reported by each VoltWatcher
      unsigned long timeArray[maxChannels];
   
   // Number of total slots
      int numChannels;
   
   // Number of columns (size of the second dimension) in <b>database</b>
      int numColumns;
   
   public:
   
   /** Default Constructor.  Every item in <b>database</b> is set to zero
   */
      VoltBase()
      : numChannels(maxChannels), numColumns(3 + maxHarmonics*2)
      {
         for(int m=0; m<numChannels; m++) {
            for(int n=0; n<numColumns; n++)
               database[m][n]=0; }
      }
   
   /** Change an item in the database
   @param	id Slot number of item
   @param	column Column number of item
   @param	data New value
   */
      virtual void setData(const int id, const int column, const float data)
      { database[id][column] = data; }
   
   /** Change a time in the time array
   @param	id Slot number of time to change
   @param	GPSsecs New value (in GPS seconds)
   */
      virtual void setTimeArray(const int id, const unsigned long GPSsecs)
      { timeArray[id] = GPSsecs; }
   
      const unsigned long getTime(const int id) const
      { 
         return timeArray[id]; }
   
   // Destructor
      virtual ~VoltBase() {
      }
   
   };

#endif
