// BicoMon: The LIGO Bicoherence Background Monitor
//
// Version 1.0 - 10 January 2004 - Steven Penn
// ---------------------------------------------
//
// BicoMon runs as a background data monitor for use with LIGO's Data Monitoring
// Tool (DMT).  The monitor calculates the autobicoherence for a single input 
// channel or the cross-bicoherence if there are either 2 or 3 input data 
// channels.  The monitor then tracks the integrated bicoherence at a given 
// frequency pair (f1, f2) within a specified radius (f_rad).  The result is 
// recorded in a results file.  It is also sent to the DMT Viewer where it is 
// displayed in a minute trend format.  When the monitor is first started it 
// reads in a configuration file which specified each bicoherence array to be 
// calculated and, for each array, it lists which integrated areas are to be recorded.
// 
// 
// A typical Configuration file looks like:
// 
// ---------- (START OF FILE) -------------
// 2  9
//
// C 1     H1:LSC-AS_Q             16384   
//       256       1.0     0.5     64
//    
// M  Bico:H1:AS_Q_ALL           0   0 0
// M  Bico:H1:AS_Q_120_2_2     120   2 2
// M  Bico:H1:AS_Q_120_10_2    120  10 2
// M  Bico:H1:AS_Q_120_38_2    120  38 2
// 
// C 2     H1:LSC-AS_Q             16384   
//       H1:SUS-ITMY_OPLEV_PERROR  2048
//       256       1.0     0.5     64
//    
// M  Bico:H1:AS_Q-ITMY_OPLEV_PERROR_ALL         0  0 0
// M  Bico:H1:AS_Q-ITMY_OPLEV_PERROR_60_2_2     60  2 2
// M  Bico:H1:AS_Q-ITMY_OPLEV_PERROR_60_10_2    60 10 2
// M  Bico:H1:AS_Q-ITMY_OPLEV_PERROR_60_38_2    60 38 2
// M  Bico:H1:AS_Q-ITMY_OPLEV_PERROR_60_50_2    60 50 2
// 
// ---------- (END OF FILE) -------------
// 
// The first line lists the (# of configurations), (# of measurements)
// 
// A configuration is denoted by the first character being a "C"
// A measurement is denoted by the first character being an "M"
// 
// A configuration is specified as:
// --------------------------------
// C (# channels)  (Channel 1 name)  (Channel 1 Rate, in Hz)
//                 (Channel 2 name)  (Channel 2 Rate, in Hz)     if needed
//                 (Channel 3 name)  (Channel 3 Rate, in Hz)     if needed
//    (Freq. Max) (Freq. Resolution) (% Overlap of adj. FFTs) (Measurement Span, in sec)
// 
// 
// 
// A measurement is specified as:
// ------------------------------
// M (Name of measurement)  (f1)  (f2) (f_rad)
// 
// 
// Usage:
// ------
// 
//  BicoMon ConfigFileName ResultsFileName
//  
//  
// Future Planned Additions:
// -------------------------
//  -- fix interaction with Trend Server so that minute trends recorded even when
//         update to server occur at intervals greater than 1 minute.
//  -- include test for lock state.  Suspend operation while not locked. 
//  -- auto default choice for config and results files.
//  -- calculate gaussianity from total integrated autobicoherence.
//  -- including heterodyning to allow greater resolution for selected freq regions
//  -- calculate the change in bicoherence.
//  
// ---------------------------------------------------------------------------
//
//
//
// Version 0.1 - 10 September 2003 - Steven Penn
// ---------------------------------------------
//
// This monitor is the first iteration of a monitor which is being designed 
// to perform several common measurements associated with the bicoherence. 
// This first beta version will calculate the autobicoherence of the 
// gravity wave channel, integrate over the unique region, compare the result
// to a chi-squared distribution, and return the probability that the input
// time series was gaussian.  
//
//
// ---------------------------------------------------------------------------

// INCLUDE FILES
//--------------
#include <string>
#include <cstring>
#include <fstream>      // Needed by ConfigFile routines
#include <iostream>
#include "BicoMon.hh"


using namespace std;


// DECLARE CONSTANTS
// -----------------
const char* const kMonName = "BicoMon";   // Monitor name
const Int_t kFreqBinMin = 64 ;            // Min Number of Freq Bins


// MAIN:  Connect BicoMon into DMT Framework
// -----------------------------------------
EXECDAT (BicoMon);  


// *************************************************************************
//
//   CONSTRUCT BICOMON: Initialize Data Server, and Subscribe Data Objects
//
// *************************************************************************
BicoMon::BicoMon (int argc, const char* argv[]) 
   : DatEnv (argc, argv), MonServer (kMonName), 
  fNumMeas (0), fNumConf (0), fNumChanUnique (0), fMeasLimit (0),
  fBicoTrendTS (0), fBicoSum (0), fSpanMax (0),
  fOnline (false), fTestIFOstate (false)

{
   // Initialize Everything
   // ---------------------
   for (int ik = 0; ik < fMaxChan; ++ik) {
      fDataUnique[ik] = 0;
//       fDecFilter[ik] = 0;
//       fChanList[ik] = 0;
   }
   
   
   // Check Data Source, Determine IFO, Set State to Test
   std::string fDMTinput = std::string(getDacc().getFile());	  // Get input source
   fOnline = getDacc().isOnline();                // Test if Dacc input is Online  
   //    std::string fDMTinput = getenv("DMTINPUT");	  // Get input source
   //    fOnline = (fDMTinput.find("online") > -1);	  // Test if online
   if (fOnline) {
      if (fDMTinput.find("LHO") != string::npos) {
         fIFOname = "LHO";
      } else if (fDMTinput.find("LLO") != string::npos){
         fIFOname = "LLO";
      } else {
         fIFOname = "";
      }
   }

   // Process Options. Get Config and Results file names 
   for (int ik = 1; ik < argc; ++ik) {
      std::string argik = argv[ik];
      if (argik[0] != '-') {
	 if (fConfigFileName.empty()) {
	    fConfigFileName = argv[ik];
	 } else if (fResultsFileName.empty()) {
	    fResultsFileName = argv[ik];
	 } else {
	    cout << "Error:  Unrecognized parameter = " << argv[ik] << endl;
	 }
      } else if (argik == "-h" || argik == "--help") {
	 cout << "Usage: BicoMon [ConfigFileName ResultsFileName]" << endl;
	 cout << "       where [options] should be entered without brackets." 
	      << endl;
	 cout << "Options: ConfigFileName = name of the Configuration File, and"
	      << endl;
	 cout << "         ResultsFileName = name of the Results File" << endl;
	 finish();
      }
   }
   
   // Build Config and Results file names if none provided
   if (fResultsFileName.empty()) {
      char nowDateString[20];
      TimeStr(Now(), nowDateString, "%d%M%y_%H%N");
      fResultsFileName = std::string("BICO_") + nowDateString + std::string(".txt");
   }
   if (fConfigFileName.empty()) {
      fConfigFileName = std::string("BicoConfig") + std::string(fIFOname) + std::string(".txt");
   }   
   
   
     
   // Open Config File
   if (!BicoMon::OpenConfigFile()) {
      cout << " Error opening configuration file. Exiting."<<endl;
      finish();
   }


   // Read Configuration File
   if (!BicoMon::ReadConfigFile()) {
      cout << " Error reading configuration file. Exiting."<<endl;
      finish();
   }

      
   
   // Find Unique Data Channels
   if (!BicoMon::FindUniqueChans()) {
      cout << " Error finding unique data channels. Exiting."<<endl;
      finish();
   }
      
   
   // Here is where we need to insert a routine to select the proper state to test
   // Put in the default for now since we are not testing 
   fIFOstate = "L1:Both_arms_locked_strict_cm";
   
   
   // Print Configuration
   #ifdef BM_DEBUG
   BicoMon::PrintConfig();
   #endif
      
   
   // Assign Data Channels
   if (!BicoMon::AssignDataChannels()) {
      cout << " Error assigning data channels. Exiting."<<endl;
      finish();
   }
      
   
   // Start the Results File
   #ifdef BM_DEBUG 
   cout << "Writing Results file" << endl;
   #endif
   
   if (!BicoMon::StartResultsFile()) {
   	cout << " Error Starting the Results File.  Exiting." << endl;
   	finish();
   }

   
	// Serve the calculations to the DMT Viewer
	// ----------------------------------------
   Time t0 = Now() ;
   Interval dT = fBicoConf[0].fStride ;
   Time t_12 = t0 - Interval(43200) ;
   float fPadValue = -0.02 ;

   for (int ik = 0; ik < fNumMeas; ++ik) {
      #ifdef BM_DEBUG
         cout << "Setting trend time series size" << endl;
      #endif
   	fBicoTrendTS[ik].setMaxLen(12.0*3600.0);
      
      #ifdef BM_DEBUG
		   cout << "Setting trend time series label" << endl;
   	   cout << "fMeas["<<ik<<"]="<<fMeas[ik].fLabel<<", "<<fMeas[ik].fFreq1<<", "<<fMeas[ik].fFreq2<<", "<<fMeas[ik].fRad<<endl;
      #endif
	   servedChannel = fMeas[ik].fLabel.c_str() ;

      #ifdef BM_DEBUG
         cout << "Served channel label ["<<ik<<"] = " << servedChannel << endl;
      #endif
	   serveData (servedChannel, &fBicoTrendTS[ik]);  
      
      // Pad the Trend Time Series
	   isTrendPadded[ik] = false ; 
      fBicoTrendTS[ik].fixedAppend(t_12, dT, &fPadValue);
      fBicoTrendTS[ik].fixedAppend(t0-dT, dT, &fPadValue);
      isTrendPadded[ik] = true ;
   }

   #ifdef BM_DEBUG
      cout << "exiting constructor" << endl;
   #endif
} 



// *************************************************************************
//
//   PROCESS BICOMON: Get data, calculate bicoherence, & serve results
//
// *************************************************************************
//
// BicoMon::~BicoMon() 
//  {
//    delete [] fDecFilter;
//    delete [] fBicoData;
//    delete [] fBicoConf;
//    delete [] fMeasLimit;
//    delete [] fMeas;
//    delete [] fBicoTrendTS;
//    delete [] isTrendPadded;
// }
 


// *************************************************************************
//
//   PROCESS BICOMON: Get data, calculate bicoherence, & serve results
//
// *************************************************************************
//
void BicoMon::ProcessData() 
{ 
   #ifdef BM_DEBUG
	   cout << "Process Data ... "<< endl;
   #endif
   
   TSeries fData1;
   TSeries fData2;
   TSeries fData3;
   
   // Check IFO Operational State & Set Calculation Flag
   // --------------------------------------------------
   // We may want this to be configuration specific
   bool PerformCalc = true;
   if (fTestIFOstate) {
      OperStateCondList mOSC(getDacc());
      mOSC.readConfig("LockLoss.conf");
      if (!mOSC.satisfied(fIFOstate)) {
         PerformCalc = false;
      }
   }
   
   if (PerformCalc) {
      
      // Loop through Configurations
      // ---------------------------
      for (int ik = 0; ik < fNumConf; ++ik) {

         // Set New Data
         //------------- 
         #ifdef BM_DEBUG
         cout << endl;
         cout << "Setting Data for Configuration "<< ik <<": "<<endl;
         #endif

         switch (fBicoConf[ik].fNChans) {
          case 3:
            // #ifdef BM_DEBUG
            // cout << "     Channel 3: ... decimate ";
            // #endif
            // fData3 = fDataUnique[fBicoConf[ik].fChanNum3]->decimate(fBicoConf[ik].fDecim3);    // Decimate
            // fData3 = fDecFilter[ik][2].apply((fDataUnique[fBicoConf[ik].fChanNum3]));
				#ifdef BM_DEBUG
            	cout << " ... and set."<<endl;
            #endif
//             fData3 = &(fDataUnique[fBicoConf[ik].fChanNum3]);    
//             fBicoData[ik].SetData(3, fData3);    																// Set BicoDir data
            fBicoData[ik].SetData(3, fDataUnique[fBicoConf[ik].fChanNum3],fOnline);
               
          case 2:
            // #ifdef BM_DEBUG
            // cout << "     Channel 2: ... decimate ";
            // #endif
            // fData2 = fDataUnique[fBicoConf[ik].fChanNum2]->decimate(fBicoConf[ik].fDecim2);    // Decimate
            // fData2 = fDecFilter[ik][1].apply((fDataUnique[fBicoConf[ik].fChanNum2]));
            #ifdef BM_DEBUG
            cout << " ... and set."<<endl;
            #endif
//             fData2 = &(fDataUnique[fBicoConf[ik].fChanNum2]);    
//             fBicoData[ik].SetData(2, fData2);    																// Set BicoDir data
            fBicoData[ik].SetData(2, fDataUnique[fBicoConf[ik].fChanNum2],fOnline);

          case 1:
            // #ifdef BM_DEBUG
            // cout << "     Channel 1: ... decimate ";
            // #endif
            // fData1 = fDataUnique[fBicoConf[ik].fChanNum1]->decimate(fBicoConf[ik].fDecim1);    // Decimate
            // TSeries fTStest = &fDataUnique[fBicoConf[ik].fChanNum1];
            // fData1 = fDecFilter[ik][0].apply(fTStest);
            // delete fTStest ;
            #ifdef BM_DEBUG
               cout << " ... and set."<<endl;
            #endif
//             fData1 = &(fDataUnique[fBicoConf[ik].fChanNum1]);    
//             fBicoData[ik].SetData(1, fData1);    																// Set BicoDir data
            fBicoData[ik].SetData(1, fDataUnique[fBicoConf[ik].fChanNum1],fOnline);
            break;
            
          default:
            cout << "Incorrect Channel Number in SetData "<<endl;
            finish();
         }
         
         
         // Calculate Bicoherence
         // ---------------------   	
         #ifdef BM_DEBUG
         cout << "Calculating Bicoherence ..."<<endl;
         #endif
         fBicoData[ik].Calculate();
         
         
         // Update the Trend Time Series (monitored by DMT Viewer) & Results File
         // ---------------------------------------------------------------------
         Time t0 = fData1.getStartTime() ;
         Interval dT = fBicoConf[ik].fStride ;
         if (ik == 0) fResultsFile << t0.getS() << "       " ;
         #ifdef BM_DEBUG
         cout << "Time = " << fData1.getStartTime().getS() << "     ";
         #endif
         for (int im = fMeasLimit[ik]; im < fMeasLimit[ik+1]; ++im) {
            //          if (!isTrendPadded[im]) {
            //             if (dbg) {cout << "Padding Trend TS " << im << endl;}
            //             Time t_12 = t0 - Interval(43200) ;
            //             float fPadValue = -0.02 ;
            //             fBicoTrendTS[im].fixedAppend(t_12, dT, &fPadValue);
            //             fBicoTrendTS[im].fixedAppend(t0-dT, dT, &fPadValue);
            //             isTrendPadded[im] = true ;
            //          }
            #ifdef BM_DEBUG
            cout << "Calculating Measurement " << im <<endl;
            #endif
            fBicoSum = fBicoData[ik].Sum(fMeas[im]);  // Sum over the unique area
            #ifdef BM_DEBUG
            cout << "Incrementing Trend TS " << im <<endl;
            #endif
            fBicoTrendTS[im].fixedAppend(t0, dT, &fBicoSum);
            fResultsFile << fBicoSum << "        ";
            #ifdef BM_DEBUG
            cout << fBicoSum << "     ";
            #endif
         }     
         
      }		// End of loop over configurations
      fResultsFile << endl ;
      #ifdef BM_DEBUG
      cout << endl ;
      #endif
      
   }  // End of test on PerformCalc
   
}



// **********************************************************
//
//   OPEN CONFIGURATION FILE: Resolve Filename and Open. 
//
// **********************************************************
// 
bool BicoMon::OpenConfigFile()
{
   // Test file name
   // --------------
   if (fConfigFileName.empty()) {
   	cout << "Error: No Configurations File Specified" << endl;
      cout << "usage:  BicoMon ConfigFileName ResultsFileName" << endl;
      finish();
   }

   // Open the file,  Add Directory if needed  	
   // ---------------------------------------
   if (!strstr("/",fConfigFileName.c_str())) {
   	std::string fConfigDir;
   	if (getenv("DMTPARS")) {
	      fConfigDir = getenv("DMTPARS") ;
   	} else {
   		fConfigDir = getenv("HOME");
   	}	     
   	if (!fConfigDir.empty()) {
	      fConfigFileName = fConfigDir + "/" + fConfigFileName ;
   	}
   }
   #ifdef BM_DEBUG
      cout << "Config File name = " << fConfigFileName << endl;
   #endif

   fConfigFile.open(fConfigFileName.c_str(), std::ios::in);
   if (!fConfigFile.is_open()) {
      cerr << "ERROR: Unable to open configuration file: " 
	   << fConfigFileName << endl;
      return false;
   }
   
   return true;
}   



// *************************************************************************
//
//   READ CONFIG FILE: Check if config file has been updated
//
// *************************************************************************
//
bool BicoMon::ReadConfigFile()
{   
   // Read in number of configurations/measurements
   // ---------------------------------------------
   fConfigFile >> fNumConf ;  // Read in the number of Configurations
   fConfigFile >> fNumMeas ;  // Read in the number of Measurements
   if (fNumConf > fMaxConf) {
      cout << "ERROR:  Configurations exceed maximum value of "<< fMaxConf << endl;
      return false ;
   }
   fBicoData = new BicoDir[fNumConf];			// create a bicoherence class for each configuration
   fBicoConf = new BicoDirConf[fNumConf];		// create a configuration record for each bicoherence class
   fMeasLimit = new int[fNumConf+1];			// Array of limits associating measurements with configurations
   fMeas = new BicoMeas[fNumMeas];				// Array of bicoherence measurements
   fBicoTrendTS = new FixedLenTS[fNumMeas];	// Trend time series for each measurement
   isTrendPadded = new bool[fNumMeas];			// Flag indicating if Trend Time series is padded with zeroes
   
   int iMeas = -1;               				// index of measurement
   int iConf = -1;               				// index of configuration
   std::string fConfMeasLabel;   				// Label distinguishing Configurations and Measurements
   std::string fConfLabel = std::string("C");
   std::string fMeasLabel = std::string("M");
   fSpanMax = 1;          							// Maximum Span/Stride required
   float  fSpanMin = 1;          				// Minimum Span/Stride required

   
   // Loop through Configuration & Measurement Parameters
   // ---------------------------------------------------
   while (!fConfigFile.eof() && iMeas+1 < fNumMeas) {

      fConfigFile >> fConfMeasLabel;

      // Check for Proper Configuration/Measurement Label       
      // ------------------------------------------------
      if (fConfMeasLabel != fConfLabel && fConfMeasLabel != fMeasLabel) {
         cout << "ERROR: Config File Label not recognized " << iMeas << iConf << endl;
         return false ;
      
               
      // Read in a Measurement      
      // ---------------------
      } else if (fConfMeasLabel == fMeasLabel) {
         if (++iMeas >= fNumMeas) {
            cout << "ERROR: Number of Measurements > Limit " << iMeas << endl;
            return false;
         }

         fConfigFile >> fMeas[iMeas].fLabel;
         fConfigFile >> fMeas[iMeas].fFreq1;
         fConfigFile >> fMeas[iMeas].fFreq2;
         fConfigFile >> fMeas[iMeas].fRad;
         fMeas[iMeas].fNConf = iConf;
         
         #ifdef BM_DEBUG
            cout << "fMeas["<<iMeas<<"]="<<fMeas[iMeas].fLabel<<", "<<fMeas[iMeas].fFreq1<<", "<<fMeas[iMeas].fFreq2<<", "<<fMeas[iMeas].fRad<<endl;
         #endif

      // Read in a Configuration
      // -----------------------
      } else if (fConfMeasLabel == fConfLabel) {
         if (++iConf >= fNumConf) {
            cout << "ERROR: Number of Configurations > Limit " << iConf << endl;
            return false;
         }

         fMeasLimit[iConf] = iMeas + 1 ;		 // Set which range of measurements associated with this configuration
                                        		          
         fBicoConf[iConf].fValid = false ;    // Invalidate until configuration read correctly 
         fBicoConf[iConf].fNew = true ;       // Set flag for new configuration
         
         
         fConfigFile >> fBicoConf[iConf].fNChans ;        // Number of Channels
         fBicoConf[iConf].fRate = 16384 ;
         float fFreqTemp = 0 ;
         switch (fBicoConf[iConf].fNChans) {
          case 3:
            fConfigFile >> fBicoConf[iConf].fChanName3 ;  // Chan Name 3
            fConfigFile >> fBicoConf[iConf].fRate3;       // Chan Rate 3
            fBicoConf[iConf].fRate = (fBicoConf[iConf].fRate < fBicoConf[iConf].fRate3) ? fBicoConf[iConf].fRate : fBicoConf[iConf].fRate3 ;
          case 2:
            fConfigFile >> fBicoConf[iConf].fChanName2 ;  // Chan Name 2
            fConfigFile >> fBicoConf[iConf].fRate2;       // Chan Rate 2
            fBicoConf[iConf].fRate = (fBicoConf[iConf].fRate < fBicoConf[iConf].fRate2) ? fBicoConf[iConf].fRate : fBicoConf[iConf].fRate2 ;
          case 1:
            fConfigFile >> fBicoConf[iConf].fChanName1 ;  // Chan Name 1
            fConfigFile >> fBicoConf[iConf].fRate1;       // Chan Rate 1
            fBicoConf[iConf].fRate = (fBicoConf[iConf].fRate < fBicoConf[iConf].fRate1) ? fBicoConf[iConf].fRate : fBicoConf[iConf].fRate1 ;
            break;
          default:
            cout << "Error reading configuration file "<<endl;
            finish();
         }
         fConfigFile >> fFreqTemp ;             			// Max Frequency (temp)
         fConfigFile >> fBicoConf[iConf].fFreqRes ;    	// Frequency Resolution
         fConfigFile >> fBicoConf[iConf].fOverlap ;    	// Overlap of adjacent segments
         fConfigFile >> fBicoConf[iConf].fSpan ;			// Total Data Span 

         // Overlap must be within [0, 0.5]
         fBicoConf[iConf].fOverlap = (fBicoConf[iConf].fOverlap > 0.5 || fBicoConf[iConf].fOverlap < 0) ? 0 : fBicoConf[iConf].fOverlap ;

         // Span must be greater than SpanMin and SpanMax must be reset if < Span
         // fSpanMax used to set data stride
         fBicoConf[iConf].fSpan = (fBicoConf[iConf].fSpan > fSpanMin) ? fBicoConf[iConf].fSpan : fSpanMin ;
         fSpanMax = (fBicoConf[iConf].fSpan > fSpanMax) ? fBicoConf[iConf].fSpan : fSpanMax ;
         
         float fFreqNyq = fBicoConf[iConf].fRate / 2 ;   // Nyquist Frequency
         fBicoConf[iConf].fFreqMax = (fFreqTemp > fFreqNyq) ? fFreqNyq : fFreqTemp ; //Max Freq
         fBicoConf[iConf].fDecim = int(fFreqNyq / fBicoConf[iConf].fFreqMax) ;
         
         
         // Set the Basic Configuration parameters 
         fBicoData[iConf].SetConfig(fBicoConf[iConf]);  
         
         // Read back All the Configuration Parameters
         fBicoConf[iConf] = fBicoData[iConf].GetConfig();  
         
         
         
      }  // End of Test on Measurement/Configuration Label

   }  // End of while loop to read in file      

   fMeasLimit[iConf+1] = iMeas + 1 ;      // Set the upper limit on measurements
   
	return true;
}





// ****************************************************
//
//   PRINT CONFIGURE PARAMETERS: For debugging purposes
//
// ****************************************************
//
void BicoMon::PrintConfig()
{
   // Loop over Configurations and Print all Parameters
   // -------------------------------------------------
   for (int iConf=0; iConf<fNumConf; ++iConf) {   

      switch (fBicoConf[iConf].fNChans) {
       case 3:
         cout << "fBicoConf["<<iConf<<"].fChanName3 = " << fBicoConf[iConf].fChanName3 << endl;
         cout << "fBicoConf["<<iConf<<"].fRate3     = " << fBicoConf[iConf].fRate3 << endl;
         cout << "fBicoConf["<<iConf<<"].fDecim3    = " << fBicoConf[iConf].fDecim3 << endl;
         cout << "fBicoConf["<<iConf<<"].fChanNum3 = " << fBicoConf[iConf].fChanNum3 << endl;
       case 2:
         cout << "fBicoConf["<<iConf<<"].fChanName2 = " << fBicoConf[iConf].fChanName2 << endl;
         cout << "fBicoConf["<<iConf<<"].fRate2     = " << fBicoConf[iConf].fRate2 << endl;
         cout << "fBicoConf["<<iConf<<"].fDecim2    = " << fBicoConf[iConf].fDecim2 << endl;
         cout << "fBicoConf["<<iConf<<"].fChanNum2 = " << fBicoConf[iConf].fChanNum2 << endl;
       case 1:
         cout << "fBicoConf["<<iConf<<"].fChanName1 = " << fBicoConf[iConf].fChanName1 << endl;
         cout << "fBicoConf["<<iConf<<"].fRate1     = " << fBicoConf[iConf].fRate1 << endl;
         cout << "fBicoConf["<<iConf<<"].fDecim1    = " << fBicoConf[iConf].fDecim1 << endl;
         cout << "fBicoConf["<<iConf<<"].fChanNum1 = " << fBicoConf[iConf].fChanNum1 << endl;
         break;
       default:
         cout << "Error reading configuration file "<<endl;
         finish();
      }
      cout << "fBicoConf["<<iConf<<"].fNChans    = " << fBicoConf[iConf].fNChans << endl;
      cout << "fBicoConf["<<iConf<<"].fRate      = " << fBicoConf[iConf].fRate << endl;
      cout << "fBicoConf["<<iConf<<"].fDecim     = " << fBicoConf[iConf].fDecim << endl;
      cout << "fBicoConf["<<iConf<<"].fFreqRes   = " << fBicoConf[iConf].fFreqRes << endl;
      cout << "fBicoConf["<<iConf<<"].fFreqMax   = " << fBicoConf[iConf].fFreqMax << endl;
      cout << "fBicoConf["<<iConf<<"].fNBins     = " << fBicoConf[iConf].fNBins << endl;
      cout << "fBicoConf["<<iConf<<"].fNPoints   = " << fBicoConf[iConf].fNPoints << endl;
      cout << "fBicoConf["<<iConf<<"].fNFFT      = " << fBicoConf[iConf].fNFFT << endl;
      cout << "fBicoConf["<<iConf<<"].fNWinRad   = " << fBicoConf[iConf].fNWinRad << endl;
      cout << "fBicoConf["<<iConf<<"].fOverlap   = " << fBicoConf[iConf].fOverlap << endl;
      cout << "fBicoConf["<<iConf<<"].fWindow1D  = " << fBicoConf[iConf].fWindow1D << endl;
      cout << "fBicoConf["<<iConf<<"].fAverage   = " << fBicoConf[iConf].fAverage << endl;
      cout << "fBicoConf["<<iConf<<"].fNTotal    = " << fBicoConf[iConf].fNTotal << endl; 
      cout << "fBicoConf["<<iConf<<"].fSpan      = " << fBicoConf[iConf].fSpan << endl;
      cout << "fBicoConf["<<iConf<<"].fStride    = " << fBicoConf[iConf].fStride << endl;
      cout << "fBicoConf["<<iConf<<"].fNSlices   = " << fBicoConf[iConf].fNSlices << endl;
      cout << "fBicoConf["<<iConf<<"].fWindow2D  = " << fBicoConf[iConf].fWindow2D << endl;

   }  // End of loop over Configurations
}




// *************************************************************************
//
//   FIND UNIQUE CHANNELS: List the Unique Channels for All Configurations 
//
// *************************************************************************
//
bool BicoMon::FindUniqueChans()
{   
   fNumChanUnique = 0;								// Number of unique channels
   
   
   // Find unique channels. Store DAQ info for each.
   // ----------------------------------------------
   for (int ik = 0; ik < fNumConf; ++ik) {
      switch (fBicoConf[ik].fNChans) {
       case 3:  
         fBicoConf[ik].fChanNum3 = BicoMon::ChanMatch(fBicoConf[ik].fChanName3);
       case 2:  
         fBicoConf[ik].fChanNum2 = BicoMon::ChanMatch(fBicoConf[ik].fChanName2);
       case 1:  
         fBicoConf[ik].fChanNum1 = BicoMon::ChanMatch(fBicoConf[ik].fChanName1);
      }
   }

   #ifdef BM_DEBUG
      for (int ik=0; ik<fNumChanUnique; ++ik) {
         cout << "fChanList["<<ik<<"].fChanName = "<<fChanList[ik].fChanName<<endl;
         cout << "fChanList["<<ik<<"].fDecim = "<<fChanList[ik].fDecim<<endl;
      }
   #endif
 
   return true;
}


         
         

// ****************************************************************************
//
//   MATCH CHANNELS: Test for Matching Channel Names and number unique channels 
//
// ****************************************************************************
//
Int_t BicoMon::ChanMatch(std::string fChanName)
{  
   if (fNumChanUnique > 0) {
      for (int in=0; in < fNumChanUnique; ++in) {
         if (fChanName == fChanList[in].fChanName) {
            return in ;
         }
      }
   }
   
   // No Match so add Channel Name to List
	fChanList[fNumChanUnique].fChanName = fChanName ;
   fChanList[fNumChanUnique].fDecim = 1 ;
   return fNumChanUnique++ ;
}



// *************************************************************************
//
//   ASSIGN DATA CHANNELS: Add Unique Data Channels and Map to Time Series
//
// *************************************************************************
//
bool BicoMon::AssignDataChannels()
{   
   // Read in number of configurations/measurements
   // ---------------------------------------------
   // 
   // HOW DO I MAKE A DYNAMICALLY ASSIGNED ARRAY OF POINTERS?  THE LINE BELOW FAILS.
   // 
	//    fDataUnique = new (TSeries*)[fNumChanUnique];

   
   // Add Channels to Dacc with Decimation
   // ---------------------------------------------
	for (int ik = 0; ik < fNumChanUnique; ++ik) {
      getDacc().addChannel(fChanList[ik].fChanName.c_str(), fChanList[ik].fDecim, &(fDataUnique[ik]));
      #ifdef BM_DEBUG
         cout<<"Chan = : "<<fChanList[ik].fChanName<<", Dec = "<<fChanList[ik].fDecim<<endl;
      #endif
	}

	// Assign the unique data channels to the non-unique data channels
	// ---------------------------------------------------------------
   //	for (int ik = 0; ik < fNumConf; ++ik) {
   //	   switch (fBicoConf[ik].fNChans) {
   //	    case 3:  
   //	      fData3[ik] = fDataUnique[fBicoConf[ik].fChanNum3];
   //	    case 2:
   //	      fData2[ik] = fDataUnique[fBicoConf[ik].fChanNum2];
   //	    case 1:
   //	      fData1[ik] = fDataUnique[fBicoConf[ik].fChanNum1];
   //	   }
   //	}
   
   // Set the Stride and Align
   // ------------------------
   getDacc().setStride(fSpanMax);   
	Time tEnd   = getDacc().getCurrentTime();
	int tSec    = tEnd.getS();
	int iStride = int(getDacc().getStride());
	if ((tSec % iStride) != 0 || tEnd.getN()) {
	    tSec += iStride - (tSec % iStride);
	    #ifdef BM_DEBUG
          cout << "BicoMon: Skipping to " << tSec << " GPS." << endl;
       #endif
	    getDacc().seek(Time(tSec));
	}
	return true;
}
      

// *************************************************************************
//
//    START RESULTS FILE:  Open and write header information
//
// *************************************************************************
//
bool BicoMon::StartResultsFile()
{


   // Open the file,  Add Directory if needed  	
   // ---------------------------------------
   if (!strstr("/",fResultsFileName.c_str())) {
   	std::string fResultsDir ;
   	if (getenv("DMTHTMLOUT")) {
	      fResultsDir = getenv("DMTHTMLOUT") ;
   	} else if (getenv("DMTPARS")) {
	      fResultsDir = getenv("DMTPARS") ;
   	} else {
	      fResultsDir = getenv("HOME") ;
   	}
      fResultsFileName = fResultsDir + "/" + fResultsFileName ;
   }
   #ifdef BM_DEBUG
      cout << "Results File name = " << fResultsFileName << endl;
   #endif
   
   fResultsFile.open(fResultsFileName.c_str(), std::ios::out);
   if (!fResultsFile.is_open()) {
      cerr << "ERROR: Unable to open Results file: "
	   << fResultsFileName << endl;
      return false;
   }
   
   
   // Write the header information
   fResultsFile << "BICOMON RESULTS FILE" << endl;
   fResultsFile << "--------------------" << endl;
   fResultsFile << endl;
   fResultsFile << endl;
   
   for (int ik = 0; ik < fNumConf; ++ik) {  
      fResultsFile << " CONFIGURATION " << ik << endl;
      fResultsFile << "------------------" << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fNChans    = " << fBicoConf[ik].fNChans << endl;
   	switch (fBicoConf[ik].fNChans) {
   	 case 3:
   		fResultsFile << "fBicoConf["<<ik<<"].fChanName3 = " << fBicoConf[ik].fChanName3 << endl;
   		fResultsFile << "fBicoConf["<<ik<<"].fRate3     = " << fBicoConf[ik].fRate3 << endl;
   		fResultsFile << "fBicoConf["<<ik<<"].fDecim3    = " << fBicoConf[ik].fDecim3 << endl;
   	 case 2:
   		fResultsFile << "fBicoConf["<<ik<<"].fChanName2 = " << fBicoConf[ik].fChanName2 << endl;
   		fResultsFile << "fBicoConf["<<ik<<"].fRate2     = " << fBicoConf[ik].fRate2 << endl;
   		fResultsFile << "fBicoConf["<<ik<<"].fDecim2    = " << fBicoConf[ik].fDecim2 << endl;
   	 case 1:
   		fResultsFile << "fBicoConf["<<ik<<"].fChanName1 = " << fBicoConf[ik].fChanName1 << endl;
   		fResultsFile << "fBicoConf["<<ik<<"].fRate1     = " << fBicoConf[ik].fRate1 << endl;
   		fResultsFile << "fBicoConf["<<ik<<"].fDecim1    = " << fBicoConf[ik].fDecim1 << endl;
   		break;
   	 default:
   	   cout<<"ERROR: Channel Number not 1-3"<<endl;
   	   return false;
   	}
   	fResultsFile << "fBicoConf["<<ik<<"].fRate      = " << fBicoConf[ik].fRate << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fDecim     = " << fBicoConf[ik].fDecim << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fFreqRes   = " << fBicoConf[ik].fFreqRes << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fFreqMax   = " << fBicoConf[ik].fFreqMax << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fNBins     = " << fBicoConf[ik].fNBins << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fNPoints   = " << fBicoConf[ik].fNPoints << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fNFFT      = " << fBicoConf[ik].fNFFT << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fNWinRad   = " << fBicoConf[ik].fNWinRad << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fOverlap   = " << fBicoConf[ik].fOverlap << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fWindow1D  = " << fBicoConf[ik].fWindow1D << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fAverage   = " << fBicoConf[ik].fAverage << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fNTotal    = " << fBicoConf[ik].fNTotal << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fSpan      = " << fBicoConf[ik].fSpan << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fStride    = " << fBicoConf[ik].fStride << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fNSlices   = " << fBicoConf[ik].fNSlices << endl;
   	fResultsFile << "fBicoConf["<<ik<<"].fWindow2D  = " << fBicoConf[ik].fWindow2D << endl;
      fResultsFile << endl;
      fResultsFile << endl;
   }
   
   fResultsFile << "RESULTS" << endl;
   fResultsFile << "-------" << endl;
   fResultsFile << " GPS Sec";
   for (int ik = 0; ik < fNumMeas; ++ik) {
   	fResultsFile << "     " << fMeas[ik].fLabel;
   }
   fResultsFile << endl;
   fResultsFile << " -------";
   for (int ik = 0; ik < fNumMeas; ++ik) {
   	fResultsFile << "     ---------------";
   }
   fResultsFile << endl;

	return true;
}

// ***************************************************************
// *                                                             *
// *   ---  BicoChanList: Bicoherence Unique Channel List  ---   *
// *                                                             *
// ***************************************************************

// Constructor
// -----------
BicoChanList::BicoChanList () 
  : fChanName (""), fDecim (0)
{
//   fDecim = 0;
 
//   fChanRate = 0.0;
//   fFreqMax = 0.0;
}




// Destructor
// ----------
BicoChanList::~BicoChanList ()   
{
}



