#ifndef GaussNoise_HH
#define GaussNoise_HH
#include "Chirp.hh"
#include "Interval.hh"

/**  Gaussian noise template generator.
  *  GaussNoise is a template generator based on the Chirp class. It 
  *  generates gaussian noise of unit amplitude.
  */
class GaussNoise : public Chirp {
public:

  /**  GaussNoise Constructor. 
    *  Generate a gaussian noise template. The template extends from T0 
    *  for a time dT. and has a flat frequency spectrum from 0-fNy.
    */
  explicit GaussNoise(double fNy, const Interval& dT = Interval(60.0), 
		      const Time& t0=Time(0));

  /**  GaussNoise destructor.
    */
  ~GaussNoise(void) {}

  /**  Complex Frequency domain representation of the template at frequency
    *  Freq.
    *  @memo Frequency domain template bin.
    */
  fComplex Fspace(double Freq, double dF=0) const;

  /**  Tspace returns the waveform value at time 't0'.
    *  @memo Time domain template bin.
    */ 
  double Tspace(const Time& t0) const;

  /**  Return the template start time. 
    *  @memo Get the t0.
    */
  Time getT0(void)   const;

  /**  Return the template end time. The end-time is defined to be the 
    *  time at which the template frequency reaches a maximum.
    *  @memo Get the End time.
    */
  Time getTEnd(void) const;

  /**  Return the template critical time. The critical time is defined 
    *  explicitly or implicitly when the object is constructed.
    *  @memo Get the Critical time.
    */
  Time getTc(void) const;

  /**  TvsF(f) returns the absolute (GPS) time at which the in-spiral 
    *  frequency is exactly 'f'.
    *  @memo Time versus Frequency.
    */
  Time   TvsF(double f) const;

  /**  freq(t) returns the in-spiral frequency at Absolute time t.
    *  @memo Waveform Frequency.
    */
  double freq(const Time& t) const;

  /**  phi(t) returns the phase angle at absolute time t.
    *  @memo Waveform Phase angle.
    */
  double  phi(const Time& t) const;

  /**  Ampl(t) returns the wave-form amplitude.
    *  @memo Amplitude.
    */
  double Ampl(const Time& t) const;

private:
  /**  Absolute time of the start of template generation.
    *  @memo Template start time.
    */
  Time   mT0;

  /**  The template limit time is defined to be the absolute (GPS) time at 
    *  which the template ends.
    *  @memo Template limit time.
    */
  Time   mTl;

  /**  The critical time is the absolute (GPS) time at which the Phi_c is 
    *  defined.
    *  @memo Critical time.
    */
  Time   mTc;

  /**  This is the Nyquist frequency.
    *  @memo Nyquist frequency.
    */
  double mFny;
};

#endif  //  GaussNoise_HH
