#include "constant.hh"
#include "rndm.hh"
#include <cmath>
#include <fstream>

/* A C-program for MT19937: Real number version([0,1)-interval) (1998/4/6) */
/*   genrand() generates one pseudorandom real number (double) */
/* which is uniformly distributed on [0,1)-interval, for each  */
/* call. sgenrand(seed) set initial values to the working area */
/* of 624 words. Before genrand(), sgenrand(seed) must be      */
/* called once. (seed is any 32-bit integer except for 0).     */
/* Integer generator is obtained by modifying two lines.       */
/*   Coded by Takuji Nishimura, considering the suggestions by */
/* Topher Cooper and Marc Rieffel in July-Aug. 1997.           */

/* This library is free software; you can redistribute it and/or   */
/* modify it under the terms of the GNU Library General Public     */
/* License as published by the Free Software Foundation; either    */
/* version 2 of the License, or (at your option) any later         */
/* version.                                                        */
/* This library is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of  */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.            */
/* See the GNU Library General Public License for more details.    */
/* You should have received a copy of the GNU Library General      */
/* Public License along with this library; if not, write to the    */
/* Free Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA   */ 
/* 02111-1307  USA                                                 */

/* Copyright (C) 1997 Makoto Matsumoto and Takuji Nishimura.       */
/* When you use this, send an email to: matumoto@math.keio.ac.jp   */
/* with an appropriate reference to your work.                     */

/* REFERENCE                                                       */
/* M. Matsumoto and T. Nishimura,                                  */
/* "Mersenne Twister: A 623-Dimensionally Equidistributed Uniform  */
/* Pseudo-Random Number Generator",                                */
/* ACM Transactions on Modeling and Computer Simulation,           */
/* Vol. 8, No. 1, January 1998, pp 3--30.                          */

/* Period parameters */  
#define N 624
#define M 397
#define MATRIX_A 0x9908b0df   /* constant vector a */
#define UPPER_MASK 0x80000000 /* most significant w-r bits */
#define LOWER_MASK 0x7fffffff /* least significant r bits */

/* Tempering parameters */   
#define TEMPERING_MASK_B 0x9d2c5680
#define TEMPERING_MASK_C 0xefc60000
#define TEMPERING_SHIFT_U(y)  (y >> 11)
#define TEMPERING_SHIFT_S(y)  (y << 7)
#define TEMPERING_SHIFT_T(y)  (y << 15)
#define TEMPERING_SHIFT_L(y)  (y >> 18)

typedef unsigned long ulong_t;
static ulong_t mt[N]; /* the array for the state vector  */
static int mti=N+1; /* mti==N+1 means mt[N] is not initialized */

/* initializing the array with a NONZERO seed */
void
RndmSeed(ulong_t seed)
{
    /* setting initial seeds to mt[N] using         */
    /* the generator Line 25 of Table 1 in          */
    /* [KNUTH 1981, The Art of Computer Programming */
    /*    Vol. 2 (2nd Ed.), pp102]                  */
    mt[0]= seed & 0xffffffff;
    for (mti=1; mti<N; mti++)
        mt[mti] = (69069 * mt[mti-1]) & 0xffffffff;
}

double
Rndm(void)
{
    ulong_t y;
    static ulong_t mag01[2]={0x0, MATRIX_A};
    /* mag01[x] = x * MATRIX_A  for x=0,1 */

    if (mti >= N) { /* generate N words at one time */

        if (mti == N+1)     /* if RndmSeed() has not been called, */
            RndmSeed(4357); /* a default initial seed is used   */

        for (int kk=0 ; kk<N-M ; kk++) {
            y = (mt[kk] & UPPER_MASK) | (mt[kk+1] & LOWER_MASK);
            mt[kk] = mt[kk+M] ^ (y >> 1) ^ mag01[y & 0x1];
        }

        for (int kk=N-M ; kk<N-1 ; kk++) {
            y = (mt[kk] & UPPER_MASK) | (mt[kk+1] & LOWER_MASK);
            mt[kk] = mt[kk+(M-N)] ^ (y >> 1) ^ mag01[y & 0x1];
        }

        y = (mt[N-1] & UPPER_MASK) | (mt[0] & LOWER_MASK);
        mt[N-1] = mt[M-1] ^ (y >> 1) ^ mag01[y & 0x1];

        mti = 0;
    }
  
    y = mt[mti++];
    y ^= TEMPERING_SHIFT_U(y);
    y ^= TEMPERING_SHIFT_S(y) & TEMPERING_MASK_B;
    y ^= TEMPERING_SHIFT_T(y) & TEMPERING_MASK_C;
    y ^= TEMPERING_SHIFT_L(y);

    return ( (double)y * 2.3283064365386963e-10 );
}

void 
Rannor(double &a, double &b)
{
   // Return 2 numbers distributed following a gaussian with mean=0 and 
   // sigma=1. code stolen from root.

   // Local variables
   double r, x, y, z;

   y = Rndm();
   if (!y) y = Rndm();
   z = Rndm();
   x = z * twopi;
   r = sqrt(-2.0*log(y));
   a = r * sin(x);
   b = r * cos(x);
}

static bool haveRN = false;
static double GRN(0.0);

double 
Rannor(void)
{
    double r;
    if (haveRN) {
        r=GRN;
	haveRN = false;
    } else {
	Rannor(r, GRN);
	haveRN = true;
    }
    return r;
}

long 
PoissonRndm(double x)
{
    if (x <= 0.0) return 0;
    if (x > 32.0) return long(sqrt(x)*Rannor() + x);
    double P=Rndm();
    double term = exp(-x);
    P -= term;
    if (P <= 0) return 0;
    for (long i=1; i<100 ; i++) {
	term *= x/double(i);
        P -= term;
	if (P <= 0.0) return i;
    }
    return 100;
}

//======================================  Save the random number state
void 
SaveRndmState(const std::string& file) {
  std::ofstream out(file.c_str());
  out << N << " " << mti << std::endl;
  if (mti != N+1) {
    for (int i=0; i<N; i++) {
      out << mt[i] << std::endl;
    }
  }
  out.close();
}

//======================================  Restore the randon number state.
void 
RestoreRndmState(const std::string& file) {
  std::ifstream inp(file.c_str());
  if (!inp.is_open()) return;
  int en;
  inp >> en >> mti;
  if (!inp) return;
  if (en == N && mti != N+1) {
    for (int i=0; i<N; i++) {
      inp >> mt[i];
    }
  }
  inp.close();
}
