/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef GEN_VECT_HH
#define GEN_VECT_HH
#include <cstddef>

/**  The vec_math class performs a bunch of vector operations optimized 
 *  depending on the capabilities of the target machine.
 */
class gen_vect {
public:
   enum cpu_sse {
      nosse,
      sse2,
      ssse3,
      sse4_1,
      sse4_2,
      avx
   };

   typedef double math_type;

public:
   gen_vect(void);

   virtual ~gen_vect(void) {}

   /**  Force a specific cpu implementation.
    */
   const char* sse_id_name(cpu_sse capable);

   /**  Force a specific cpu implementation.
    */
   void set_sse_id(cpu_sse capable);

   /**  Add two vectors
    */
   template <typename T> 
   void add(T out[], const T in[], std::size_t N) const;

   /**  Add two vectors producing a third
    */
   template <typename T> 
   void add(T out[], const T in1[], const T in2[], std::size_t N) const;

   /**  Add a constant to a vector
    */
   template <typename T> 
   void add(T out[], T in, std::size_t N) const;

   /**  Type conversion
    */
   template <typename R, typename T> 
   void cvt(R out[], const T in[], std::size_t N) const;

   /**  Divide two vectors
    */
   template <typename T> 
   void div(T out[], const T in[], std::size_t N) const;

   /**  Divide two vectors producing a third
    */
   template <typename T> 
   void div(T out[], const T in1[], const T in2[], std::size_t N) const;

   /**  Divide a vector by a constant
    */
   template <typename T> 
   void div(T out[], T in, std::size_t N) const;

   /**  Inner product
    */
   template <typename T> 
   math_type dot(const T in1[], const T in2[], std::size_t N) const;

   /**  Multiply two vectors
    */
   template <typename T> 
   void mul(T out[], const T in[], std::size_t N) const;

   /** Multiply two vectors producing a third
    */
   template <typename T> 
   void mul(T out[], const T in1[], const T in2[], std::size_t N) const;

   /**  Multiply a vector by a constant
    */
   template <typename T> 
   void mul(T out[], T in, std::size_t N) const;

   /**  Multiply a vector of arbitrary type by a double vector
    */
   template <typename T> 
   void muld(T out[], const double in[], std::size_t n) const;

   /** Multiply two vectors producing and add to a third vector
    */
   template <typename T> 
   void muladd(T out[], const T in1[], const T in2[], std::size_t N) const;


   /** Multiply two vectors producing and add to a third vector
    */
   template <typename T> 
   void muladd(T out[], const T in1[], T in2, std::size_t N) const;

   /**  Subtract two vectors
    */
   template <typename T> 
   void scale(T out[], math_type in, std::size_t N) const;

   /**  Subtract two vectors
    */
   template <typename T> 
   void sub(T out[], const T in[], std::size_t N) const;

   /**  Subtract two vectors producing a third
    */
   template <typename T> 
   void sub(T out[], const T in1[], const T in2[], std::size_t N) const;

   /**  Subtract a constant from a vector
    */
   template <typename T> 
   void sub(T out[], T in, std::size_t N) const;

   /**  Sum a vector
    */
   template <typename T> 
   T sum(const T in1[], std::size_t N) const;

   /**  Modulo square of a complex vector.
    */
   template <typename T> 
   void cmodsq(double out[], const T in[], std::size_t n) const;

private:
   cpu_sse _sseID;
};

#include "gen_vect.icc"

#endif // !defined(VEC_MATH_HH)
