/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef _LIGO_WEBCACHE_H
#define _LIGO_WEBCACHE_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: webcache						*/
/*                                                         		*/
/* Module Description: a cache for web servers				*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 4Jan02   D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: webcache.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "gmutex.hh"
#include "thread_base.hh"
#include "Time.hh"
#include <sys/types.h>
#include <string>
#include <map>
#include <vector>
#include <utility>

namespace web {


/** Web cache class   
    @memo Web cache
    @author Written January 2005 by Daniel Sigg
    @version 1.0
    @ingroup IO_web
 ************************************************************************/
    class webcache : private thread::thread_base {
    public:
       /// Cache page
      class cachepage {
      public:
         /// Header entry (name/token)
         typedef std::pair <std::string, std::string> header_entry;
         /// header type (name/token list)
         typedef std::vector <header_entry> header_type;
      
         /// Default constructor
         cachepage();
         /** Constructor.
             @memo Constructor
             @param exp Expiration date
             @param d Pointer to web page
             @param len Length of web page
             @param header Web page header
          ***************************************************************/
         cachepage (const Time& exp, const char* d, int len,
                   const header_type& header);
         /** Constructor.
             @memo Constructor
             @param exp Expiration date
             @param s Web page
             @param header Web page header
          ***************************************************************/
         cachepage (const Time& exp, const std::string& s,
                   const header_type& header);
         /// Copy constructor
         cachepage (const cachepage& page);
         /// Destructor
         virtual ~cachepage();
         /// Copy operator
         cachepage& operator= (const cachepage& page);
         /** Load data.
             @memo Load data
             @param d Pointer to web page
             @param len Length of web page
          ***************************************************************/
         void load (const char* d, int len);
         /** Load string.
             @memo Load string
             @param s Web page
          ***************************************************************/
         void load (const std::string& s);
      	 /// Get time
         const Time& getTime() const {
            return fTime; }
      	 /// Set time
         void setTime (Time& t) {
            fTime = t; }
      	 /// Get data pointer
         char* get() {
            return fData; }
      	 /// Get data pointer
         const char* get() const {
            return fData; }
      	 /// Get length
         int size() const {
            return fLen; }
      	 /// Get header
         header_type& header() {
            return fHeader; }
      	 /// Get header
         const header_type& header() const {
            return fHeader; }
      	 /// Set header
         void setHeader (const header_type& header) {
            fHeader = header; }
      	 /// Get allow compress
         bool allowCompress() const {
            return fAllowCompress; }
      	 /// Set time
         void setAllowCompress (bool allow = true) {
            fAllowCompress = allow; }
      
      protected:
      	 /// Expiration time
         Time		fTime;
      	 /// Data
         char*		fData;
      	 /// Data length
         int		fLen;
      	 /// Header
         header_type	fHeader;
      	 /// Allow compression
         bool		fAllowCompress;
      };
   
       /// URL compare
      struct urlcompare {
      	 /// compare two URLs
         bool operator() (const std::string& s1, const std::string& s2) const;
      };
       /// Cache list
      typedef std::map<std::string, cachepage, urlcompare> cachelist; 
   
      /** Default constructor.
          @memo Default constructor
          @param maxsize Maximum size of cache in bytes
          @param autocleanup Enable auto cleanup
       ******************************************************************/
      webcache (int maxsize = 1024*1024, bool autocleanup = false);
      /// Destructor
      virtual ~webcache();
      /** Add a cache page.
          @memo Add a cache page
          @param url Web address
          @param page Web cache page
   	  @return true if successful
       ******************************************************************/
      bool add (const char* url, const cachepage& page);
      /** Lookup cache page.
          @memo Lookup cache page
          @param url Web address
          @param page Web cache page (return)
   	  @return true if successful
       ******************************************************************/
      bool lookup (const char* url, cachepage& page);
      /** Cleanup cache.
          Remove expired entries
          @memo Cleanup cache
          @param now Time now
       ******************************************************************/
      void cleanup (const Time& now);
      /** Ask for automatic cleanup.
          Remove expired entries
          @memo Ask for automatic cleanup
          @return true if successful
       ******************************************************************/
      bool enableAutoCleanup ();
      /** Set maximum size of cache.
          If the requested size is zero, caching is disabled. if the
          requested size is negative, no limit is applied.
          @memo Set maximum size of cache
          @param size Maximum size
       ******************************************************************/
      void setMax (int size);	
      /** Get maximum size of cache (bytes)
          @memo Get maximum size of cache
          @return maximum size of cache
       ******************************************************************/
      int getMax() const {
         return fMaxSize; }	
      /** Get size of cache
          @memo Get size of cache
          @return size of cache
       ******************************************************************/
      int size() const;	
      /** Reduce cache to desired size.
          Remove expired entries
          @memo Reduce cache to desired size
          @param size Desired size
          @return true if successful
       ******************************************************************/
      bool reduceCache (int size);

   private:
      /** Cleanup thread executor.
       */
      void* thread_entry(void);
   protected:
      /// Mutex
      mutable thread::recursivemutex	fMux;
      /// Maximum cache size (kbytes)
      int		fMaxSize;
      /// Cache list
      cachelist		fCache;
      /// Thread ID for cleanup task
      pthread_t		fCleanupID;
   };


}

#endif // _LIGO_WEBCACHE_H
