/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LXR_HH
#define LXR_HH

#include "Translate.hh"
#include <string>
#include <vector>
#include <iosfwd>

/**  This class takes in a stream of characters and uses them to build an
  *  output string. Each character read in is optionally stored into the
  *  output string and is used to calculate a new state. When the end of a 
  *  token is detected, the analysis method (token()) returns the token
  *  string and ID as specified by the state tables.
  *  Lexographic analysis is based on a state table and a character 
  *  translation table provided by the user. 
  *  @memo Table driven state machine lexographic analysis. 
  *  @author J. Zweizig
  *  @version 1.1; Last modified March 3, 2008
  *  @ingroup IO_lxr
  */
class lxr {
public:

    /** State data type
     */
    typedef int  state_type;

    /**  Character data type
     */
    typedef char char_type;

    /** Token data type
     */
    typedef int  token_type;

public:
    /**  Enumerate selection type
      */
    enum select_type {
	kChar,
	kGroup,
        kDefault,
	kEOF
    };

    /**  Enumerate flag bits.
      */
    enum flag_bits {
        kUnget,
	kNoSave,
	kReturn
    };

    /**  Sub-class to contain state transition parameters.
      */
    class transition {
    public:
      /**  Transition entry constructor.
        */
      transition(select_type sel, char_type sel_id, int flags, 
		 state_type next);
      /**  Test the specified bit.
        */
      bool fTest(flag_bits b) const;

      /**  Get the next state, or return token.
        */
      state_type next(void) const;

      /**  Get the selection type code.
        */
      select_type sType(void) const;

      /**  Get the selection ID.
        */
      char_type selID(void) const;
    private:
      select_type _select;
      int         _selID;
      int         _flags;
      state_type  _next;
    };

    /** Define transition list data type.
      */
    typedef std::vector<transition> trans_list;

public:
  /**  Construct a lxr object. Allocate tables for the specified number
    *  of states.
    */
  lxr(int N);

  /**  Check the state table for undefined states and for ambiguously defined
    *  transitions. If an undefined state or ambiguous transition is found,
    *  it is reported and an invlid state table exception is thrown. If the 
    *  \c warn argument is true, check will also report missing EOF and 
    *  Default values.
    */
  void check(bool warn) const;

  /**  Dump out the state transition table.
    */
  void dump(void) const;

  /**  Find the state transition entry
    */
  const transition& find(int state, int c) const;

  /**  Get the next token. The token type is return as the function value, 
    *  and the token string is returned to the second argument value.
    */
  token_type token(std::istream& in, std::string& tkn) const;

  /**  Allocate a new state. No transition entries are added.
    */
  int push_state(void);

  /**  Add a transition entry to the table for the specified state.
    */
  void addTransition(int state, select_type sel, char_type sel_id, int flags, 
		    int next);

  /**  Set the initial state number. By default, state 0 is the initial state.
    */
  void setState0(int state);

  /**  Set a translation table that defines the group ID for each character.
    */
  void setTable(const Translate<char>& table);

private:
  int              mStart;
  std::vector<int> mStateInx;
  trans_list       mTranList;
  Translate<char>  mTrTable;
};

//======================================  Inline methods
inline lxr::state_type
lxr::transition::next(void) const {
    return _next;
}

inline lxr::select_type 
lxr::transition::sType(void) const {
    return _select;
}

inline lxr::char_type 
lxr::transition::selID(void) const {
    return _selID;
}

inline bool
lxr::transition::fTest(flag_bits b) const {
    return (_flags & (1 << b)) != 0;
}

#endif // !defined(LXR_HH)
