/* -*- mode: c++; c-basic-offset: 4; -*- */
#include <time.h>
#include "lxr.hh"
#include <iostream>
#include <stdexcept>

using namespace std;

#ifndef EOF
#define EOF int(-1)
#endif

//======================================  Construct a transition descriptor
lxr::transition::transition(select_type sel, char_type sel_id, int flgs, 
			     int next)
  : _select(sel), _selID(sel_id), _flags(flgs), _next(next)
{
}

//======================================  Construct a lxr 
lxr::lxr(int N) 
  : mStart(0), mStateInx(N)
{
    for (int i=0; i<N; i++) mStateInx[i] = 0;
}

//======================================  Dump tables.
void
lxr::check(bool warn) const {
     bool syntax = false;
     int nState = mStateInx.size();
     if (!nState) {
         cout << "Table contains no states" << endl;
         syntax = true;
     }
     int j=0;
     for (int i=0; i<nState; ++i) {
	 int end_i = mStateInx[i];
         if (end_i <= j) {
	     syntax = true;
	     cout << "State " << i << " contains no transitions" << endl;
	     continue;
	 }
	 int tEOF(0), tDef(0), tChar[256];
	 for (int k=0; k<256; k++) tChar[k] = 0;
	 for (; j<end_i; j++) {
	     int sid = int(mTranList[j].selID()) & 255;
	     switch (mTranList[j].sType()) {
	     case kGroup:
	         break;
	     case kChar:
	         tChar[sid]++;
	         break;
	     case kEOF:
	         tEOF++;
	         break;
	     case kDefault:
	         tDef++;
	         break;
	     }
	     int nxt = mTranList[j].next();
	     if (!mTranList[j].fTest(kReturn) && (nxt >= nState || nxt < 0)) {
	         syntax = true;
		 cout << "State " << i << " Invalid next state (" << nxt
		      << ")" << endl;
	     }
	 }
	 if (warn) {
	     if (!tEOF) cout << "State " << i << " EOF transition not defined" 
			     << endl;
	     if (!tDef) cout << "State " << i << " default transition not defined" 
			     << endl;
	 }
	 if (tEOF > 1) {
	     cout << "State " << i << " EOF transition multiply defined" 
		  << endl;
	     syntax = true;
	 }
	 if (tDef > 1) {
	     cout << "State " << i << " default transition multiply defined" 
		  << endl;
	     syntax = true;
	 }
	 bool dupchar = false;
	 for (int k=0; k<256; ++k) {
	      if (tChar[k] > 1) {
		  string p("x");
		  p[0] = k;
		  if (!dupchar) cout << "lxr: State " << i << "Multiple "
				     << "transitions for character(s) \"";
		  dupchar = true;
		  cout << p;
	      }
	 }
	 if (dupchar) {
	     cout << "\"" << endl;
	     syntax =true;
	 }
     }
     if (syntax) throw runtime_error("lxr: Table check failed");
}

//======================================  Dump tables.
void
lxr::dump(void) const {
     int nState = mStateInx.size();
     int j=0;
     for (int i=0; i<nState; ++i) {
         cout << "++++++++++++++++++++++++++  State: " << i << endl;
	 int end_i = mStateInx[i];
	 for (; j<end_i; j++) {
	     string flags;
	     switch (mTranList[j].sType()) {
	     case kGroup:
	       cout << "Group:   " << int(mTranList[j].selID());
	       break;
	     case kChar:
	       flags = "X";
	       flags[0] = char(mTranList[j].selID());
	       cout << "Char:    " << flags;
	       flags.erase(flags.begin(), flags.end());
	       break;
	     case kEOF:
	       cout << "EOF:     ";
	       break;
	     case kDefault:
	       cout << "Default: ";
	       break;
	     }
	     if (mTranList[j].fTest(kNoSave)) flags = "nosave";
	     if (mTranList[j].fTest(kUnget)) {
	         if (!flags.empty()) flags += ",";
		 flags += "unget";
	     }
	     if (mTranList[j].fTest(kReturn)) {
	         if (!flags.empty()) flags += ",";
		 flags += "return";
	     }
	     if (flags.empty()) flags = "0";
	     cout << " " << flags;
	     if (mTranList[j].fTest(kReturn)) cout << " Token: ";
	     else                             cout << " State: ";
	     cout << mTranList[j].next() << endl;
	 }
     }
}

//======================================  Find the transition descriptor
const lxr::transition& 
lxr::find(int state, int c) const {
    int tFirst = 0;
    if (state) tFirst = mStateInx[state-1]; 
    int tLast  = mStateInx[state];
    if (c == EOF) {
        for (int i=tFirst; i<tLast; ++i) {
	    if (mTranList[i].sType() == kEOF) return mTranList[i];
	}
	throw runtime_error("lxr: missing EOF transition");
    }

    int j=-1;
    int group = mTrTable[c];
    for (int i=tFirst; i<tLast; ++i) {
        if (mTranList[i].sType() == kChar) {
	    if (mTranList[i].selID() == c) return mTranList[i];
	} else if (mTranList[i].sType() == kGroup) {
	    if (mTranList[i].selID() == group) j = i;
	} else if (mTranList[i].sType() == kDefault) {
	    if (j<0) j = i;
	}
    }
    if (j<0) throw runtime_error("lxr: Missing default transition");
    return mTranList[j];
}

//======================================  Get a token
lxr::token_type 
lxr::token(std::istream& in, std::string& tkn) const {
    state_type state=mStart;
    tkn.clear();
    while (1) {
        int c = in.get();
        const transition& tr=find(state, c);
	if (!tr.fTest(kNoSave)) tkn += char(c);
	if (tr.fTest(kUnget))   in.unget();
	if (tr.fTest(kReturn))  return tr.next();
	state = tr.next();
    }
}

//======================================  Add a state to the end of the list
int 
lxr::push_state(void) {
    int rc = mStateInx.size();
    mStateInx.push_back(mTranList.size());
    return rc;
}

//======================================  Add a transition to a specified state
void
lxr::addTransition(int state, select_type sel, char_type sel_id, int flags, 
		   int next) {
    int N=mStateInx.size();
    if (state >= N) throw runtime_error("lxr: Undefined state");
    transition tr(sel, sel_id, flags, next);
    mTranList.insert(mTranList.begin()+mStateInx[state], tr);
    for (int i=state; i<N; ++i) mStateInx[i]++;
}

//======================================  Add a transition to a specified state
void 
lxr::setState0(int state) {
    mStart = state;
}

//======================================  Add a transition to a specified state
void 
lxr::setTable(const Translate<char>& table) {
    mTrTable = table;
}
