/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "thread_base.hh"
#include "gmutex.hh"
#include "lsmp_con.hh"
#include "lsmp_prod.hh"
#include <iostream>
#include <stdexcept>
#include <vector>
#include <memory>
#include <cstring>
#include <ctime>

//======================================  Output partition writer
class out_part {

public:
   out_part(void);
   bool connect(void);
   int copy(size_t id, const void* data, size_t length);
   const std::string& name(void) const {return _name;}
   void set_name(const std::string& name);
private:
   std::string   _name;
   size_t        _nbuf;
   size_t        _lbuf;
   std::unique_ptr<LSMP_PROD>    _prod;
   size_t        _lastID;
   thread::mutex _mux;
};

//====================================== Input partition reader thread
class one_part : public thread::thread_base {
public:
   one_part(out_part& outp, const std::string& name);
   const std::string& name(void) const {return _name;}
   void set_penalty(double penalty);
   void set_verbose(int v);
   void* thread_entry(void);
private:
   out_part&   _out;
   std::string _name;
   double      _penalty;
   int         _verbose;
   std::unique_ptr<LSMP_CON>   _cons;
};

class muxer {
public:
   muxer(int argc, const char* argv[]);
   ~muxer(void);
   void run(void);
   void syntax(void) const;
private:
   int       _verbose;
   typedef std::vector<one_part> part_vect;
   part_vect _threads;
   out_part  _out;
};

using namespace std;

//======================================  Main function; build muxer and run it
int
main(int argc, const char* argv[]) {
   muxer m(argc, argv);
   try {
      m.run();
   } catch (std::exception& e) {
      cout << "caught exception while running muxer: " << e.what() << endl;
   }
};

//======================================  Muxer contructor
muxer::muxer(int argc, const char* argv[])
   : _verbose(0)
{
   if ( ::strcmp( argv[1], "--help") == 0 ) {
      syntax();
      return;
   }
   bool syntax_error = false;
   double penalty = 0;
   for (int i=1; i<argc-1; i++) {
      string argi = argv[i];
      if (argi == "-v") {
	 _verbose++;
      }
      else if (argi == "-penalty") {
	 penalty = strtod(argv[++i], 0);
      }
      else if (argi[0] != '-') {
	 _threads.push_back(one_part(_out, argi));
	 _threads.back().set_penalty(penalty);
	 _threads.back().set_verbose(_verbose);
      }
      else {
	 cerr << "unknown option: " << argi << endl;	 
      }     
   }
   _out.set_name(argv[argc-1]);

   if (_threads.empty()) {
      syntax();
   }
   
}

//======================================  Muxer contructor
muxer::~muxer(void) {
}

//======================================  Run the muxer
void
muxer::run(void) {
   size_t N = _threads.size();
   if (!N) {
      cerr << "No threads defined!" << endl;
      return;
   }

   //------------------------------------  Connect to tht output partition
   if (!_out.connect()) {
      throw std::runtime_error(string("Unable to connect to partition: ")
			       + _out.name());
   }
   
   //------------------------------------  Start one thread per input partition
   for (size_t i=0; i<N; i++) {
      if (_verbose) cout << "launch thread for: " << _threads[i].name() << endl;
      _threads[i].set_detached(false); //make thread joinable
      _threads[i].start_thread();
   }

   //------------------------------------  Wait for everything to stop
   void* rc = 0;
   for (size_t i=0; i<N; i++) {
      if (_threads[i].join(&rc) < 0) {
	 perror("unable to join thread");
      } else {
	 cout << "thread: " << i << " terminated with rc=" << (long)rc << endl;
      }
   }
}

//======================================  Run the muxer
void
muxer::syntax(void) const {
   cout << "lsmp_mux command syntax:" << endl;
   cout << "    lsmp_mux [<options>] <ipart-0> [<ipart-1> [... <ipart-n>]] "
	<< "<opart>" << endl;
} 

//=======================================  Input partition reader constructor
one_part::one_part(out_part& out, const std::string& name)
   : _out(out), _name(name), _penalty(0), _verbose(0)
{
}

//=======================================  Input partition frame read loop
void*
one_part::thread_entry(void) {
   if (_verbose) cout << "starting thread for: " << _name
		      << " run = " << run() << endl;
   void* rc = 0;
   _cons.reset(new LSMP_CON(_name.c_str()));
   if (!_cons || !_cons->attached()) {
      cerr << "unable to attach partition: " << _name << endl;
      stop_thread();
      rc = (void*)2;
   }

   //-----------------------------------  Loop over incoming buffers
   while (run()) {
      if (_verbose > 2) cout << "read buffer from: " << _name << endl; 
      const char* b = _cons->get_buffer();
      if (!b) {
	 cerr << "Empty buffer in: " << _name << endl;
	 rc = (void*)1;
	 break;
      }
      LSMP::eventid_type ev = _cons->getEvtID();
      if (_verbose > 1) cout << "read buffer ID: " << ev << endl; 
      int l = _cons->getLength();
      if (_penalty != 0) {
	 struct timespec tm;
	 tm.tv_sec  = time_t(_penalty);
	 tm.tv_nsec = long((_penalty - double(tm.tv_sec)) * 1e9 + 0.5);
	 if (nanosleep(&tm, 0) < 0) {
	    perror("Error in penalty box");
	 }
      }
      int len = _out.copy(ev, b, l);
      _cons->free_buffer();
      if (len > 0 && _verbose) {
	 cout << "wrote: " << ev << " from: " << _name << endl;
      } else if (len < 0) {
	 cerr << "Error copying frame from: " << _name << endl;
	 rc = (void*)3;
      }
   }
   return rc;
}

//=======================================  Set a time penalty
void
one_part::set_penalty(double p) {
   _penalty = p;
}

//=======================================  Set a verbosity index
void
one_part::set_verbose(int v) {
   _verbose = v;
}

//=======================================  Output partition constructor
out_part::out_part(void)
   : _nbuf(0), _lbuf(0)
{
}

//=======================================  Copy frame to output partition
bool
out_part::connect(void) {
   _prod.reset(new LSMP_PROD(_name.c_str()));
   if (!_prod || !_prod->attached()) return false;
   _lbuf = _prod->getBufferLength();
   _nbuf = _prod->getBufferCount();
   return true;
}

//=======================================  Copy frame to output partition
int
out_part::copy(size_t id, const void* buffer, size_t length) {
   thread::semlock lock(_mux);
   if (id <= _lastID) return 0;
   if (_lbuf < length) return -1;
   char* b = _prod->get_buffer();
   memcpy(b, buffer, length);
   _prod->SetID(id);
   _prod->release(length);
   _lastID = id;
   return length;
}


//=======================================  Copy frame to output partition
void
out_part::set_name(const std::string& name) {
   _name = name;
}
