/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_TRANSINPUT_HH
#define LMSG_TRANSINPUT_HH

#include <stdexcept>
#include <string>
#include "lmsg/MsgTypes.hh"
#include "lmsg/Grinder.hh"

class Time;

namespace lmsg {

    class Buffer;

    /**  Message input stream with translation capability. The TransInput 
      *  class is effectively an input stream that is intelligent enough 
      *  to translate between the big-endian external format and an internal
      *  format appropriate to the execution time machine.
      *
      *  @author John Zweizig
      *  @version 1.1; Modified August 20, 2001
      *  @ingroup IO_lmsg
      */
    class TransInput {
    public:
	/**  Construct an input stream and attach a full message buffer to it.
	 *  @memo data constructor.
	 *  @param b Message buffer to be read by the input stream.
	 */
	TransInput(const Buffer& b);

	/**  Default input stream constructor.
	 */
	TransInput(void);

	/**  Align data to appropriate boundary.
	 */
	void align(size_type bdry);

	/**  Test whether the specified number of bytes remain in the 
	  *  input buffer.
	  */
	bool available(size_type N) const;

	/**  Test whether there is any more data in the input buffer.
	 *  @memo Test whether buffer is empty;
	 *  @return true if the buffer is empty.
	 */
	bool empty(void) const;

	/**  Template to read and translate a character array.
	 *  @memo read a character array.
	 *  @return number of characters read.
	 *  @param x Pointer to a character array to receive the data.
	 *  @param N Number of characters to be read.
	 */
	template<class T> 
	size_type read(T x[], size_type N=1);
 
	/** Stream input operator template;
	 */
	template <class T> 
	TransInput& operator>>(T& x);

    private:
	const char* mData;
	size_type      mIndex;
	size_type      mLength;
    };  //  class lmsg::TransInput

    //======================================  Specific templates

    /**  Read a character string of length N.
      *  @brief Read a character string
      *  @param x Character array to receive string.
      *  @param N Number of characters to read.
      *  @return Number of characters read.
      */
    template<> 
    size_type TransInput::read(char x[], size_type N);

    /**  Read N message addresses.
      *  @brief Read MsgAddr's.
      *  @param x MsgAddr array to receive data.
      *  @param N Number of items to read.
      *  @return Number of data items read.
      */
    template<> 
    size_type TransInput::read(MsgAddr x[], size_type N);

    /**  Read N GPS times.
      *  @brief Read GPS Time's.
      *  @param x Time array to receive data.
      *  @param N Number of items to read.
      *  @return Number of data items read.
      */
    template<> 
    size_type TransInput::read(Time x[], size_type N);

    /**  Read N byte-count strings.
      *  @brief Read byte-count strings.
      *  @param x String array to receive data.
      *  @param N Number of items to read.
      *  @return Number of data items read.
      */
    template<> 
    size_type TransInput::read(std::string x[], size_type N);

    //======================================  Inline methods
    template<class T>
    inline TransInput&
    TransInput::operator>> (T& x) {
	if (!read(&x, 1)) throw std::runtime_error("End-Of-Data");
	return *this;
    }

    inline bool 
    TransInput::available(size_type N) const {
	return mIndex + N <= mLength;
    }
  
    inline bool
    TransInput::empty(void) const {
	return mIndex >= mLength;
    }

    template<class T> 
    inline size_type 
    TransInput::read(T x[], size_type N) {
	align(sizeof(T));
	if (empty()) return 0;
	if (!available(N*sizeof(T))) N = (mLength - mIndex)/sizeof(T);
	export_format_grinder.SwapIn(mData+mIndex, x, N);
	mIndex += N*sizeof(T);
	return N;
    }

}    // namespace lmsg

#endif  //  LMSG_TRANSINPUT_HH
