/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_BUFFERPOOL_HH
#define LMSG_BUFFERPOOL_HH
#include "lmsg/MsgTypes.hh"
#include "gmutex.hh"
#include <list>

#define DEFAULTBUFFERPOOLSIZE  65536
#define DEFAULTBUFFERPOOLCOUNT    16

namespace lmsg {
    class Buffer;

    /**  The BufferPool class defines a pool of pre-allocated fixed-length 
      *  buffers to be used for asynchronous I/O in the Ligo DMT Message (lmsg)
      *  facility.
      *  @brief Buffer Pool class.
      *  @author John Zweizig
      *  @version 1.2; Last modified March 4, 2008
      *  @ingroup IO_lmsg
      */
    class BufferPool {
    public:
	/**  Destroy a buffer pool and free all its buffers. If the number 
	  *  of buffers freed is not equal to the number allocated, an 
	  *  error message is printed to \c cerr.
	  *  @brief Destructor.
	  */
	~BufferPool(void);

	/**  Construct a buffer pool containing the specified number of
	  *  buffers of the specified length.
	  *  @brief Construct a buffer pool.
	  *  @param number Number of buffer to allocate.
	  *  @param length Length of buffer to allocate.
	  */
	BufferPool(size_type number, size_type length);

	/**  Get a buffer from the pool. If the pool is empty, a null 
	  *  pointer is returned.
	  *  @brief Get a buffer.
	  */
	Buffer* getBuffer(void);

	/**  Return the specified buffer to the pool.
	  *  @brief Return a buffer.
	  *  @param buff Pointer to the buffer to be returned to the pool.
	  */
	void    returnBuffer(Buffer* buff);

	/**  Get the length of a buffer in bytes.
	  *  @brief Get the bufffer length.
	  *  @return Buffer length in bytes.
	  */
	size_type  getLength(void) const;

	/**  Get the length of the data portion of the longest message that
	  *  can fit in the buffer.
	  *  @brief Get maximum data length.
	  *  @return Longest message data portion in bytes.
	  */
	size_type  getDataLength(void) const;

    private:
	typedef std::list<Buffer*> PoolList;
	typedef PoolList::iterator Pool_Iter;

    private:
	size_type   mLength;
	size_type   mCount;
	PoolList mPool;
	thread::mutex mMux;
    }; // Buffer class

    inline size_type
    BufferPool::getLength(void) const {
	return mLength;
    }

} // namespace lmsg

/**  Default buffer pool.
  */
extern lmsg::BufferPool defaultLMsgBufferPool;

#endif // LMSG_BUFFER_HH
