/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "jsonStack.hh"
#include <json/reader.h>
#include <fstream>

using namespace std;

//======================================  Json stack model constructor
jsonStack::jsonStack(void) {
   _stack.reserve(16);
}

//======================================  Json stack model initializer
jsonStack::jsonStack(const std::string& file) {
   _stack.reserve(16);
   parse(file);
}

//======================================  Fetch bool data element
size_t 
jsonStack::fetch_data(const std::string name, bool& val) const {
   if (!value().isMember(name)) return 0;
   const Json::Value& el = value()[name];
   if (!el.isBool()) return 0;
   val = el.asBool();
   return 1;
}

//======================================  Fetch numeric data element
size_t 
jsonStack::fetch_data(const std::string name, double& val) const {
   if (!value().isMember(name)) return 0;
   const Json::Value& el = value()[name];
   if (!el.isNumeric()) return 0;
   val = el.asDouble();
   return 1;
}

//======================================  Fetch string data element
size_t 
jsonStack::fetch_data(const std::string name, std::string& val) const {
   if (!value().isMember(name)) return 0;
   val = value()[name].asString();
   return 1;
}

//======================================  Fetch bool data vector
size_t 
jsonStack::fetch_data(const std::string name, std::vector<bool>& val) {
   if (!push_element(name)) return 0;
   const Json::Value& vec(value());
   Json::ArrayIndex   N = vec.size();
   val.clear();
   val.reserve(N);
   for (Json::ArrayIndex i=0; i < N; i++) {
      if (!vec[i].isBool()) N = 0;
      else                  val.push_back(vec[i].asBool());
   }
   pop();
   return N;
}

//======================================  Fetch bool data vector
size_t 
jsonStack::fetch_data(const std::string name, std::vector<double>& val) {
   if (!push_element(name)) return 0;
   const Json::Value& vec(value());
   Json::ArrayIndex   N = vec.size();
   val.clear();
   val.reserve(N);
   for (Json::ArrayIndex i=0; i < N; i++) {
      if (!vec[i].isNumeric()) N = 0;
      else                     val.push_back(vec[i].asDouble());
   }
   pop();
   return N;
}

//======================================  Fetch bool data vector
size_t 
jsonStack::fetch_data(const std::string name, std::vector<std::string>& val) {
   if (!push_element(name)) return 0;
   const Json::Value& vec(value());
   Json::ArrayIndex   N = vec.size();
   val.clear();
   val.reserve(N);
   for (Json::ArrayIndex i=0; i < N; i++) {
      val.push_back(vec[i].asString());
   }
   pop();
   return N;
}

//======================================  Parse out the file
bool
jsonStack::parse(const std::string& file) {
   //------------------------------------  Read the configuration file.
   ifstream in(file.c_str());
   if (!in.is_open()) {
      throw runtime_error(string("unable to open file: ") + file);
   }
   string conf_str;
   getline(in, conf_str, (char)EOF);
   if (conf_str.empty()) {
      throw runtime_error(string("Configuration file: ") + file + " is empty");
   }
   Json::Value  doc;
   Json::Reader rdr;
   rdr.parse(conf_str.data(), conf_str.data() + conf_str.size(), doc);
   push_element(doc);
   return true;
}

//======================================  Push a sub-document onto the stack
bool 
jsonStack::push_element(void) {
   stack_element el;
   _stack.push_back(el);
   return true;
}

//======================================  Push a sub-document onto the stack
bool 
jsonStack::push_element(const std::string& name) {
   if (!value().isMember(name)) return false;
   return push_element(value()[name]);
}

//======================================  Push a sub-document onto the stack
bool 
jsonStack::push_element(const Json::Value& doc) {
   _stack.push_back(stack_element(doc));
   return true;
}
   
//======================================  Push a sub-document onto the stack
bool 
jsonStack::pop(void) {
   if (empty()) throw logic_error("jsonStack: Attempt to pop from empty stack");
   _stack.pop_back();
   return !empty();
}

//======================================  Iterate over array elements
bool
jsonStack::iterate(void) {
   if (empty()) {
      throw logic_error("jsonSstack: Attempted operation on empty stack");
   }
   stack_element& el = _stack.back();
   if (el._iter >= el._size) return false;
   push_element(el._value[el._iter++]);
   return true;
}

//======================================  stack_element default constructor
jsonStack::stack_element::stack_element(void) {
   _size  = 0;
   _iter  = 0;
}

//======================================  stack_element data constructor
jsonStack::stack_element::stack_element(const Json::Value& doc) {
   _value = doc;
   _size  = doc.size();
   _iter  = 0;
}
